package cn.com.duiba.scrm.common.enums.wechat;

import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 微信错误码
 * https://work.weixin.qq.com/api/doc/90000/90139/90313
 *
 * @author xiaotian
 * @date 2020-01-15
 */
@Slf4j
public enum WxErrorCodeEnum {
    E_10000("-10000", "错误码暂时找不到对应"),
    //请求发生未知错误，可能已经成功，不可盲目重试
    E_8("-8", "请求发送异常"),
    //确认请求未发出。可选择重试
    E_7("-7", "httpclient 发送请求时错误"),
    //微信有些功能是下限不可用的，活着API中功能未开放
    E_6("-6", "功能下线"),
    E_5("-5", "参数错误"),
    //socket超时，但请求已经发出
    E_4("-4", "请求没有回复"),
    //特殊情况
    E_3("-3", "配置不发送消息"),
    E_2("-2", "参数为空"),
    E_1("-1", "系统繁忙，请开发者重试"),
    E0("0", "请求成功"),
    E6000("6000", "数据版本冲突"),
    E40001("40001", "不合法的 secret 参数"),
    E40003("40003", "无效的 UserID"),
    E40004("40004", "不合法的媒体文件类型"),
    E40005("40005", "不合法的 type 参数"),
    E40006("40006", "不合法的文件大小"),
    E40007("40007", "不合法的 media_id 参数"),
    E40008("40008", "不合法的 msgtype 参数"),
    E40009("40009", "上传图片大小不是有效值"),
    E40011("40011", "上传视频大小不是有效值"),
    E40013("40013", "不合法的 CorpID"),
    E40014("40014", "不合法的 access_token"),
    E40016("40016", "不合法的按钮个数"),
    E40017("40017", "不合法的按钮类型"),
    E40018("40018", "不合法的按钮名字长度"),
    E40019("40019", "不合法的按钮 KEY 长度"),
    E40020("40020", "不合法的按钮 URL 长度"),
    E40022("40022", "不合法的子菜单级数"),
    E40023("40023", "不合法的子菜单按钮个数"),
    E40024("40024", "不合法的子菜单按钮类型"),
    E40025("40025", "不合法的子菜单按钮名字长度"),
    E40026("40026", "不合法的子菜单按钮 KEY 长度"),
    E40027("40027", "不合法的子菜单按钮 URL 长度"),
    E40029("40029", "不合法的 oauth_code"),
    E40031("40031", "不合法的 UserID 列表"),
    E40032("40032", "不合法的 UserID 列表长度"),
    E40033("40033", "不合法的请求字符"),
    E40035("40035", "不合法的参数"),
    E40036("40036", "不合法的模板 id 长度"),
    E40037("40037", "无效的模板 id"),
    E40039("40039", "不合法的 url 长度"),
    E40050("40050", "chatid 不存在"),
    E40054("40054", "不合法的子菜单 url 域名"),
    E40055("40055", "不合法的菜单 url 域名"),
    E40056("40056", "不合法的 agentid"),
    E40057("40057", "不合法的 callbackurl 或者 callbackurl 验证失败"),
    E40058("40058", "不合法的参数"),
    E40059("40059", "不合法的上报地理位置标志位"),
    E40063("40063", "参数为空"),
    E40066("40066", "不合法的部门列表"),
    E40068("40068", "不合法的标签 ID"),
    E40070("40070", "指定的标签范围结点全部无效"),
    E40071("40071", "不合法的标签名字"),
    E40072("40072", "不合法的标签名字长度"),
    E40073("40073", "不合法的 openid"),
    E40074("40074", "news 消息不支持保密消息类型"),
    E40077("40077", "不合法的 pre_auth_code 参数"),
    E40078("40078", "不合法的 auth_code 参数"),
    E40080("40080", "不合法的 suite_secret"),
    E40082("40082", "不合法的 suite_token"),
    E40083("40083", "不合法的 suite_id"),
    E40084("40084", "不合法的 permanent_code 参数"),
    E40085("40085", "不合法的的 suite_ticket 参数"),
    E40086("40086", "不合法的第三方应用 appid"),
    E40088("40088", "jobid 不存在"),
    E40089("40089", "批量任务的结果已清理"),
    E40091("40091", "secret 不合法"),
    E40092("40092", "导入文件存在不合法的内容"),
    E40093("40093", "jsapi 签名错误"),
    E40094("40094", "不合法的 URL"),
    E40096("40096", "不合法的外部联系人 userid"),
    E40097("40097", "该成员尚未离职"),
    E40098("40098", "成员尚未实名认证"),
    E40099("40099", "外部联系人的数量已达上限"),
    E40100("40100", "此用户的外部联系人已经在转移流程中"),
    E40102("40102", "域名或 IP 不可与应用市场上架应用重复"),
    E40123("40123", "上传临时图片素材，图片格式非法"),
    E40124("40124", "推广活动里的 sn 禁止绑定"),
    E40125("40125", "无效的 openuserid 参数"),
    E40126("40126", "企业标签个数达到上限，最多为 3000 个"),
    E40127("40127", "不支持的 uri schema"),
    E40128("40128", "客户转接过于频繁（90 天内只允许转接一次，同一个客户最多只能转接两次）"),
    E40129("40129", "当前客户正在转接中"),
    E40130("40130", "原跟进人与接手人一样，不可继承"),
    E40131("40131", "handover_userid 并不是外部联系人的跟进人"),
    E41001("41001", "缺少 access_token 参数"),
    E41002("41002", "缺少 corpid 参数"),
    E41004("41004", "缺少 secret 参数"),
    E41006("41006", "缺少 media_id 参数"),
    E41008("41008", "缺少 auth code 参数"),
    E41009("41009", "缺少 userid 参数"),
    E41010("41010", "缺少 url 参数"),
    E41011("41011", "缺少 agentid 参数"),
    E41016("41016", "缺少 title 参数"),
    E41019("41019", "缺少 department 参数"),
    E41017("41017", "缺少 tagid 参数"),
    E41021("41021", "缺少 suite_id 参数"),
    E41022("41022", "缺少 suite_access_token 参数"),
    E41023("41023", "缺少 suite_ticket 参数"),
    E41024("41024", "缺少 secret 参数"),
    E41025("41025", "缺少 permanent_code 参数"),
    E41033("41033", "缺少 description 参数"),
    E41035("41035", "缺少外部联系人 userid 参数"),
    E41036("41036", "不合法的企业对外简称"),
    E41037("41037", "缺少「联系我」type 参数"),
    E41038("41038", "缺少「联系我」scene 参数"),
    E41039("41039", "无效的「联系我」type 参数"),
    E41040("41040", "无效的「联系我」scene 参数"),
    E41041("41041", "使用员工最多100人"),
    E41042("41042", "无效的「联系我」style 参数"),
    E41043("41043", "缺少「联系我」config_id 参数"),
    E41044("41044", "该活码已失效"),
    E41045("41045", "使用员工不得超过100个"),
    E41046("41046", "缺少企业群发消息 id"),
    E41047("41047", "无效的企业群发消息 id"),
    E41048("41048", "无可发送的客户"),
    E41049("41049", "缺少欢迎语 code 参数"),
    E41050("41050", "无效的欢迎语 code"),
    E41051("41051", "客户和服务人员已经开始聊天了"),
    E41052("41052", "无效的发送时间"),
    E41053("41053", "客户未同意聊天存档"),
    E41054("41054", "部分员工未激活"),
    E41055("41055", "群欢迎语模板数量达到上限"),
    E41056("41056", "外部联系人 id 类型不正确"),
    E41057("41057", "企业或服务商未绑定微信开发者账号"),
    E41059("41059", "缺少 moment_id 参数"),
    E41060("41060", "不合法的 moment_id 参数"),
    E41061("41061", "不合法朋友圈发送成员 userid，当前朋友圈并非此用户发表"),
    E41062("41062", "企业创建的朋友圈尚未被成员 userid 发表"),
    E41063("41063", "群发消息正在被派发中，请稍后再试"),
    E41064("41064", "附件大小超过限制"),
    E41065("41065", "无效的附件类型"),
    E41066("41066", "用户视频号名称错误"),
    E41067("41067", "朋友圈 moment_id 类型错误"),
    E41102("41102", "缺少菜单名"),
    E42001("42001", "access_token 已过期"),
    E42007("42007", "pre_auth_code 已过期"),
    E42009("42009", "suite_access_token 已过期"),
    E42012("42012", "jsapi_ticket 不可用，一般是没有正确调用接口来创建 jsapi_ticket"),
    E42013("42013", "小程序未登陆或登录态已经过期"),
    E42014("42014", "任务卡片消息的 task_id 不合法"),
    E42015("42015", "更新的消息的应用与发送消息的应用不匹配"),
    E42016("42016", "更新的 task_id 不存在"),
    E42017("42017", "按钮 key 值不存在"),
    E42018("42018", "按钮 key 值不合法"),
    E42019("42019", "缺少按钮 key 值不合法"),
    E42020("42020", "缺少按钮名称"),
    E42021("42021", "device_access_token 过期"),
    E42022("42022", "code 已经被使用过。只能使用一次"),
    E43004("43004", "指定的 userid 未绑定微信或未关注微工作台（原企业号）"),
    E43009("43009", "企业未验证主体"),
    E43012("43012", "应用需配置回调 url"),
    E44001("44001", "多媒体文件为空"),
    E44004("44004", "文本消息 content 参数为空"),
    E45001("45001", "多媒体文件大小超过限制"),
    E45002("45002", "消息内容大小超过限制"),
    E45004("45004", "应用 description 参数长度不符合系统限制"),
    E45007("45007", "语音播放时间超过限制"),
    E45008("45008", "图文消息的文章数量不符合系统限制"),
    E45009("45009", "接口调用超过限制"),
    E45022("45022", "应用 name 参数长度不符合系统限制"),
    E45024("45024", "帐号数量超过上限"),
    E45026("45026", "触发删除用户数的保护"),
    E45029("45029", "回包大小超过上限"),
    E45032("45032", "图文消息 author 参数长度超过限制"),
    E45033("45033", "接口并发调用超过限制"),
    E45034("45034", "url 必须有协议头"),
    E46003("46003", "菜单未设置"),
    E46004("46004", "指定的用户不存在"),
    E48002("48002", "API 接口无权限调用"),
    E48003("48003", "不合法的 suite_id"),
    E48004("48004", "授权关系无效"),
    E48005("48005", "API 接口已废弃"),
    E48006("48006", "接口权限被收回"),
    E49004("49004", "签名不匹配"),
    E49008("49008", "群已经解散"),
    E50001("50001", "redirect_url 未登记可信域名"),
    E50002("50002", "成员不在权限范围"),
    E50003("50003", "应用已禁用"),
    E50100("50100", "分页查询的游标无效"),
    E60001("60001", "部门长度不符合限制"),
    E60003("60003", "部门 ID 不存在"),
    E60004("60004", "父部门不存在"),
    E60005("60005", "部门下存在成员"),
    E60006("60006", "部门下存在子部门"),
    E60007("60007", "不允许删除根部门"),
    E60008("60008", "部门已存在"),
    E60009("60009", "部门名称含有非法字符"),
    E60010("60010", "部门存在循环关系"),
    E60011("60011", "指定的成员 / 部门 / 标签参数无权限"),
    E60012("60012", "不允许删除默认应用"),
    E60020("60020", "访问 ip 不在白名单之中"),
    E60021("60021", "userid 不在应用可见范围内"),
    E60028("60028", "不允许修改第三方应用的主页 URL"),
    E60102("60102", "UserID 已存在"),
    E60103("60103", "手机号码不合法"),
    E60104("60104", "手机号码已存在"),
    E60105("60105", "邮箱不合法"),
    E60106("60106", "邮箱已存在"),
    E60107("60107", "微信号不合法"),
    E60110("60110", "用户所属部门数量超过限制"),
    E60111("60111", "UserID 不存在"),
    E60112("60112", "成员 name 参数不合法"),
    E60123("60123", "无效的部门 id"),
    E60124("60124", "无效的父部门 id"),
    E60125("60125", "非法部门名字"),
    E60127("60127", "缺少 department 参数"),
    E60129("60129", "成员手机和邮箱都为空"),
    E60132("60132", "is_leader_in_dept 和 department 的元素个数不一致"),
    E60136("60136", "记录不存在"),
    E60137("60137", "家长手机号重复"),
    E60203("60203", "不合法的模版 ID"),
    E60204("60204", "模版状态不可用"),
    E60205("60205", "模版关键词不匹配"),
    E60206("60206", "该种类型的消息只支持第三方独立应用使用"),
    E60207("60207", "第三方独立应用只允许发送模板消息"),
    E60208("60208", "第三方独立应用不支持指定 @all，不支持参数 toparty 和 totag"),
    E60209("60209", "缺少操作者 vid"),
    E60210("60210", "选择成员列表为空"),
    E60211("60211", "SelectedTicket 为空"),
    E60214("60214", "仅支持第三方应用调用"),
    E60215("60215", "传入 SelectedTicket 数量超过最大限制（10 个）"),
    E60217("60217", "当前操作者无权限，操作者需要授权或者在可见范围内"),
    E60218("60218", "仅支持成员授权模式的应用可调用"),
    E60219("60219", "消费 SelectedTicket 和创建 SelectedTicket 的应用 appid 不匹配"),
    E60220("60220", "缺少 corpappid"),
    E60221("60221", "open_userid 对应的服务商不是当前服务商"),
    E60222("60222", "非法 SelectedTicket"),
    E60223("60223", "非法 BundleId"),
    E60224("60224", "非法 PackageName"),
    E60225("60225", "当前操作者并非 SelectedTicket 相关人，不能创建群聊"),
    E60226("60226", "选人数量超过最大限制（2000）"),
    E60227("60227", "缺少 ServiceCorpid"),
    E60228("60228", "缺少 bind_corpid 字段"),
    E60229("60229", "成员或者部门 id 不正确"),
    E60230("60230", "缺少 shareticket"),
    E60231("60231", "shareticket 非法"),
    E60233("60233", "shareticket 非法"),
    E60234("60234", "shareticket 非法"),
    E60235("60235", "缺少 payment_id 字段"),
    E60236("60236", "缺少 trade_no 字段"),
    E60237("60237", "传入的 payment_id 对应的收款项目不是由当前应用发起的"),
    E60239("60239", "收款人未实名"),
    E60240("60240", "收款企业尚未验证或者认证"),
    E60241("60241", "付款学生或者部门 id 不正确"),
    E60242("60242", "shareticket 不能跨域名使用"),
    E60243("60243", "trade_no 不合法"),
    E60244("60244", "shareticket 不能跨 APP 使用"),
    E65000("65000", "学校已经迁移"),
    E65001("65001", "无效的关注模式"),
    E65002("65002", "导入家长信息数量过多"),
    E65003("65003", "学校尚未迁移"),
    E65004("65004", "组织架构不存在"),
    E65005("65005", "无效的同步模式"),
    E65006("65006", "无效的管理员类型"),
    E65007("65007", "无效的家校部门类型"),
    E65008("65008", "无效的入学年份"),
    E65009("65009", "无效的标准年级类型"),
    E65010("65010", "此 userid 并不是学生"),
    E65011("65011", "家长 userid 数量超过限制"),
    E65012("65012", "学生 userid 数量超过限制"),
    E65013("65013", "学生已有家长"),
    E65014("65014", "非学校企业"),
    E65015("65015", "父部门类型不匹配"),
    E65018("65018", "家长人数达到上限"),
    E65022("65022", "家校通迅录无权限"),
    E660001("660001", "无效的商户号"),
    E660002("660002", "无效的企业收款人 id"),
    E660003("660003", "userid 不在应用的可见范围"),
    E660004("660004", "partyid 不在应用的可见范围"),
    E660005("660005", "tagid 不在应用的可见范围"),
    E660006("660006", "找不到该商户号"),
    E660007("660007", "申请已经存在"),
    E660008("660008", "商户号已经绑定"),
    E660009("660009", "商户号主体和商户主体不一致"),
    E660010("660010", "超过商户号绑定数量限制"),
    E660011("660011", "商户号未绑定"),
    E670001("670001", "应用不在共享范围"),
    E72023("72023", "发票已被其他公众号锁定"),
    E72024("72024", "发票状态错误"),
    E72037("72037", "存在发票不属于该用户"),
    E80001("80001", "可信域名不正确，或者无 ICP 备案"),
    E81001("81001", "部门下的结点数超过限制（3W）"),
    E81002("81002", "部门最多 15 层"),
    E81003("81003", "标签下节点个数超过 30000 个"),
    E81011("81011", "无权限操作标签"),
    E81012("81012", "缺失可见范围"),
    E81013("81013", "UserID、部门 ID、标签 ID 全部非法或无权限"),
    E81014("81014", "标签添加成员，单次添加 user 或 party 过多"),
    E81015("81015", "邮箱域名需要跟企业邮箱域名一致"),
    E81016("81016", "logined_userid 字段缺失"),
    E81017("81017", "请求个数超过限制"),
    E81018("81018", "该服务商可获取名字数量配额不足"),
    E81019("81019", "items 数组成员缺少 id 字段"),
    E81020("81020", "items 数组成员缺少 type 字段"),
    E81021("81021", "items 数组成员的 type 字段不合法"),
    E82001("82001", "指定的成员 / 部门 / 标签全部为空"),
    E82002("82002", "不合法的 PartyID 列表长度"),
    E82003("82003", "不合法的 TagID 列表长度"),
    E82004("82004", "不合法的消息内容"),
    E84014("84014", "成员票据过期"),
    E84015("84015", "成员票据无效"),
    E84019("84019", "缺少 templateid 参数"),
    E84020("84020", "templateid 不存在"),
    E84021("84021", "缺少 register_code 参数"),
    E84022("84022", "无效的 register_code 参数"),
    E84023("84023", "不允许调用设置通讯录同步完成接口"),
    E84024("84024", "无注册信息"),
    E84025("84025", "不符合的 state 参数"),
    E84052("84052", "缺少 caller 参数"),
    E84053("84053", "缺少 callee 参数"),
    E84054("84054", "缺少 auth_corpid 参数"),
    E84055("84055", "超过拨打公费电话频率"),
    E84056("84056", "被拨打用户安装应用时未授权拨打公费电话权限"),
    E84057("84057", "公费电话余额不足"),
    E84058("84058", "caller 呼叫号码不支持"),
    E84059("84059", "号码非法"),
    E84060("84060", "callee 呼叫号码不支持"),
    E84061("84061", "不存在外部联系人的关系"),
    E84062("84062", "未开启公费电话应用"),
    E84063("84063", "caller 不存在"),
    E84064("84064", "callee 不存在"),
    E84065("84065", "caller 跟 callee 电话号码一致"),
    E84066("84066", "服务商拨打次数超过限制"),
    E84067("84067", "管理员收到的服务商公费电话个数超过限制"),
    E84069("84069", "拨打方被限制拨打公费电话"),
    E84070("84070", "不支持的电话号码"),
    E84071("84071", "不合法的外部联系人授权码"),
    E84072("84072", "应用未配置客服"),
    E84073("84073", "客服 userid 不在应用配置的客服列表中"),
    E84074("84074", "没有外部联系人权限"),
    E84075("84075", "不合法或过期的 authcode"),
    E84076("84076", "缺失 authcode"),
    E84077("84077", "订单价格过高，无法受理"),
    E84078("84078", "购买人数不正确"),
    E84079("84079", "价格策略不存在"),
    E84080("84080", "订单不存在"),
    E84081("84081", "存在未支付订单"),
    E84082("84082", "存在申请退款中的订单"),
    E84083("84083", "非服务人员"),
    E84084("84084", "非跟进用户"),
    E84085("84085", "应用已下架"),
    E84086("84086", "订单人数超过可购买最大人数"),
    E84087("84087", "打开订单支付前禁止关闭订单"),
    E84088("84088", "禁止关闭已支付的订单"),
    E84089("84089", "订单已支付"),
    E84090("84090", "缺失 user_ticket"),
    E84091("84091", "订单价格不可低于下限"),
    E84092("84092", "无法发起代下单操作"),
    E84093("84093", "代理关系已占用，无法代下单"),
    E84094("84094", "该应用未配置代理分润规则，请先联系应用服务商处理"),
    E84095("84095", "免费试用版，无法扩容"),
    E84096("84096", "免费试用版，无法续期"),
    E84097("84097", "当前企业有未处理订单"),
    E84098("84098", "固定总量，无法扩容"),
    E84099("84099", "非购买状态，无法扩容"),
    E84100("84100", "未购买过此应用，无法续期"),
    E84101("84101", "企业已试用付费版本，无法全新购买"),
    E84102("84102", "企业当前应用状态已过期，无法扩容"),
    E84103("84103", "仅可修改未支付订单"),
    E84104("84104", "订单已支付，无法修改"),
    E84105("84105", "订单已被取消，无法修改"),
    E84106("84106", "企业含有该应用的待支付订单，无法代下单"),
    E84107("84107", "企业含有该应用的退款中订单，无法代下单"),
    E84108("84108", "企业含有该应用的待生效订单，无法代下单"),
    E84109("84109", "订单定价不能未 0"),
    E84110("84110", "新安装应用不在试用状态，无法升级为付费版"),
    E84111("84111", "无足够可用优惠券"),
    E84112("84112", "无法关闭未支付订单"),
    E84113("84113", "无付费信息"),
    E84114("84114", "虚拟版本不支持下单"),
    E84115("84115", "虚拟版本不支持扩容"),
    E84116("84116", "虚拟版本不支持续期"),
    E84117("84117", "在虚拟正式版期内不能扩容"),
    E84118("84118", "虚拟正式版期内不能变更版本"),
    E84119("84119", "当前企业未报备，无法进行代下单"),
    E84120("84120", "当前应用版本已删除"),
    E84121("84121", "应用版本已删除，无法扩容"),
    E84122("84122", "应用版本已删除，无法续期"),
    E84123("84123", "非虚拟版本，无法升级"),
    E84124("84124", "非行业方案订单，不能添加部分应用版本的订单"),
    E84125("84125", "购买人数不能少于最少购买人数"),
    E84126("84126", "购买人数不能多于最大购买人数"),
    E84127("84127", "无应用管理权限"),
    E84128("84128", "无该行业方案下全部应用的管理权限"),
    E84129("84129", "付费策略已被删除，无法下单"),
    E84130("84130", "订单生效时间不合法"),
    E84200("84200", "文件转译解析错误"),
    E85002("85002", "包含不合法的词语"),
    E85004("85004", "每企业每个月设置的可信域名不可超过 20 个"),
    E85005("85005", "可信域名未通过所有权校验"),
    E86001("86001", "参数 chatid 不合法"),
    E86003("86003", "参数 chatid 不存在"),
    E86004("86004", "参数 群名不合法"),
    E86005("86005", "参数 群主不合法"),
    E86006("86006", "群成员数过多或过少"),
    E86007("86007", "不合法的群成员"),
    E86008("86008", "非法操作非自己创建的群"),
    E86101("86101", "仅群主才有操作权限"),
    E86201("86201", "参数 需要 chatid"),
    E86202("86202", "参数 需要群名"),
    E86203("86203", "参数 需要群主"),
    E86204("86204", "参数 需要群成员"),
    E86205("86205", "参数 字符串 chatid 过长"),
    E86206("86206", "参数 数字 chatid 过大"),
    E86207("86207", "群主不在群成员列表"),
    E86214("86214", "群发类型不合法"),
    E86215("86215", "会话 ID 已经存在"),
    E86216("86216", "存在非法会话成员 ID"),
    E86217("86217", "会话发送者不在会话成员列表中"),
    E86220("86220", "指定的会话参数不合法"),
    E86224("86224", "不是受限群，不允许使用该接口"),
    E90001("90001", "未认证摇一摇周边"),
    E90002("90002", "缺少摇一摇周边 ticket 参数"),
    E90003("90003", "摇一摇周边 ticket 参数不合法"),
    E90100("90100", "非法的对外属性类型"),
    E90101("90101", "对外属性：文本类型长度不合法"),
    E90102("90102", "对外属性：网页类型标题长度不合法"),
    E90103("90103", "对外属性：网页 url 不合法"),
    E90104("90104", "对外属性：小程序类型标题长度不合法"),
    E90105("90105", "对外属性：小程序类型 pagepath 不合法"),
    E90106("90106", "对外属性：请求参数不合法"),
    E90200("90200", "缺少小程序 appid 参数"),
    E90201("90201", "小程序通知的 content_item 个数超过限制"),
    E90202("90202", "小程序通知中的 key 长度不合法"),
    E90203("90203", "小程序通知中的 value 长度不合法"),
    E90204("90204", "小程序通知中的 page 参数不合法"),
    E90206("90206", "小程序未关联到企业中"),
    E90207("90207", "不合法的小程序 appid"),
    E90208("90208", "小程序 appid 不匹配"),
    E90211("90211", "登录时传入的 suiteid 不合法"),
    E90212("90212", "登录时传入的 suiteid 和使用的小程序绑定的第三方应用不匹配"),
    E90300("90300", "orderid 不合法"),
    E90302("90302", "付费应用已过期"),
    E90303("90303", "付费应用超过最大使用人数"),
    E90304("90304", "订单中心服务异常，请稍后重试"),
    E90305("90305", "参数错误，errmsg 中有提示具体哪个参数有问题"),
    E90306("90306", "商户设置不合法，详情请见 errmsg"),
    E90307("90307", "登录态过期"),
    E90308("90308", "在开启 IP 鉴权的前提下，识别为无效的请求 IP"),
    E90309("90309", "订单已经存在，请勿重复下单"),
    E90310("90310", "找不到订单"),
    E90311("90311", "关单失败，可能原因：该单并没被拉起支付页面；已经关单；已经支付；渠道失败；单处于保护状态；等等"),
    E90312("90312", "退款请求失败，详情请看 errmsg"),
    E90313("90313", "退款调用频率限制，超过规定的阈值"),
    E90314("90314", "订单状态错误，可能未支付，或者当前状态操作受限"),
    E90315("90315", "退款请求失败，主键冲突，请核实退款 refund_id 是否已使用"),
    E90316("90316", "退款原因编号不对"),
    E90317("90317", "尚未注册成为供应商"),
    E90318("90318", "参数 nonce_str 为空或者重复，判定为重放攻击"),
    E90319("90319", "时间戳为空或者与系统时间间隔太大"),
    E90320("90320", "订单 token 无效"),
    E90321("90321", "订单 token 已过有效时间"),
    E90322("90322", "旧套件（包含多个应用的套件）不支持支付系统"),
    E90323("90323", "单价超过限额"),
    E90324("90324", "商品数量超过限额"),
    E90325("90325", "预支单已经存在"),
    E90326("90326", "预支单单号非法"),
    E90327("90327", "该预支单已经结算下单"),
    E90328("90328", "结算下单失败，详情请看 errmsg"),
    E90329("90329", "该订单号已经被预支单占用"),
    E90330("90330", "创建供应商失败"),
    E90331("90331", "更新供应商失败"),
    E90332("90332", "还没签署合同"),
    E90333("90333", "创建合同失败"),
    E90338("90338", "已经过了可退款期限"),
    E90339("90339", "供应商主体名包含非法字符"),
    E90340("90340", "创建客户失败，可能信息真实性校验失败"),
    E90341("90341", "退款金额大于付款金额"),
    E90342("90342", "退款金额超过账户余额"),
    E90343("90343", "退款单号已经存在"),
    E90344("90344", "指定的付款渠道无效"),
    E90345("90345", "超过 5w 人民币不可指定微信支付渠道"),
    E90346("90346", "同一单的退款次数超过限制"),
    E90347("90347", "退款金额不可为 0"),
    E90348("90348", "管理端没配置支付密钥"),
    E90349("90349", "记录数量太大"),
    E90350("90350", "银行信息真实性校验失败"),
    E90351("90351", "应用状态异常"),
    E90352("90352", "延迟试用期天数超过限制"),
    E90353("90353", "预支单列表不可为空"),
    E90354("90354", "预支单列表数量超过限制"),
    E90355("90355", "关联有退款预支单，不可删除"),
    E90356("90356", "不能 0 金额下单"),
    E90357("90357", "代下单必须指定支付渠道"),
    E90358("90358", "预支单或代下单，不支持部分退款"),
    E90359("90359", "预支单与下单者企业不匹配"),
    E90381("90381", "参数 refunded_credit_orderid 不合法"),
    E90456("90456", "必须指定组织者"),
    E90457("90457", "日历 ID 异常"),
    E90458("90458", "日历 ID 列表不能为空"),
    E90459("90459", "日历已删除"),
    E90460("90460", "日程已删除"),
    E90461("90461", "日程 ID 异常"),
    E90462("90462", "日程 ID 列表不能为空"),
    E90463("90463", "不能变更组织者"),
    E90464("90464", "参与者数量超过限制"),
    E90465("90465", "不支持的重复类型"),
    E90466("90466", "不能操作别的应用创建的日历 / 日程"),
    E90467("90467", "星期参数异常"),
    E90468("90468", "不能变更组织者"),
    E90469("90469", "每页大小超过限制"),
    E90470("90470", "页数异常"),
    E90471("90471", "提醒时间异常"),
    E90472("90472", "没有日历 / 日程操作权限"),
    E90473("90473", "颜色参数异常"),
    E90474("90474", "组织者不能与参与者重叠"),
    E90475("90475", "不是组织者的日历"),
    E90479("90479", "不允许操作用户创建的日程"),
    E90500("90500", "群主并未离职"),
    E90501("90501", "该群不是客户群"),
    E90502("90502", "群主已经离职"),
    E90503("90503", "满人 & 99 个微信成员，没办法踢，要客户端确认"),
    E90504("90504", "群主没变"),
    E90507("90507", "离职群正在继承处理中"),
    E90508("90508", "离职群已经继承"),
    E90509("90509", "非企业微信客户群"),
    E90510("90510", "企业一年内无活跃用户"),
    E90511("90511", "opengid 不存在或者无效"),
    E90603("90603", "事件分类 id 不合法"),
    E90604("90604", "网格单元 id 不合法"),
    E90606("90606", "该网格单元管理员达到上限，一个网格单元最多有 20 个管理员"),
    E90607("90607", "含有成员的网格单元不能被删除"),
    E90608("90608", "网格单元的名字重复了"),
    E90609("90609", "网格单元的成员数超过上限"),
    E90610("90610", "网格单元的成员数超过上限"),
    E91040("91040", "获取 ticket 的类型无效"),
    E92000("92000", "成员不在应用可见范围之内"),
    E92001("92001", "应用没有敏感信息权限"),
    E92002("92002", "不允许跨企业调用"),
    E92006("92006", "该直播已经开始或取消"),
    E92007("92007", "该直播回放不能被删除"),
    E92008("92008", "当前应用没权限操作这个直播"),
    E93000("93000", "机器人 webhookurl 不合法或者机器人已经被移除出群"),
    E93004("93004", "机器人被停用"),
    E93008("93008", "不在群里"),
    E94000("94000", "应用未开启工作台自定义模式"),
    E94001("94001", "不合法的 type 类型"),
    E94002("94002", "缺少 keydata 字段"),
    E94003("94003", "keydata 的 items 列表长度超出限制"),
    E94005("94005", "缺少 list 字段"),
    E94006("94006", "list 的 items 列表长度超出限制"),
    E94007("94007", "缺少 webview 字段"),
    E94008("94008", "应用未设置自定义工作台模版类型"),
    E95000("95000", "不合法的 open_kfid"),
    E95001("95001", "发送客服消息次数限制"),
    E95002("95002", "发送客服消息时间限制"),
    E95003("95003", "发送客服消息可接待客户咨询数限制"),
    E95004("95004", "open_kfid 不存在"),
    E95005("95005", "客服帐号数超过限制"),
    E95006("95006", "不合法的客服帐号名"),
    E95007("95007", "不合法的 msgtoken"),
    E95008("95008", "菜单消息的菜单项个数超过上限"),
    E95009("95009", "不合法的菜单消息的菜单项类型"),
    E95012("95012", "未在企业微信使用微信客服"),
    E95013("95013", "会话已经结束"),
    E95014("95014", "用户不是接待人员"),
    E95015("95015", "管理端已经配置了专属服务"),
    E95016("95016", "不允许这种状态转移"),
    E95017("95017", "系统应用权限下，api 开关处于关闭状态"),
    E95018("95018", "发送客服消息时当前会话状态不允许发送"),
    E301002("301002", "无权限操作指定的应用"),
    E301005("301005", "不允许删除创建者"),
    E301012("301012", "参数 position 不合法"),
    E301013("301013", "参数 telephone 不合法"),
    E301014("301014", "参数 english_name 不合法"),
    E301015("301015", "参数 mediaid 不合法"),
    E301016("301016", "上传语音文件不符合系统要求"),
    E301017("301017", "上传语音文件仅支持 AMR 格式"),
    E301021("301021", "参数 userid 无效"),
    E301022("301022", "获取打卡数据失败"),
    E301023("301023", "useridlist 非法或超过限额"),
    E301024("301024", "获取打卡记录时间间隔超限"),
    E301025("301025", "审批开放接口参数错误"),
    E301036("301036", "不允许更新该用户的 userid"),
    E301039("301039", "请求参数错误，请检查输入参数"),
    E301042("301042", "ip 白名单限制，请求 ip 不在设置白名单范围"),
    E301048("301048", "sdkfileid 对应的文件不存在或已过期"),
    E301052("301052", "会话存档服务已过期"),
    E301053("301053", "会话存档服务未开启"),
    E301058("301058", "拉取会话数据请求超过大小限制，可减少 limit 参数"),
    E301059("301059", "非内部群，不提供数据"),
    E301060("301060", "拉取同意情况请求量过大，请减少到 100 个参数以下"),
    E301061("301061", "userid 或者 exteropenid 用户不存在"),
    E302003("302003", "批量导入任务的文件中 userid 有重复"),
    E302004("302004", "组织架构不合法（1 不是一棵树，2 多个一样的 partyid，3 partyid 空，4 partyid name 空，5 同一个父节点下有两个子节点 部门名字一样 可能是以上情况，请一一排查）"),
    E302005("302005", "批量导入系统失败，请重新尝试导入"),
    E302006("302006", "批量导入任务的文件中 partyid 有重复"),
    E302007("302007", "批量导入任务的文件中，同一个部门下有两个子部门名字一样"),
    E2000002("2000002", "CorpId 参数无效"),
    E600001("600001", "不合法的 sn"),
    E600002("600002", "设备已注册"),
    E600003("600003", "不合法的硬件 activecode"),
    E600004("600004", "该硬件尚未授权任何企业"),
    E600005("600005", "硬件 Secret 无效"),
    E600007("600007", "缺少硬件 sn"),
    E600008("600008", "缺少 nonce 参数"),
    E600009("600009", "缺少 timestamp 参数"),
    E600010("600010", "缺少 signature 参数"),
    E600011("600011", "签名校验失败"),
    E600012("600012", "长连接已经注册过设备"),
    E600013("600013", "缺少 activecode 参数"),
    E600014("600014", "设备未网络注册"),
    E600015("600015", "缺少 secret 参数"),
    E600016("600016", "设备未激活"),
    E600018("600018", "无效的起始结束时间"),
    E600020("600020", "设备未登录"),
    E600021("600021", "设备 sn 已存在"),
    E600023("600023", "时间戳已失效"),
    E600024("600024", "固件大小超过 5M"),
    E600025("600025", "固件名为空或者超过 20 字节"),
    E600026("600026", "固件信息不存在"),
    E600027("600027", "非法的固件参数"),
    E600028("600028", "固件版本已存在"),
    E600029("600029", "非法的固件版本"),
    E600030("600030", "缺少固件版本参数"),
    E600031("600031", "硬件固件不允许升级"),
    E600032("600032", "无法解析硬件二维码"),
    E600033("600033", "设备型号 id 冲突"),
    E600034("600034", "指纹数据大小超过限制"),
    E600035("600035", "人脸数据大小超过限制"),
    E600036("600036", "设备 sn 冲突"),
    E600037("600037", "缺失设备型号 id"),
    E600038("600038", "设备型号不存在"),
    E600039("600039", "不支持的设备类型"),
    E600040("600040", "打印任务 id 不存在"),
    E600041("600041", "无效的 offset 或 limit 参数值"),
    E600042("600042", "无效的设备型号 id"),
    E600043("600043", "门禁规则未设置"),
    E600044("600044", "门禁规则不合法"),
    E600045("600045", "设备已订阅企业信息"),
    E600046("600046", "操作 id 和用户 userid 不匹配"),
    E600047("600047", "secretno 的 status 非法"),
    E600048("600048", "无效的指纹算法"),
    E600049("600049", "无效的人脸识别算法"),
    E600050("600050", "无效的算法长度"),
    E600051("600051", "设备过期"),
    E600052("600052", "无效的文件分块"),
    E600053("600053", "该链接已经激活"),
    E600054("600054", "该链接已经订阅"),
    E600055("600055", "无效的用户类型"),
    E600056("600056", "无效的健康状态"),
    E600057("600057", "缺少体温参数"),
    E610001("610001", "永久二维码超过每个员工 5000 的限制"),
    E610003("610003", "scene 参数不合法"),
    E610004("610004", "userid 不在客户联系配置的使用范围内"),
    E610014("610014", "无效的 unionid"),
    E610015("610015", "小程序对应的开放平台账号未认证"),
    E610016("610016", "企业未认证"),
    E610017("610017", "小程序和企业主体不一致"),
    E640001("640001", "微盘不存在当前空间"),
    E640002("640002", "文件不存在"),
    E640003("640003", "文件已删除"),
    E640004("640004", "无权限访问"),
    E640005("640005", "成员不在空间内"),
    E640006("640006", "超出当前成员拥有的容量"),
    E640007("640007", "超出微盘的容量"),
    E640008("640008", "没有空间权限"),
    E640009("640009", "非法文件名"),
    E640010("640010", "超出空间的最大成员数"),
    E640011("640011", "json 格式不匹配"),
    E640012("640012", "非法的 userid"),
    E640013("640013", "非法的 departmentid"),
    E640014("640014", "空间没有有效的管理员"),
    E640015("640015", "不支持设置预览权限"),
    E640016("640016", "不支持设置文件水印"),
    E640017("640017", "微盘管理端未开通 API 权限"),
    E640018("640018", "微盘管理端未设置编辑权限"),
    E640019("640019", "API 调用次数超出限制"),
    E640020("640020", "非法的权限类型"),
    E640021("640021", "非法的 fatherid"),
    E640022("640022", "非法的文件内容的 base64"),
    E640023("640023", "非法的权限范围"),
    E640024("640024", "非法的 fileid"),
    E640025("640025", "非法的 space_name"),
    E640026("640026", "非法的 spaceid"),
    E640027("640027", "参数错误"),
    E640028("640028", "空间设置了关闭成员邀请链接"),
    E640029("640029", "只支持下载普通文件，不支持下载文件夹等其他非文件实体类型"),
    E844001("844001", "非法的 output_file_format"),

    ;
    private final String code;
    private final String desc;

    WxErrorCodeEnum(String code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    private static Map<String, WxErrorCodeEnum> valueMap = new HashMap<>();

    static {
        for (WxErrorCodeEnum value : WxErrorCodeEnum.values()) {
            valueMap.put(value.getCode(), value);
        }
    }

    public String getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }

    public static WxErrorCodeEnum getByCode(String code) {
        if (Objects.isNull(code)) {
            return null;
        }
        WxErrorCodeEnum wxErrorCodeEnum = valueMap.get(code);
        if (Objects.isNull(wxErrorCodeEnum)) {
            log.warn("wx error code miss {}", code);
            return WxErrorCodeEnum.E_10000;
        }
        return wxErrorCodeEnum;
    }


}
