package cn.com.duiba.scrm.common.util;

import cn.com.duiba.wolf.utils.UrlUtils2;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.*;

/**
 * @author lizhi
 * @date 2019/7/16 11:08 AM
 */
public class PushEmbedUtils {

    private PushEmbedUtils(){}

    private static final String SEPARATOR = "_";




    @Override
    public int hashCode() {
        return super.hashCode();
    }

    /**
     * 将推送参数值，转换成推送参数map
     * @param pushParamStr 推送参数值
     * @return 推送参数map
     */
    public static Map<String, String> getPushParamMapByStr(String pushParamStr) {//NOSONAR
        if (StringUtils.isBlank(pushParamStr)) {
            return Collections.emptyMap();
        }
        Long oaId = null;
        String pushId = null;
        Integer messageType = null;
        Integer urlOrder = null;
        String dpm = null;
        String[] split = pushParamStr.split(SEPARATOR);
        for (int i = 0; i < split.length; i++) {
            String str = split[i];
            if (StringUtils.isBlank(str)) {
                continue;
            }
            if (i == 0) {
                if (NumberUtils.isNumber(str)) {
                    oaId = Long.parseLong(str) - 3;
                } else {
                    oaId = IdMakerUtil.decodingId(str);
                }
            }
            if (i == 1) {
                pushId = str;
            }
            if (i == 2 && NumberUtils.isNumber(str)) {
                messageType = Integer.parseInt(str);
            }
            if (i == 3 && NumberUtils.isNumber(str)) {
                urlOrder = Integer.parseInt(str);
            }
            if (i == 4) {
                dpm = str;
            }
        }
        return getPushParamMap(oaId, pushId, messageType, urlOrder, dpm, null);
    }


    /**
     * 获取map中的推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数（不包含dpm）
     */
    public static String getPushParam(Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils2.buildUrlParams(pushParamMap);
    }





    /**
     * 增加推送唯一标识
     * @param page 小程序模板消息，跳转页面
     * @param markValue 唯一标识
     * @return 带参数的页面地址
     */
    public static String buildMpPushParam(String page, String markValue) {
        Map<String, String> params = new HashMap<>();
        if (markValue == null) {
            markValue = "";
        }
        params.put("unique_mark", markValue);
        return UrlUtils2.appendParams(page, params);
    }

    /**
     * URL追加map中的推送埋点参数（不包含dpm）
     * @param url 链接
     * @param parameterMap request中的所有参数
     * @return 带推送参数的URL
     */
    public static String buildPushParam(String url, Map<String, String[]> parameterMap) {
        Map<String, String> pushParamMap = getPushParamMap(parameterMap);
        return UrlUtils2.appendParams(url, pushParamMap);
    }

    /**
     * 获取请求中推送埋点参数（不包含dpm）
     * @param parameterMap request中的所有参数
     * @return 推送埋点参数map
     */
    public static Map<String, String> getPushParamMap(Map<String, String[]> parameterMap) {
        if (MapUtils.isEmpty(parameterMap)) {
            return Collections.emptyMap();
        }
        Map<String, String> params = new HashMap<>(16);
        Set<String> pushKey = getPushParamMap(0L, "", 0, 0, null, "").keySet();
        for (String key : pushKey) {
            String[] param = parameterMap.get(key);
            if (isNotEmpty(param)) {
                params.put(key, param[0]);
            }
        }
        return params;
    }

    /**
     * 根据推送事件，获取推送点击埋点
     * @param eventEnum 推送事件
     * @return 点击埋点
     */
    public static String getDpmByPushEventEnum(String eventEnum) {//NOSONAR
        if (eventEnum == null) {
            return "";
        }
        if (Objects.equals(eventEnum, "D3")) {
            return "11.9.0.0";
        }
        if (Objects.equals(eventEnum, "D4")) {
            return "11.10.0.0";
        }
        if (Objects.equals(eventEnum, "S1")) {
            return "11.12.0.0";
        }
        if (Objects.equals(eventEnum, "S5")) {
            return "11.14.0.0";
        }
        if (Objects.equals(eventEnum, "S6")) {
            return "11.15.0.0";
        }
        if (Objects.equals(eventEnum, "S7")) {
            return "11.16.0.0";
        }
        if (Objects.equals(eventEnum, "S8")) {
            return "11.17.0.0";
        }
        if (Objects.equals(eventEnum, "S9")) {
            return "11.18.0.0";
        }
        if (Objects.equals(eventEnum, "S10")) {
            return "11.19.0.0";
        }
        if (Objects.equals(eventEnum, "S11")) {
            return "11.20.0.0";
        }
        if (Objects.equals(eventEnum, "V1")) {
            return "11.22.0.0";
        }
        if (Objects.equals(eventEnum, "V2")) {
            return "11.23.0.0";
        }
        if (Objects.equals(eventEnum, "V4")) {
            return "11.24.0.0";
        }
        if (Objects.equals(eventEnum, "S12")) {
            return "11.25.0.0";
        }
        if (Objects.equals(eventEnum, "S13")) {
            return "11.26.0.0";
        }
        if (Objects.equals(eventEnum, "S14")) {
            return "11.27.0.0";
        }
        if (Objects.equals(eventEnum, "S15")) {
            return "11.28.0.0";
        }
        if (Objects.equals(eventEnum, "S17")) {
            return "11.30.0.0";
        }
        if (Objects.equals(eventEnum, "S18")) {
            return "11.31.0.0";
        }
        if (Objects.equals(eventEnum, "S19")) {
            return "11.32.0.0";
        }
        if (Objects.equals(eventEnum, "S4")) {
            return "11.34.0.0";
        }
        if (Objects.equals(eventEnum, "S27")) {
            return "11.37.0.0";
        }
        //如果dpm没有特殊要求，请不要再加if，走下面的方法自动计算dpm
        return calculateDpm(eventEnum);
    }



    /**
     * 封装推送埋点参数map
     * @param oaId 公众号ID
     * @param pushId 推送事件Id
     * @param messageType 推送消息类型
     * @param urlOrder 推送中第几个链接
     * @param dpm dpm，为空则不拼接dpm参数
     * @return 推送埋点参数map
     */
    private static Map<String, String> getPushParamMap(Long oaId, String pushId, Integer messageType, Integer urlOrder, String dpm, String markValue) {
        Map<String, String> params = new HashMap<>(16);
        params.put("push", "push");
        if (Objects.nonNull(oaId)) {
            params.put("oa_id", IdMakerUtil.encodingIdByBase64(oaId));
        }
        if (Objects.nonNull(pushId)) {
            params.put("push_id", pushId);
        }
        if (Objects.nonNull(messageType)) {
            params.put("message_type", String.valueOf(messageType));
        }
        if (Objects.nonNull(urlOrder)) {
            params.put("url_order", String.valueOf(urlOrder));
        }
        params.put("access_source", getAccessSourceByPushId(pushId));
        if (StringUtils.isNotBlank(dpm)) {
            params.put("dpm", dpm);
        }
        if (markValue != null) {
            params.put("unique_mark", markValue);
        }
        return params;
    }

    /**
     * 根据pushId获取accessSource
     * @param pushId
     * @return
     */
    private static String getAccessSourceByPushId(String pushId) {
        if (StringUtils.isBlank(pushId) || pushId.length() < 2) {
            return "1";
        }
        //根据pushId的首字母，得出数字
        String numByPushId = getNumByPushId(pushId);
        if (StringUtils.isBlank(numByPushId)) {
            return "1";
        }
        //accessSource= 10 + 首字母对应数字 + pushId的数字（不足3位补0）
        String accessSource = "10";
        accessSource += numByPushId;
        String num = pushId.substring(1);
        while (num.length() < 3) {
            num = "0" + num;
        }
        return accessSource + num;
    }

    /**
     * 根据pushId的首字母获取对应数字
     * @param pushId 推送ID
     * @return 首字母对应数字
     */
    private static String getNumByPushId(String pushId) {
        if (StringUtils.isBlank(pushId) || pushId.length() < 1) {
            return "";
        }
        String word = pushId.substring(0, 1);
        switch (word) {
            case "S":
                return "1";
            case "D":
                return  "2";
            case "H":
                return  "3";
            case "V":
                return  "4";
            case "X":
                return  "5";
            case "Z":
                return "6";
            case "L":
                return "7";
            default:
                return  "";
        }
    }

    /**
     * 根据eventEnum计算dpm
     * @param eventType
     * @return
     */
    private static String calculateDpm(String eventType) {
        //dpm由4位组成，用"."隔开
        //规则：11.0. + 推送ID首字母对应数字 + . + 推送ID数字
        if (StringUtils.isBlank(eventType)) {
            return "";
        }
        if (eventType.length() < 2) {
            return "";
        }
        String numByPushId = getNumByPushId(eventType);
        if (StringUtils.isBlank(numByPushId)) {
            return "";
        }
        return "11.0." + numByPushId + "." + eventType.substring(1);
    }

    /**
     * URL编码
     * @param url 链接
     * @return 编码后的链接
     */
    public static String encode(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    /**
     * URL解码
     * @param url 链接
     * @return 解码后的链接
     */
    public static String decode(String url) {
        try {
            return URLDecoder.decode(url, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            return url;
        }
    }

    private static boolean isNotEmpty(String[] param) {
        return Objects.nonNull(param) && param.length > 0;
    }

    /**
     * 给文本信息插入唯一标识
     * @param msg 文本信息
     * @param eventType 推送事件
     * @return 插入唯一标识后的文本
     */
    public static String insertUniqueMark(String msg, String eventType, String markValue) {
        if (StringUtils.isBlank(msg) || eventType == null) {
            return msg;
        }
        String pushIdKey = "push_id";
        //最终结果收集
        StringBuilder sb = new StringBuilder();
        //push_id的长度
        int pushIdKeyLength = pushIdKey.length();
        //push_id的值
        //push_id值的长度
        int eventTypeLength = eventType.length();
        int lastIndex = 0;
        while (lastIndex < msg.length()) {
            //push_id开始的下标
            int pushIdIndex = msg.indexOf(pushIdKey, lastIndex);
            if (pushIdIndex < 0) {
                //没有push_id
                break;
            }
            //push_id值开始的下标
            int eventTypeIndex = msg.indexOf(eventType, pushIdIndex);
            if (eventTypeIndex < 0) {
                //没有push_id值，非推送参数
                break;
            }
            //push_id结束的下标
            int pushIdLastIndex = pushIdIndex + pushIdKeyLength;
            //push_id值结束的下标
            int eventTypeLastIndex = eventTypeIndex + eventTypeLength;
            //获取push_id和值之间的符号，用于判断是否编码过
            String equalSign = msg.substring(pushIdLastIndex, eventTypeIndex);
            //获取编码次数
            int count = encodeCount(equalSign);
            //根据编码次数，获取要插入的唯一标识
            String mark = getMark(count, markValue);
            //追加push_id之前的数据
            sb.append(msg, lastIndex, pushIdIndex);
            //追加唯一标识
            sb.append(mark);
            //追加push_id及push_id的值
            sb.append(msg, pushIdIndex, eventTypeLastIndex);
            //将处理下标移动到push_id值后
            lastIndex = eventTypeLastIndex;
        }
        sb.append(msg.substring(lastIndex));
        return sb.toString();
    }

    /**
     * 根据编码次数，获取唯一标识
     * @param encodeCount 编码次数
     * @return 唯一标识
     */
    private static String getMark(int encodeCount, String markValue) {
        String mark = "unique_mark=" + markValue + "&";
        if (encodeCount <= 0) {
            return mark;
        }
        for (int i = 0; i < encodeCount; i++) {
            mark = encode(mark);
        }
        return mark;
    }

    /**
     * 获取编码次数
     * @param equalSign 编码后的 =
     * @return 编码次数
     */
    private static int encodeCount(String equalSign) {
        int encodeCount = 0;
        if (StringUtils.isBlank(equalSign)) {
            return encodeCount;
        }
        String sign = "=";
        int maxCount = 5;
        for (int i = 0; i < maxCount; i++) {
            if (StringUtils.equals(equalSign, sign)) {
                return encodeCount;
            }
            equalSign = decode(equalSign);
            encodeCount++;
        }
        return encodeCount;
    }
}
