package cn.com.duiba.scrm.wechat.service.proxy;

import cn.com.duiba.scrm.wechat.service.annotation.HttpBean;
import cn.com.duiba.scrm.wechat.service.annotation.HttpFile;
import cn.com.duiba.scrm.wechat.service.annotation.HttpParam;
import cn.com.duiba.scrm.wechat.service.bo.FileParam;
import cn.com.duiba.scrm.wechat.service.bo.HttpHeader;
import cn.com.duiba.scrm.wechat.service.bo.HttpRequestParam;
import cn.com.duiba.scrm.wechat.service.bo.HttpUrl;
import cn.com.duiba.scrm.wechat.service.bo.MethodParamResult;
import cn.com.duiba.scrm.wechat.service.bo.NameValueParam;
import cn.com.duiba.scrm.wechat.service.bo.UploadFile;
import cn.com.duiba.scrm.wechat.service.enums.HttpRequestMethod;
import cn.com.duiba.scrm.wechat.service.exception.ParamException;
import cn.com.duiba.scrm.wechat.service.factorybean.HttpFactoryBean;
import com.alibaba.fastjson.JSON;
import org.apache.commons.lang.StringUtils;
import org.springframework.web.multipart.MultipartFile;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

/**
 * @author linzhou
 * @version 1.0.0
 * @ClassName AbstractHttpProxy.java
 * @Description TODO
 * @createTime 2021年07月06日 13:36:00
 */
public abstract class AbstractHttpProxy implements HttpProxy {


    /**
     * 获取请求方式
     *
     * @param httpFactoryBean
     * @param methodAnnotation
     * @return
     */
    protected HttpRequestMethod getHttpRequestMethod(HttpFactoryBean httpFactoryBean, HttpBean methodAnnotation) {
        //从方法注解中获取请求类型
        HttpRequestMethod httpRequestMethod = methodAnnotation != null ? methodAnnotation.method() : null;
        if (httpRequestMethod == null || httpRequestMethod == HttpRequestMethod.NULL) {
            //如果方法注解中没有获取到请求类型则从类注解中获取请求类型
            httpRequestMethod = httpFactoryBean.getMethod();
        }
        if (httpRequestMethod == HttpRequestMethod.NULL) {
            //如果方法注解和类注解中都没有标注请求类型,则默认get
            httpRequestMethod = HttpRequestMethod.GET;
        }
        return httpRequestMethod;
    }


    /**
     * 解析方法参数
     *
     * @param param
     * @return
     */
    protected MethodParamResult analysisMethodParam(HttpRequestParam param) {
        Object[] args = param.getArgs();
        Annotation[][] parameterAnnotations = param.getParameterAnnotations();
        MethodParamResult result = new MethodParamResult();

        HttpHeader httpHeader = new HttpHeader();
        if (args != null) {

            for (int i = 0; i < args.length; i++) {
                Object arg = args[i];
                if (arg == null) {
                    continue;
                }
                Annotation[] parameterAnnotation = parameterAnnotations[i];
                HttpParam httpParam = getHttpAnnotation(parameterAnnotation, HttpParam.class);
                HttpFile httpFile = getHttpAnnotation(parameterAnnotation, HttpFile.class);
                if (httpParam != null) {
                    //处理表单参数
                    NameValueParam nameValueParam = getNameValueParam(i, arg, httpParam);
                    result.addNameValueParam(nameValueParam);
                } else if (arg instanceof HttpHeader) {
                    //设置请求头
                    httpHeader.addHeader((HttpHeader) arg);
                } else if (httpFile != null) {
                    //处理文件上传
                    result.setUploadFile( getUploadFile(i, arg, httpFile));
                } else if (arg instanceof HttpUrl) {
                    result.setHttpUrl((HttpUrl) arg);
                } else {
                    //处理body
                    result.setBody(getBody(arg));
                }
            }
        }

        result.setHttpHeader(httpHeader);
        return result;
    }

    /**
     * 处理body
     *
     * @param arg
     * @return
     */
    private String getBody(Object arg) {
        String body;
        if (arg instanceof String) {
            body = (String) arg;
        } else {
            body = JSON.toJSONString(arg);
        }
        return body;
    }

    /**
     * 处理文件上传
     *
     * @param i
     * @param arg
     * @param httpFile
     * @return
     */
    private UploadFile getUploadFile(int i, Object arg, HttpFile httpFile) {
        UploadFile uploadFile;
        //保存需要上传的文件信息
        if (arg instanceof MultipartFile) {
            uploadFile = new UploadFile(httpFile, (MultipartFile) arg);
        } else if (arg instanceof FileParam) {
            FileParam fileParam = (FileParam) arg;
            uploadFile = new UploadFile(httpFile, fileParam.getFile(), fileParam.getParam());
        } else {
            throw new ParamException("第" + i + "个参数格式错误,上传文件应为MultipartFile类型,当前类型:" + arg.getClass().getName());
        }
        return uploadFile;
    }

    /**
     * 处理表单
     */
    private NameValueParam getNameValueParam(int i, Object arg, HttpParam httpParam) {
        //如果是表单参数,则当做表单处理
        String name = httpParam.value();
        if (StringUtils.isBlank(name)) {
            throw new ParamException("第" + i + "个参数格式错误,没有发现表单参数对应的名称");
        }
        String value;
        if (arg == null) {
            value = null;
        } else if (isNameValuePair(arg)) {
            value = arg.toString();
        } else {
            value = JSON.toJSONString(arg);
        }
        NameValueParam nameValueParam = new NameValueParam(httpParam, value);
        return nameValueParam;
    }

    /**
     * 创建get请求url
     *
     * @param url
     * @param nameValueParams
     * @return
     */
    protected String getNewGetUrl(String url, List<NameValueParam> nameValueParams) {

        if (nameValueParams == null || nameValueParams.isEmpty()) {
            return url;
        }
        if (!url.contains("?")) {
            url = url + "?";
        }
        int lastIndex = url.length() - 1;
        if (url.lastIndexOf("?") != lastIndex && url.charAt(lastIndex) != '&') {
            //如果最后一位不是?说明url中已经带了参数,但是最后一位不是&,补一个
            url = url + "&";
        }

        StringBuilder sb = new StringBuilder();
        for (NameValueParam nameValueParam : nameValueParams) {
            sb.append(nameValueParam.getName()).append("=").append(nameValueParam.getValue()).append("&");
        }

        return url + sb.toString();
    }


    /**
     * 获取请求地址
     *
     * @param param
     * @param result
     * @return
     */
    protected String getUrl(HttpRequestParam param, MethodParamResult result) {

        if (result.getHttpUrl()!=null){
            return result.getHttpUrl().getUrl();
        }

        HttpBean methodAnnotation = param.getMethodAnnotation();
        HttpFactoryBean httpFactoryBean = param.getHttpFactoryBean();
        Method method = param.getMethod();

        String url = methodAnnotation == null ? null : methodAnnotation.url();
        if (StringUtils.isBlank(url)) {
            url = httpFactoryBean.getUrl();
        }

        if (StringUtils.isBlank(url)) {
            throw new IllegalArgumentException("url:格式错误,url is blank");
        }


        if (!url.contains("://")) {
            url = "https://" + url;
        }
        //如果方法没有注解,则取方法名称作为路径
        String path = "";

        if (methodAnnotation != null) {
            path = methodAnnotation.path();
            if (StringUtils.isBlank(path) && methodAnnotation.pathMethodName()) {
                //如果path为空,但是注解标注使用方法名,则path使用方法名称
                path = method.getName();
            }
        } else if (httpFactoryBean.isPathMethodName()) {
            //如果类注解上标注了使用方法名称
            path = method.getName();
        }

        String separate = "/";
        if (url.lastIndexOf(separate) != url.length() - 1 && path.indexOf(separate) != 0) {
            //如果url最后没有"/",并且path也没有,则添加一个"/"
            url += separate;
        } else if (url.lastIndexOf(separate) == url.length() - 1 && path.indexOf(separate) == 0) {
            //如果url最后有"/",并且path也有,则删除一个"/"
            url = url.substring(0, url.length() - 1);
        }
        url = url + path;
        try {
            //检测url格式
            new URL(url);
        } catch (MalformedURLException e) {
            throw new IllegalArgumentException("url:格式错误" + url, e);
        }

        return url;
    }

    protected boolean isNameValuePair(Object o) {
        if (o instanceof Integer
                || o instanceof String
                || o instanceof Double
                || o instanceof Float
                || o instanceof Long
                || o instanceof BigDecimal) {
            return true;
        }
        return false;
    }


    public <T extends Annotation> T getHttpAnnotation(Annotation[] annotations, Class<T> clazz) {
        for (Annotation annotation : annotations) {
            if (clazz.isInstance(annotation)) {
                return (T) annotation;
            }
        }
        return null;
    }

}
