package cn.com.duiba.scrm.wechat.service.proxy;

import cn.com.duiba.scrm.wechat.service.annotation.HttpBean;
import cn.com.duiba.scrm.wechat.service.annotation.HttpFile;
import cn.com.duiba.scrm.wechat.service.bo.HttpHeader;
import cn.com.duiba.scrm.wechat.service.bo.HttpRequestParam;
import cn.com.duiba.scrm.wechat.service.bo.MethodParamResult;
import cn.com.duiba.scrm.wechat.service.bo.NameValueParam;
import cn.com.duiba.scrm.wechat.service.bo.UploadFile;
import cn.com.duiba.scrm.wechat.service.enums.HttpRequestMethod;
import cn.com.duiba.scrm.wechat.service.factorybean.HttpFactoryBean;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author linzhou
 * @version 1.0.0
 * @ClassName DefaultHttpProxy.java
 * @Description TODO
 * @createTime 2021年07月05日 16:25:00
 */
@Slf4j
public class DefaultHttpProxy extends AbstractHttpProxy implements InvocationHandler {

    private HttpFactoryBean httpFactoryBean;

    private CloseableHttpClient client;

    public static final String DEFAULT_ENCODING = "UTF-8";

    public DefaultHttpProxy(HttpFactoryBean httpFactoryBean) {
        this.httpFactoryBean = httpFactoryBean;
        this.client = httpFactoryBean.getApplicationContext().getBean(CloseableHttpClient.class);
    }

    @Override
    public <T> T newProxyInstance() {
        Class<?> clazz = httpFactoryBean.getType();
        return (T) Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class[]{clazz}, this);
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {

        HttpRequestParam param = new HttpRequestParam(httpFactoryBean,method,args);
        //获取url
//        String url = getUrl(method, httpFactoryBean, methodAnnotation);
        //获取请求类型
        HttpRequestMethod httpRequestMethod = getHttpRequestMethod(httpFactoryBean, param.getMethodAnnotation());

        param.setHttpRequestMethod(httpRequestMethod);

        HttpRequestBase httpRequest = getHttpRequestBase(param);

        try (CloseableHttpResponse httpResponse = client.execute(httpRequest)) {

            return getReturnObject(httpResponse, method.getReturnType());
        } catch (Exception e) {
            throw e;
        }
    }


    private Object getReturnObject(CloseableHttpResponse httpResponse, Class<?> returnType) throws IOException {

        if (returnType == MultipartFile.class) {
            //如果是文件下载
            byte[] bytes = EntityUtils.toByteArray(httpResponse.getEntity());
            InputStream inputStream = new ByteArrayInputStream(bytes);
            //创建文件
            return new MockMultipartFile(ContentType.APPLICATION_OCTET_STREAM.toString(), inputStream);
        }

        String result = EntityUtils.toString(httpResponse.getEntity());
        log.info("result:{}", result.length() > 2000 ? "过长" : result);
        if (returnType == String.class) {
            return result;
        }
        if (returnType == Integer.class) {
            return Integer.parseInt(result);
        }
        if (returnType == Double.class) {
            return Double.parseDouble(result);
        }
        if (returnType == Float.class) {
            return Float.parseFloat(result);
        }
        if (returnType == Long.class) {
            return Long.parseLong(result);
        }
        if (returnType == BigDecimal.class) {
            return new BigDecimal(result);
        }

        return JSON.parseObject(result, returnType);
    }

    /**
     * 获取httpRequest
     *
     * @param param
     * @return
     * @throws UnsupportedEncodingException
     */
    private HttpRequestBase getHttpRequestBase(HttpRequestParam param) throws Throwable {
        //解析方法参数
        MethodParamResult result = analysisMethodParam(param);
        //创建HttpRequest
        HttpRequestBase httpRequest = createHttpRequest(param, result);
        //设置请求头 提交带文件请求有影响
//        setHeaders(result.getHttpHeader(), httpRequest);
        return httpRequest;
    }


    /**
     * 创建HttpRequest
     *
     * @param param
     * @param result
     * @return
     * @throws UnsupportedEncodingException
     */
    private HttpRequestBase createHttpRequest(HttpRequestParam param, MethodParamResult result) throws Throwable {
        String url = getUrl(param, result);
        if (HttpRequestMethod.POST == param.getHttpRequestMethod()) {
            return getHttpPost(url, result);
        } else {
            return getHttpGet(url, result);
        }
    }

    /**
     * 创建get请求
     *
     * @param url
     * @param result
     * @return
     */
    private HttpRequestBase getHttpGet(String url, MethodParamResult result) {
        url = getNewGetUrl(url, result.getNameValueParams());
        log.info("url:{}", url);
        return new HttpGet(url);
    }

    /**
     * 创建post请求
     *
     * @param url
     * @param result
     * @return
     * @throws UnsupportedEncodingException
     */
    private HttpRequestBase getHttpPost(String url, MethodParamResult result) throws Throwable {
        List<NameValueParam> nameValueParams = result.getNameValueParams();
        String body = result.getBody();
        if (!nameValueParams.isEmpty()) {
            //如果同时又body和表单参数,则将表单参数拼接到url上
            url = getNewGetUrl(url, nameValueParams);
        }
        log.info("url:{},result:{},", url, JSONObject.toJSONString(result.getBody()));
        HttpPost httpPost = new HttpPost(url);
        if (StringUtils.isBlank(body) && !nameValueParams.isEmpty()) {
            //如果没有body,并且有表单参数,则添加表单参数
            List<NameValuePair> params = new ArrayList<>();
            for (NameValueParam nameValueParam : nameValueParams) {
                params.add(new BasicNameValuePair(nameValueParam.getName(), nameValueParam.getValue()));
            }
            httpPost.setEntity(new UrlEncodedFormEntity(params, Charset.forName("UTF-8")));
        }
        HttpEntity entity = getHttpEntity(result);
        if (entity != null) {
            //设置entity
            httpPost.setEntity(entity);
        }
        return httpPost;
    }

    private HttpEntity getHttpEntity(MethodParamResult result) throws Throwable {
        if (StringUtils.isNotBlank(result.getBody())) {
            //设置body
            return new StringEntity(result.getBody(), DEFAULT_ENCODING);
        }
        UploadFile uploadFile = result.getUploadFile();
        if (uploadFile != null) {
            //设置文件上传
            MultipartFile file = uploadFile.getFile();
            HttpFile httpFile = uploadFile.getHttpFile();
            Map<String, String> param = uploadFile.getParam();
            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            //加上此行代码解决返回中文乱码问题
            builder.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
            ByteArrayBody byteArrayBody = new ByteArrayBody(file.getBytes(), file.getName());
            String value = StringUtils.isBlank(httpFile.value()) ? "file" : httpFile.value();
            builder.addPart(value, byteArrayBody);
            if (param != null && !param.isEmpty()) {
                for (Map.Entry<String, String> paramEntry : param.entrySet()) {
                    builder.addTextBody(paramEntry.getKey(), paramEntry.getValue());
                }
            }
            return builder.build();
        }

        return null;
    }


    /**
     * 设置请求头
     *
     * @param httpHeader
     * @param httpRequest
     */
    private void setHeaders(HttpHeader httpHeader, HttpRequestBase httpRequest) {
        httpRequest.setHeader("Content-Type", "application/x-www-form-urlencoded;charset=utf-8");
        httpRequest.setHeader("User-Agent", "Apache-HttpClient/4.1.1(java 1.5)");
        httpRequest.setHeader("Connection", "Keep-Alive");

        for (String headerKey : httpHeader.getHeaders().keySet()) {
            httpRequest.setHeader(headerKey, httpHeader.getHeaders().get(headerKey));
        }
    }


    private ContentType getContentType(String mimeType) {
        ContentType contentType;
        if (StringUtils.isBlank(mimeType)) {
            contentType = ContentType.DEFAULT_BINARY;
        } else {
            contentType = ContentType.create(mimeType, (Charset) null);
        }
        return contentType;
    }

}
