package cn.com.duiba.sign.center.api.enums.creditssign;


import cn.com.duiba.sign.center.api.dto.SignRewardRuleDto;
import cn.com.duiba.sign.center.api.dto.SignRewardRuleItemDto;
import cn.com.duiba.sign.center.api.exception.SignCenterException;
import org.apache.commons.collections.CollectionUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 签到奖励类型
 * Created by xiaoxuda on 2017/11/14.
 */
public enum SignRewardTypeEnum {
    CONTINUE(1, "连续奖励") {
        @Override
        protected SignRewardRuleItemDto getRuleItemSub(int continueDay, SignRewardRuleDto rewardRule) {
            int circle = rewardRule.getItems().size();
            int index = continueDay > circle ? circle : continueDay;
            return rewardRule.getItems().get(index - 1);
        }
    },
    CIRCLE(2, "周期奖励") {
        @Override
        protected SignRewardRuleItemDto getRuleItemSub(int continueDay, SignRewardRuleDto rewardRule) {
            int circle = rewardRule.getItems().size();
            int index = continueDay % circle;
            index = index == 0 ? circle : index;
            return rewardRule.getItems().get(index - 1);
        }
    },
    ACCUMULATE(3, "累计奖励") {
        @Override
        protected SignRewardRuleItemDto getRuleItemSub(int continueDay, SignRewardRuleDto rewardRule) {
            for (SignRewardRuleItemDto item : rewardRule.getItems()) {
                if (item.getDays() == null) {
                    continue;
                }
                if (Objects.equals(item.getDays(), continueDay)) {
                    return item;
                }
            }
            return new SignRewardRuleItemDto();
        }
    },
    SPEC_DAY(4, "特殊日期奖励"){
        @Override
        protected SignRewardRuleItemDto getRuleItemSub(int continueDay, SignRewardRuleDto rewardRule) {
            //需要外部单独写处理逻辑
            return new SignRewardRuleItemDto();
        }
    },
    FIRST_SIGN(5, "首次签到奖励"){
        @Override
        protected SignRewardRuleItemDto getRuleItemSub(int continueDay, SignRewardRuleDto rewardRule) {
            //需要外部单独写处理逻辑
            return new SignRewardRuleItemDto();
        }
    };

    private Integer code;
    private String desc;

    private static Map<Integer, SignRewardTypeEnum> typeMap = new HashMap<>();
    static{
        for(SignRewardTypeEnum rwType : values()){
            typeMap.put(rwType.getCode(), rwType);
        }
    }

    /**
     * 通过code查询类型信息
     * @param type
     * @return
     */
    public static SignRewardTypeEnum getByCode(Integer type){
        return typeMap.get(type);
    }

    SignRewardTypeEnum(Integer code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    /**
     * 根据签到规则计算连续签到第continueDay天所适用的签到奖励条目
     *
     * @param continueDay    签到完成后周期内连续签到天数,累积奖励时需要传入周期内累计的签到天数
     * @param rewardRule 奖励规则
     * @return
     */
    public SignRewardRuleItemDto getRuleItem(Integer continueDay, SignRewardRuleDto rewardRule) {
        if (continueDay < 0 || rewardRule == null ) {
            throw new SignCenterException("参数非法");
        }
        if(!Boolean.TRUE.equals(rewardRule.getOpen()) || CollectionUtils.isEmpty(rewardRule.getItems())){
            return new SignRewardRuleItemDto();
        }
        return this.getRuleItemSub(continueDay , rewardRule);
    }

    protected abstract SignRewardRuleItemDto getRuleItemSub(int continueDay, SignRewardRuleDto rewardRule);

    public Integer getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }
}
