package cn.com.duiba.sign.center.api.enums.creditssign;



import cn.com.duiba.sign.center.api.dto.SignActivityConfigDto;
import cn.com.duiba.sign.center.api.dto.SignRuleConfigDto;
import cn.com.duiba.sign.center.api.exception.SignCenterException;
import cn.com.duiba.wolf.utils.DateUtils;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 签到类型
 * Created by xiaoxuda on 2017/5/31.
 */
public enum SignTypeEnum {
    CONTINUE(0, "连续签到"){
        @Override
        public Calendar getCirStartDate(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo){
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(signInfo.signDate);
            calendar.add(Calendar.DAY_OF_MONTH, 1 - getCircleDays(signActConfig, signInfo));
            return calendar;
        }
        @Override
        public Calendar getCirEndDate(Calendar cirStartDate, SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo){
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(signInfo.signDate);
            return calendar;
        }

        @Override
        public Integer getCircleDays(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo) {
            return DateUnit.MONTH.getDays();
        }

        @Override
        public Integer getContinueDayInCircle(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo) {
            return signInfo.cntDaysAfterSigned;
        }
    },
    CIRCLE(1, "连续周期签到") {

        @Override
        public Calendar getCirStartDate(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo){
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(signInfo.signDate);
            calendar.add(Calendar.DAY_OF_MONTH, 1 - getContinueDayInCircle(signActConfig, signInfo));
            return calendar;
        }

        @Override
        public Calendar getCirEndDate(Calendar cirStartDate, SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo){
            Calendar cirEndDate = Calendar.getInstance();
            cirEndDate.setTime(cirStartDate.getTime());
            cirEndDate.add(Calendar.DAY_OF_MONTH, getCircleDays(signActConfig, signInfo) - 1);
            return cirEndDate;
        }

        @Override
        public Integer getContinueDayInCircle(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo) {
            int cirDays = getCircleDays(signActConfig, signInfo);
            int days = signInfo.cntDaysAfterSigned % cirDays;
            if (days == 0) {
                days = cirDays;
            }
            return days;
        }

        @Override
        public Integer getCircleDays(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo){
            SignRuleConfigDto ruleConfig = signActConfig.getSignRule();
            int cirDays = 1;
            if(ruleConfig.getCusCirDays() != null){
                cirDays = ruleConfig.getCusCirDays();
            }
            return cirDays;
        }


    },
    CUSTOM_CIRCLE(4, "自定义周期") {
        @Override
        public Calendar getCirStartDate(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo) {
            SignRuleConfigDto ruleConfig = signActConfig.getSignRule();
            Calendar cusCirStart = Calendar.getInstance();
            cusCirStart.setTime(ruleConfig.getCusCirStart());
            Calendar startDate = Calendar.getInstance();
            startDate.setTime(signInfo.signDate);
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(ruleConfig.getCusCirType())) {
                int cirNum = (DateUtils.daysBetween(cusCirStart.getTime(), signInfo.signDate) + 1) % ruleConfig.getCusCirDays();
                if (cirNum == 0) {
                    startDate.add(Calendar.DAY_OF_MONTH, 1 - ruleConfig.getCusCirDays());
                } else {
                    startDate.add(Calendar.DAY_OF_MONTH, 1 - cirNum);
                }
            } else if (CustomCircleTypeEnum.MONTH.equals(ruleConfig.getCusCirType())) {
                startDate.set(Calendar.DAY_OF_MONTH, cusCirStart.get(Calendar.DAY_OF_MONTH));
                int dayGap = signInfo.signDateCalendar.get(Calendar.DAY_OF_MONTH) - cusCirStart.get(Calendar.DAY_OF_MONTH);
                if (dayGap < 0) {
                    startDate.add(Calendar.MONTH, -1);
                }
            }
            return startDate;
        }

        @Override
        public Calendar getCirEndDate(Calendar cirStartDate, SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo) {
            SignRuleConfigDto ruleConfig = signActConfig.getSignRule();
            Calendar endDate = Calendar.getInstance();
            endDate.setTime(cirStartDate.getTime());
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(ruleConfig.getCusCirType())) {
                endDate.add(Calendar.DAY_OF_MONTH, ruleConfig.getCusCirDays() - 1);
            } else if (CustomCircleTypeEnum.MONTH.equals(ruleConfig.getCusCirType())) {
                endDate.add(Calendar.DAY_OF_MONTH, cirStartDate.getActualMaximum(Calendar.DAY_OF_MONTH) - 1);
            }
            return endDate;
        }

        @Override
        public Integer getCircleDays(SignActivityConfigDto signActConfig, SignTypeEnum.SignInfo signInfo) {
            SignRuleConfigDto ruleConfig = signActConfig.getSignRule();
            if (CustomCircleTypeEnum.FIXED_DAYS.equals(ruleConfig.getCusCirType())) {
                return ruleConfig.getCusCirDays();
            } else if (CustomCircleTypeEnum.MONTH.equals(ruleConfig.getCusCirType())) {
                return getCirStartDate(signActConfig, signInfo).getActualMaximum(Calendar.DAY_OF_MONTH);
            }
            throw new SignCenterException("不支持的自定义周期类型, cusCirType=" + ruleConfig.getCusCirType());
        }

        @Override
        public Integer getContinueDayInCircle(SignActivityConfigDto signActConfig, SignInfo signInfo) {
            Calendar cirStartDate = getCirStartDate(signActConfig, signInfo);
            int days = DateUtils.daysBetween(cirStartDate.getTime(), signInfo.signDate) + 1;
            int continueDay = signInfo.cntDaysAfterSigned;
            if (continueDay >= days) {
                continueDay = days;
            }
            return continueDay;
        }

    };

    private static Map<Integer, SignTypeEnum> typeEnumMap = new HashMap<>();

    static {
        for (SignTypeEnum typeEnum : values()) {
            typeEnumMap.put(typeEnum.getCode(), typeEnum);
        }
    }

    /**
     * 通过code获取签到类型
     *
     * @param code
     * @return
     */
    public static SignTypeEnum getByCode(int code) {
        SignTypeEnum type = typeEnumMap.get(code);
        if (type == null) {
            throw new SignCenterException("不支持的签到类型");
        }
        return type;
    }

    private Integer code;
    private String desc;

    SignTypeEnum(Integer code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    /**
     * 周期开始日期
     * @param signActConfig
     * @param signInfo
     * @return
     */
    public abstract Calendar getCirStartDate(SignActivityConfigDto signActConfig, SignInfo signInfo);

    /**
     * 周期结束日期
     * @param cirStartDate
     * @param signActConfig
     * @param signInfo
     * @
     * @return
     */
    public abstract Calendar getCirEndDate(Calendar cirStartDate, SignActivityConfigDto signActConfig, SignInfo signInfo);

    /**
     * 获取周期内的天数
     * @param signActConfig
     * @param signInfo
     * @return
     */
    public abstract Integer getCircleDays(SignActivityConfigDto signActConfig, SignInfo signInfo);

    /**
     * 周期内连续签到天数计算
     * @param signActConfig
     * @param signInfo
     * @return
     */
    public abstract Integer getContinueDayInCircle(SignActivityConfigDto signActConfig, SignInfo signInfo);

    public Integer getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }

    public static class SignInfo {
        private Calendar signDateCalendar;//签到日期,补签时请传入签到发起时的日期
        private Date signDate;
        private Integer cntDaysAfterSigned;//签到完成后的连续签到天数
        public SignInfo(Date signDate, int cntDaysAfterSigned){
            this.signDate = signDate;
            this.signDateCalendar = Calendar.getInstance();
            this.signDateCalendar.setTime(signDate);
            this.cntDaysAfterSigned = cntDaysAfterSigned;
        }

        public Calendar getSignDateCalendar() {
            return signDateCalendar;
        }

        public Date getSignDate() {
            return signDate;
        }

        public Integer getCntDaysAfterSigned() {
            return cntDaysAfterSigned;
        }
    }
}
