package cn.com.duiba.spring.boot.starter.autoconfigure.tensorflow.model;

import cn.com.duiba.nezha.alg.model.tf.LocalTFModel;
import cn.com.duiba.spring.boot.starter.autoconfigure.tensorflow.enums.AlgoTFModelStatusEnum;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.util.Objects;

@Slf4j
@Data
public class AlgoTFModel {

    private final static Long CLOSE_TF_MODEL_TIME = 120000L;

    // tf模型
    private LocalTFModel localTFModel;

    // 最后一次访问的时间
    private Long lastAccessTime;

    // tf模型被加载的时间
    private Long loadTime;

    // 模型的状态
    private int status;

    private String name;

    public AlgoTFModel(int status, String name){
        this.status = status;
        this.name = name;
    }

    public AlgoTFModel(LocalTFModel localTFModel, int status, String name){
        this.localTFModel = localTFModel;
        this.status = status;
        this.loadTime = System.currentTimeMillis();
        this.name = name;
    }

    public LocalTFModel getLocalTFModel(){
        this.lastAccessTime = System.currentTimeMillis();
        //CatTools.metricForCount("获取tf模型, name:" + name);
        return localTFModel;
    }

    /**
     * 判断当前tf模型是否为running状态
     * @return boolean
     */
    public boolean isRunning() {
        return this.status == AlgoTFModelStatusEnum.RUNNING.getCode();
    }

    /**
     * 判断当前tf模型是否可以关闭
     * @return
     */
    public boolean isAllowClose() {
        return lastAccessTime != null && System.currentTimeMillis() - lastAccessTime > CLOSE_TF_MODEL_TIME;
    }

    public void deleteTFModel() {
        doCloseTFModel();
    }

    public void doCloseTFModel() {
        try {
            localTFModel.close();
            log.info("关闭tf模型, name:{}", name);
        } catch (Exception e) {
            log.warn("AlgoTFModel doClose error", e);
            return;
        }
        localTFModel = null;
        status = AlgoTFModelStatusEnum.STOP.getCode();
        lastAccessTime = null;
        loadTime = null;
    }

    public boolean isLatestVersion(LocalTFModel localTFModel, String tfKey) {
        try {
            Long lastVersion = localTFModel.getLastVersion(tfKey);
            return !Objects.equals(String.valueOf(lastVersion), this.localTFModel.getVersion());
        } catch (Exception e) {
            log.warn("AlgoTFModel getLastVersion error", e);
        }
        return false;
    }

    public void loadTFModel(LocalTFModel localTFModel, String tfKey) {
        try {
            localTFModel.loadModel(tfKey);
            log.info("加载tf模型, name:{}", name);
        } catch (Exception e) {
            log.warn("AlgoTFModel loadModel error", e);
            return;
        }
        status = AlgoTFModelStatusEnum.RUNNING.getCode();
        loadTime = System.currentTimeMillis();
        this.localTFModel = localTFModel;
    }
}
