package cn.com.duiba.sso.api.web.power;

import cn.com.duiba.sso.api.common.tree.TreeView;
import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.domain.event.RoleUpdateEvent;
import cn.com.duiba.sso.api.service.eventbus.BizEventListener;
import cn.com.duiba.sso.api.domain.bizenum.CacheKeyEnum;
import cn.com.duiba.sso.api.domain.event.AdminOutLoginEvent;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemotePermissionService;
import cn.com.duiba.sso.api.service.power.PowerTreeService;
import cn.com.duiba.sso.api.tool.SystemInfo;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.BoundValueOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * 用户拥有的权限缓存
 */
@Service
@BizEventListener
public class AdminPowerCacheService {

    /**
     * 用户级别权限配置缓存
     */
    private LoadingCache<Long,PowerCache> adminPowerCache = CacheBuilder.newBuilder().expireAfterAccess(1,TimeUnit.DAYS).build(new AdminRoleCacheLoader());
    @Resource
    private StringRedisTemplate stringRedisTemplate;
    @Autowired
    private RemotePermissionService remotePermissionService;
    @Autowired
    private PowerTreeService powerTreeService;

    private ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();
    /**
     * 初始化权限树，启动刷新检测任务
     */
    @PostConstruct
    public void init(){
        executorService.scheduleWithFixedDelay(new AdminPowerFlushTask(),30,30,TimeUnit.SECONDS);
    }

    private PowerCache getPowerCache(Long adminId){
        try{
            return adminPowerCache.get(adminId);
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

    /**
     * 是否含有某项权限
     * @param adminId
     * @param url
     * @return
     */
    public Boolean hasPower(Long adminId,String url){
        Set<Long> pawerIds =  powerTreeService.getPowerIdsByUrl(url);
        if(pawerIds.isEmpty()){
            return true;
        }
        PowerCache cache = getPowerCache(adminId);
        return !Sets.intersection(pawerIds,cache.getPowerIdSet()).isEmpty();
    }

    /**
     * 获取用户在系统中所有的权限
     * @param adminId
     * @return
     */
    public  <T extends TreeView<T>> List<T> powerForAdmin(Long adminId, Function<PowerDto,T> transform){
        PowerCache cache = getPowerCache(adminId);
        return powerTreeService.getPowerTree(cache.getPowerIdSet(),transform);
    }

    /**
     * 获取用户资源白名单
     * @param adminId
     * @return
     */
    public Set<String> getAllPowerRes(Long adminId){
        PowerCache cache = getPowerCache(adminId);
        Set<Long> powerIds = cache.getPowerIdSet();
        Set<String> urlSet = Sets.newHashSet();

        for(Long id:powerIds){
            PowerDto power = powerTreeService.getPower(id);
            if (power==null){
                continue;
            }
            urlSet.addAll(power.getUrls());
        }
        return urlSet;
    }


    /**
     * 监听用户的退出登陆行为
     * @param event 用户退出登陆事件
     */
    @Subscribe
    public void AdminOutLoginEventListener(AdminOutLoginEvent event){
        flushAdminPower(event.getAdminId());
    }

    @Subscribe
    public void RoleUpdateEventListener(RoleUpdateEvent event){
        for(Long adminId:event.getAdminIds()){
            flushAdminPower(adminId);
        }
    }

    private void flushAdminPower(Long adminId){
        Long systemId = SystemInfo.getThisSystemId();
        String key = CacheKeyEnum.SSO_ADMIN_POWER_CONFIG_VERSION.getCacheKey(systemId,adminId);
        BoundValueOperations<String,String> operations = stringRedisTemplate.boundValueOps(key);
        operations.set(String.valueOf(new Date().getTime()),1,TimeUnit.DAYS);
    }


    private class AdminRoleCacheLoader extends CacheLoader<Long, PowerCache> {

        @Override
        public PowerCache load(Long adminId){
            Long systemId = SystemInfo.getThisSystemId();
            String key = CacheKeyEnum.SSO_ADMIN_POWER_CONFIG_VERSION.getCacheKey(systemId,adminId);
            BoundValueOperations<String,String> operations = stringRedisTemplate.boundValueOps(key);
            String versionStr = operations.get();

            Long version =  Optional.ofNullable(versionStr).map(Long::valueOf).orElse(new Date().getTime());

            Set<Long> powerIds =  remotePermissionService.getPowerIdsBySystemIdAndAdminId(systemId,adminId);
            PowerCache cache = new PowerCache(version);
            cache.setPowerIdSet(powerIds);


            if(StringUtils.isBlank(versionStr)){
                operations.set(version.toString(),1, TimeUnit.DAYS);
            }

            return cache;
        }
    }

    /**
     * 用户权限更新任务
     */
    private class AdminPowerFlushTask implements Runnable{

        @Override
        public void run() {

            Long systemId = SystemInfo.getThisSystemId();

            Set<Long> adminIds = adminPowerCache.asMap().keySet();
            for(Long adminId:adminIds){
                String key = CacheKeyEnum.SSO_ADMIN_POWER_CONFIG_VERSION.getCacheKey(systemId,adminId);
                BoundValueOperations<String,String> operations = stringRedisTemplate.boundValueOps(key);
                String versionStr = operations.get();
                if(StringUtils.isBlank(versionStr)){
                    return;
                }
                Long version = Long.valueOf(versionStr);
                PowerCache cache = adminPowerCache.getIfPresent(adminId);
                if(cache==null){
                    return;
                }
                if(cache.getVersion()>=version){//如果本地的版本大于远程版本，不执行
                    return;
                }
                adminPowerCache.invalidate(adminId);
            }
        }
    }

}
