package cn.com.duiba.sso.api.web.interceptor;

import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duiba.sso.api.exception.SsoException;
import cn.com.duiba.sso.api.web.interceptor.annotation.AuthCanAccess;
import cn.com.duiba.sso.api.web.interceptor.annotation.CanAccess;
import cn.com.duiba.sso.api.tool.JsonRender;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.CustomRequestInterceptor;
import cn.com.duibaboot.ext.autoconfigure.core.utils.HttpRequestUtils;
import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Set;

/**
 * Created by liuyao on 2017/5/10.
 */
public class SsoInterceptor extends HandlerInterceptorAdapter implements InitializingBean {

    @Resource
    private SsoFilterHandlerQueue queue;

    private Set<String> EXCLUDE_PATHS = Sets.newHashSet();

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {

        response.setCharacterEncoding("UTF-8");
        RequestTool.setRequestInThreadLocal(request,response);
        String url = request.getRequestURI();
        if (EXCLUDE_PATHS.contains(url) || isFeignRequest(request)){
            return true;
        }
        if(handler instanceof HandlerMethod){
            HandlerMethod method = (HandlerMethod)handler;
            CanAccess canAccess = AnnotationUtils.findAnnotation(method.getMethod(),CanAccess.class);
            if(!Objects.equal(null,canAccess)){
                return true;
            }
        }
        try {
            if (PerfTestContext.isCurrentInPerfTestMode()) {
                PerfTestContext._setPerfTestMode(false);
            }
            //进入拦截队列
            if(!queue.doHandler(handler)){
                return false;
            }
        } catch (SsoException e) {
            if (RequestTool.isAsynchronousRequests()){
                response.setHeader("Content-Type", "application/json;charset=UTF-8");
                response.getWriter().write(JsonRender.failResult(e).toJSONString());
            }else{
                response.setHeader("Content-Type", "text/html;charset=UTF-8");
                response.getWriter().write(e.getMessage());
            }
            return false;
        }
        return true;
    }


    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        RequestTool.removeRequestInThreadLocal();
    }



    /**
     * 判断是否是来自内网的Feign请求（spring-cloud调用）
     * @param request
     * @return
     */
    private boolean isFeignRequest(HttpServletRequest request){
        if (!"true".equals(request.getHeader(CustomRequestInterceptor.X_RPC))) {
            return false;
        }
        //是feign调用的情况下还得验证请求来自内网，否则会有安全问题
        if(!HttpRequestUtils.isLanRequest(request)){
            return false;
        }
        return true;
    }

    /**
     * 添加白名单
     * @param path
     */
    public void addExcludePath(String ...path){
        EXCLUDE_PATHS.addAll(Sets.newHashSet(path));
    }

    @Override
    public void afterPropertiesSet() throws Exception {

        EXCLUDE_PATHS.add("/favicon.ico");
        EXCLUDE_PATHS.add("/monitor/check");
        EXCLUDE_PATHS.add("/swagger-ui.html");
        EXCLUDE_PATHS.add("/swagger-resources");
        EXCLUDE_PATHS.add("/swagger-resources/configuration/ui");
        EXCLUDE_PATHS.add("/v2/api-docs");
    }
}
