package cn.com.duiba.sso.api.tool;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.domain.dto.PermissionDto;
import cn.com.duiba.sso.api.exception.SsoException;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.sso.api.remoteservice.RemoteManagerLogService;
import cn.com.duiba.sso.api.remoteservice.RemotePermissionService;
import cn.com.duiba.sso.api.remoteservice.RemoteSSOService;
import cn.com.duiba.sso.api.service.logger.domain.SsoLoggerParams;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.google.common.base.Objects;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Created by liuyao on 2017/5/11.
 */
public class RequestTool implements ApplicationContextAware {

    private static ThreadLocal<RequestTool> local = new ThreadLocal<>();

    private static RemoteAdminService remoteAdminService;
    private static RemotePermissionService remotePermissionService;
    private static RemoteSSOService remoteSSOService;
    private static RemoteManagerLogService remoteManagerLogService;

    private HttpServletRequest request;
    private HttpServletResponse response;
    private RequestParams params;

    public static void setRequestInThreadLocal(HttpServletRequest request, HttpServletResponse response){
        if(request==null){
            throw new SsoRunTimeException("RequestTool注入request为空");
        }
        if(response==null){
            throw new SsoRunTimeException("RequestTool注入response为空");
        }
        RequestTool tool = new RequestTool();
        tool.request = request;
        tool.response = response;
        tool.params = RequestParams.parse(request);
        local.set(tool);
    }

    private static RequestTool get(){
        if(local.get() == null){
            throw new SsoRunTimeException("请在登录拦截器中调用setRequestInThreadLocal,或者请检查是否在异步线程中使用RequestTool,如果有异步场景要获取用户信息,请在异步线程启动之前使用getRequestParams获取当前用户的请求封装信息");
        }
        return local.get();
    }

    /**
     * RequestTool中的变量只能在当前线程中使用,当启用异步线程的时候,使用此方法复制一个新的封装对象,防止空指针
     * @return 复制品
     */
    public static RequestParams getRequestParams(){
        if(local.get() == null){
            return null;
        }
        return get().params.clone();
    }

    public static HttpServletResponse getResponse(){
        return get().response;
    }

    public static HttpServletRequest getRequest(){
        return get().request;
    }

    public static void removeRequestInThreadLocal(){
        local.remove();
    }

    /**
     * 获取管理员Id
     * @return
     */
    public static Long getAdminId(){
        AdminDto admin = getAdmin();
        return Optional.ofNullable(admin).map(AdminDto::getId).orElse(null);
    }

    public static AdminDto getAdmin() {
        RequestTool tool = RequestTool.get();
        return tool.params.getAdmin();
    }

    public static void setAdmin(AdminDto admin) {
        RequestTool tool = RequestTool.get();
        tool.params.setAdmin(admin);
    }

    public static void addCookie(Cookie cookie){
        if(cookie==null){
            return;
        }
        RequestTool tool = RequestTool.get();
        tool.params.addCookie(cookie.getName(),cookie.getValue());
        tool.response.addCookie(cookie);
    }

    public static String getUrl(){
        return get().params.getUrl();
    }

    public static boolean isHttpsRequest(){
        HttpServletRequest request= get().request;
        return "true".equals(request.getHeader("Use-Https"));
    }

    public static String getCookie(String name){
        RequestTool tool = RequestTool.get();
        return tool.params.getCookie(name);
    }

    /**
     * 提取request中的参数
     * @return
     */
    public static Map<String,String> getRequestParamMap(){
        HttpServletRequest request= get().request;
        Map<String, String[]> map=request.getParameterMap();
        Map<String, String> resultMap=new HashMap<>();
        for(Map.Entry<String, String[]> entry :map.entrySet()){
            resultMap.put(entry.getKey(), entry.getValue()[0]);
        }
        return resultMap;
    }

    public static Boolean isAsynchronousRequests(){
        HttpServletRequest request= get().request;
        String re = request.getHeader("x-requested-with");
        String accept = request.getHeader("Accept");
        return (StringUtils.isNotBlank(re) || StringUtils.isNotBlank(accept) && accept.contains("json"));
    }

    public static String getIp(){
        RequestTool tool = RequestTool.get();
        return tool.params.getIp();
    }

    public static Boolean isLocalHost(){
        RequestTool tool = RequestTool.get();
        return Objects.equal(tool.params.getIp(),RequestParams.LOCALHOST_IP);
    }

    public static void googleVerify(String code) throws SsoException{
        Long adminId = RequestTool.getAdminId();
        DubboResult<Boolean> result = RequestTool.remoteAdminService.googleVerify(adminId,code);
        if(!result.isSuccess()){
            throw new SsoException(result.getMsg());
        }
    }

    public static String getHomeURL(){
        String scheme = RequestTool.getRequest().getScheme();
        String serverName = RequestTool.getRequest().getServerName();
        int port = RequestTool.getRequest().getServerPort();
        return scheme + "://" + serverName + ":" + port;
    }

    @Deprecated
    public static List<PermissionDto> getPermissions() throws RuntimeException{
        Integer adminId = getAdminId().intValue();
        Integer systemId = SystemInfo.getThisSystemId().intValue();
        DubboResult<List<PermissionDto>> result = RequestTool.remotePermissionService.getPermissions(adminId.intValue(),systemId.intValue());
        List<PermissionDto> permissionDtoList = result.getResult();
        return permissionDtoList;
    }

    /**
     * 获取令牌
     * @return
     */
    public static String findTicket(){
        /*******************************所有系统升级1.3.0后删除***********************/
        String ticketForUrl = getRequestParamMap().get(CookieUtil.LOGIN_COOKIE_NAME);
        if(StringUtils.isNotBlank(ticketForUrl)){
            return ticketForUrl;
        }
        /*******************************所有系统升级1.3.0后删除***********************/
        return CookieUtil.getCookie(CookieUtil.LOGIN_COOKIE_NAME);
    }

    /**
     * 钉钉平台
     * @return
     */
    public static String findMobileTicket(){
        return CookieUtil.getCookie(CookieUtil.MOBILE_USER_ID_NAME);
    }

    public static void outLogin(){
        String ticket = RequestTool.findTicket();
        if(StringUtils.isBlank(ticket)){
            return;
        }
        CookieUtil.deleteCookie(CookieUtil.LOGIN_COOKIE_NAME);
        remoteSSOService.outLogin(ticket);
    }

    @Deprecated
    public static void sendOperationLog(SsoLoggerParams ssoLoggerParams){
        remoteManagerLogService.log(ssoLoggerParams);
    }

    @Override
    public void setApplicationContext(ApplicationContext context) throws BeansException {
        RequestTool.remoteAdminService = context.getBean(RemoteAdminService.class);
        RequestTool.remotePermissionService = context.getBean(RemotePermissionService.class);
        RequestTool.remoteSSOService = context.getBean(RemoteSSOService.class);
        RequestTool.remoteManagerLogService = context.getBean(RemoteManagerLogService.class);
    }

}
