package cn.com.duiba.sso.api.web.power;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.sso.api.common.tree.TreeView;
import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.domain.event.SsoSystemInitSuccessEvent;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.sso.api.remoteservice.RemotePermissionService;
import cn.com.duiba.sso.api.service.eventbus.BizEventListener;
import cn.com.duiba.sso.api.service.power.PowerTreeService;
import cn.com.duiba.sso.api.tool.SystemInfo;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import com.google.common.eventbus.Subscribe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;

/**
 * 用户拥有的权限缓存
 */
@Service
@BizEventListener
public class AdminPowerCacheService {

    private static Logger logger = LoggerFactory.getLogger(AdminPowerCacheService.class);
    /**
     * 用户级别权限配置缓存
     */
    private LoadingCache<Long,PowerCache> adminPowerCache = CacheBuilder.newBuilder().expireAfterWrite(30,TimeUnit.MINUTES).build(new AdminRoleCacheLoader());
    @Autowired
    private RemotePermissionService remotePermissionService;
    @Autowired
    private PowerTreeService powerTreeService;
    @Autowired
    private RemoteAdminService remoteAdminService;

    private ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();

    @Subscribe
    public void SystemInfoInitComplateEventListener(SsoSystemInitSuccessEvent event){
        executorService.scheduleWithFixedDelay(new AdminPowerFlushTask(),30,30,TimeUnit.SECONDS);
    }

    private PowerCache getPowerCache(Long adminId){
        try{
            return adminPowerCache.get(adminId);
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

    /**
     * 是否含有某项权限
     * @param adminId
     * @param powerPath
     * @return
     */
    public Boolean hasPower(Long adminId,String powerPath){
        Set<Long> pawerIds =  powerTreeService.getPowerIdsByUrl(powerPath);
        if(pawerIds.isEmpty()){
            return true;
        }
        PowerCache cache = getPowerCache(adminId);
        return !Sets.intersection(pawerIds,cache.getPowerIdSet()).isEmpty();
    }

    /**
     * 获取用户在系统中所有的权限
     * @param adminId
     * @return
     */
    public <T extends TreeView<T>> List<T> powerForAdmin(Long adminId, Function<PowerDto,T> transform){
        PowerCache cache = getPowerCache(adminId);
        return powerTreeService.getPowerTree(cache.getPowerIdSet(),transform);
    }

    /**
     * 获取用户资源白名单
     * @param adminId
     * @return
     */
    public Set<String> getAllPowerRes(Long adminId){
        PowerCache cache = getPowerCache(adminId);
        Set<Long> powerIds = cache.getPowerIdSet();
        Set<String> urlSet = Sets.newHashSet();

        for(Long id:powerIds){
            PowerDto power = powerTreeService.getPower(id);
            if (power==null){
                continue;
            }
            urlSet.addAll(power.getUrls());
        }
        return urlSet;
    }

    /**
     * 获取当前用户的权限版本号
     * @param adminId
     * @return
     */
    public Long version(Long adminId){
        PowerCache cache = getPowerCache(adminId);
        return cache.getVersion();
    }

    /**
     * 清除缓存
     * @param adminId
     */
    public void clean(Long adminId){
        adminPowerCache.invalidate(adminId);
    }


    private class AdminRoleCacheLoader extends CacheLoader<Long, PowerCache> {

        @Override
        public PowerCache load(Long adminId){
            Long systemId = SystemInfo.getThisSystemId();
            Set<Long> powerIds =  remotePermissionService.getPowerIdsBySystemIdAndAdminId(systemId,adminId);
            PowerCache cache = new PowerCache(remotePermissionService.getAdminPowerVersion(systemId,adminId));
            cache.setPowerIdSet(powerIds);
            return cache;
        }
    }

    /**
     * 用户权限更新任务
     */
    private class AdminPowerFlushTask implements Runnable{

        @Override
        public void run() {
            try {
                Long systemId = SystemInfo.getThisSystemId();
                Set<Long> adminIds = adminPowerCache.asMap().keySet();
                for(Long adminId:adminIds){
                    PowerCache cache = adminPowerCache.getIfPresent(adminId);
                    if(cache==null){
                        return;
                    }
                    Long version = remotePermissionService.getAdminPowerVersion(systemId,adminId);
                    if(cache.getVersion()>=version){//如果本地的版本大于远程版本，不执行
                        return;
                    }
                    adminPowerCache.invalidate(adminId);
                }
            }catch (Exception e){
                logger.error("管理员权限检测失败",e);
            }
        }
    }

}
