package cn.com.duiba.sso.api.web.power;

import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemotePermissionService;
import cn.com.duiba.sso.api.service.power.PowerTreeCache;
import cn.com.duiba.sso.api.service.power.PowerTreeService;
import cn.com.duiba.sso.api.tool.SystemInfo;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * 用户拥有的权限缓存
 */
@Service
public class AdminPowerCacheService {

    private static Logger logger = LoggerFactory.getLogger(AdminPowerCacheService.class);
    /**
     * 用户级别权限配置缓存
     */
    private LoadingCache<Long,PowerCache> adminPowerCache = CacheBuilder.newBuilder().expireAfterWrite(30,TimeUnit.MINUTES).build(new AdminRoleCacheLoader());
    @Autowired
    private RemotePermissionService remotePermissionService;
    @Autowired
    private PowerTreeService powerTreeService;

    private PowerCache getPowerCache(Long adminId){
        try{
            return adminPowerCache.get(adminId);
        }catch (Exception e){
            throw new SsoRunTimeException(e);
        }
    }

    /**
     * 是否含有某项权限
     * @param adminId
     * @param powerPath
     * @return
     */
    public Boolean hasPower(Long adminId,String powerPath){
        PowerTreeCache powerTreeCache = powerTreeService.getPowerTreeCache(SystemInfo.getThisSystemId());
        Set<Long> pawerIds =  powerTreeCache.getPowerIdsByUrl(powerPath);
        if(pawerIds.isEmpty()){
            return true;
        }
        PowerCache cache = getPowerCache(adminId);
        return !Sets.intersection(pawerIds,cache.getPowerIdSet()).isEmpty();
    }

    /**
     * 获取员工拥有某项权限的Ids
     * @param adminId
     * @param powerPath
     * @return
     */
    public Set<Long> hasPowerIds(Long adminId,String powerPath){
        PowerTreeCache powerTreeCache = powerTreeService.getPowerTreeCache(SystemInfo.getThisSystemId());
        Set<Long> pawerIds =  powerTreeCache.getPowerIdsByUrl(powerPath);
        if(pawerIds.isEmpty()){
            return Collections.emptySet();
        }
        PowerCache cache = getPowerCache(adminId);
        return Sets.intersection(pawerIds,cache.getPowerIdSet());
    }

    /**
     * 获取用户资源白名单
     * @param adminId
     * @return
     */
    public Set<String> getAllPowerRes(Long adminId){
        PowerCache cache = getPowerCache(adminId);
        Set<Long> powerIds = cache.getPowerIdSet();
        Set<String> urlSet = Sets.newHashSet();

        PowerTreeCache powerTreeCache = powerTreeService.getPowerTreeCache(SystemInfo.getThisSystemId());

        for(Long id:powerIds){
            PowerDto power = powerTreeCache.getPower(id);
            if (power==null){
                continue;
            }
            urlSet.addAll(power.getUrls());
        }
        return urlSet;
    }

    /**
     * 获取当前用户的权限版本号
     * @param adminId
     * @return
     */
    public Long version(Long adminId){
        PowerCache cache = getPowerCache(adminId);
        return cache.getVersion();
    }

    /**
     * 清除缓存
     * @param adminId
     */
    public void clean(Long adminId){
        adminPowerCache.invalidate(adminId);
    }


    private class AdminRoleCacheLoader extends CacheLoader<Long, PowerCache> {

        @Override
        public PowerCache load(Long adminId){
            Long systemId = SystemInfo.getThisSystemId();
            Set<Long> powerIds =  remotePermissionService.getPowerIdsBySystemIdAndAdminId(systemId,adminId);
            PowerCache cache = new PowerCache(remotePermissionService.getAdminPowerVersion(systemId,adminId));
            cache.setPowerIdSet(powerIds);
            return cache;
        }
    }

    public void adminPowerFlushTask() {
        try {
            Long systemId = SystemInfo.getThisSystemId();
            Set<Long> adminIds = adminPowerCache.asMap().keySet();
            for(Long adminId:adminIds){
                PowerCache cache = adminPowerCache.getIfPresent(adminId);
                if(cache==null){
                    return;
                }
                Long version = remotePermissionService.getAdminPowerVersion(systemId,adminId);
                if(cache.getVersion()>=version){//如果本地的版本大于远程版本，不执行
                    return;
                }
                adminPowerCache.invalidate(adminId);
            }
        }catch (Exception e){
            logger.error("管理员权限检测失败",e);
        }
    }

}
