package cn.com.duiba.sso.api;

import cn.com.duiba.sso.api.constants.SsoProperties;
import cn.com.duiba.sso.api.service.logger.service.SsoLoggerMateInfoAdapter;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.sso.api.web.interceptor.MobileInterceptor;
import cn.com.duiba.sso.api.web.interceptor.SsoFilterHandlerQueue;
import cn.com.duiba.sso.api.web.interceptor.SsoInterceptor;
import cn.com.duiba.sso.api.web.interceptor.annotation.MobileHandler;
import cn.com.duiba.sso.api.web.interceptor.annotation.OnlyMobileHandler;
import cn.com.duiba.sso.api.web.interceptor.handler.SsoFilterHandler;
import cn.com.duiba.sso.api.web.interceptor.handler.impl.AuthorityFilterHandler;
import cn.com.duiba.sso.api.web.interceptor.handler.impl.LoginFilterHandler;
import cn.com.duiba.sso.api.web.interceptor.handler.impl.MobileLoginFilterHandler;
import cn.com.duiba.sso.api.web.interceptor.handler.impl.SystemFilterHandler;
import cn.com.duiba.sso.api.web.logger.DefaultSsoLoggerMateInfoAdapter;
import cn.com.duiba.sso.api.web.power.AdminPowerCacheService;
import com.google.common.base.Objects;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.*;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.core.annotation.Order;
import org.springframework.web.servlet.config.annotation.InterceptorRegistration;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;

import javax.annotation.Resource;
import java.util.List;

@Configuration
@ComponentScan("cn.com.duiba.sso.api.web")
@ConditionalOnWebApplication
@AutoConfigureAfter(SsoBasicConfiguration.class)
@EnableConfigurationProperties(SsoProperties.class)
@Import({SsoBasicConfiguration.class})//引入SSO基础配置
@ImportResource("classpath:sso-api-task.xml")
@Order(-1)
public class SsoAutoConfiguration extends WebMvcConfigurerAdapter implements ApplicationContextAware {

    @Resource
    private SsoProperties ssoProperties;

    private ApplicationContext applicationContext;

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        String pathPatterns = ssoProperties.getPathPatterns();
        String excludePathPatterns = ssoProperties.getExcludePathPatterns();

        SsoInterceptor ssoInterceptor = applicationContext.getBean(SsoInterceptor.class);

        String[] paths = StringUtils.split(pathPatterns,",");
        String[] excludePath = StringUtils.isEmpty(excludePathPatterns) ? new String[]{} : StringUtils.split(excludePathPatterns,",");
        InterceptorRegistration registration = registry.addInterceptor(ssoInterceptor)
                .addPathPatterns(paths)
                .addPathPatterns("/auth/**")
                .addPathPatterns("/sso/**")//sso内置开放接口
                .excludePathPatterns(excludePath);

        //开启移动端功能
        if(!ssoProperties.getMobileEnable()){
            return;
        }

        String mobilePathPatterns = ssoProperties.getMobilePathPatterns();
        String[] mobilePaths = StringUtils.split(mobilePathPatterns,",");
        //PC端将移动端的链接全部排除
        registration.excludePathPatterns(mobilePaths);

        MobileInterceptor mobileInterceptor = applicationContext.getBean(MobileInterceptor.class);
        InterceptorRegistration mobileRegistration = registry.addInterceptor(mobileInterceptor).addPathPatterns(mobilePaths).excludePathPatterns(excludePath);
        //移动端将PC端
        if(!StringUtils.equals(pathPatterns,"/**")){
            mobileRegistration.excludePathPatterns(paths);
        }
    }

    /**
     * 初始化RequestTool
     * @return
     */
    @Bean
    public RequestTool initRequestTool(){
        return new RequestTool();
    }

    @Bean
    public SsoInterceptor ssoInterceptor(List<SsoFilterHandler> ssoFilterHandlers){
        SsoFilterHandlerQueue queue = new SsoFilterHandlerQueue();

        SsoInterceptor ssoInterceptor = new SsoInterceptor();

        for (SsoFilterHandler handler:ssoFilterHandlers) {

            OnlyMobileHandler onlyMobileHandler = AnnotationUtils.findAnnotation(handler.getClass(),OnlyMobileHandler.class);

            if(!Objects.equal(null,onlyMobileHandler)){
                continue;
            }
            queue.addHandler(handler);
        }
        ssoInterceptor.setHandleQueue(queue);
        return ssoInterceptor;
    }

    @Bean
    @ConditionalOnProperty(name="duiba.sso.mobile-enable", havingValue="true", matchIfMissing = false)
    public MobileInterceptor mobileInterceptor(List<SsoFilterHandler> ssoFilterHandlers){

        SsoFilterHandlerQueue queue = new SsoFilterHandlerQueue();
        MobileInterceptor mobileInterceptor = new MobileInterceptor();

        for (SsoFilterHandler handler:ssoFilterHandlers) {

            MobileHandler mobileHandler = AnnotationUtils.findAnnotation(handler.getClass(),MobileHandler.class);
            OnlyMobileHandler onlyMobileHandler = AnnotationUtils.findAnnotation(handler.getClass(),OnlyMobileHandler.class);

            if(!Objects.equal(null,mobileHandler) || !Objects.equal(null,onlyMobileHandler)){
                queue.addHandler(handler);
            }
        }

        mobileInterceptor.setHandleQueue(queue);
        return mobileInterceptor;
    }

    @Bean
    public AdminPowerCacheService adminPowerCacheService(){
        return new AdminPowerCacheService();
    }

    @Bean
    @ConditionalOnMissingBean
    public SsoLoggerMateInfoAdapter ssoLoggerMateInfoAdapter(){
        return new DefaultSsoLoggerMateInfoAdapter();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Bean
    public SsoFilterHandler loginFilterHandler(){
        return new LoginFilterHandler();
    }

    @Bean
    @ConditionalOnProperty(name="duiba.sso.mobile-enable", havingValue="true", matchIfMissing = false)
    public SsoFilterHandler mobileLoginFilterHandler(){
        return new MobileLoginFilterHandler();
    }

    @Bean
    public SsoFilterHandler systemFilterHandler(){
        return new SystemFilterHandler();
    }

    @Bean
    public SsoFilterHandler authorityFilterHandler(){
        return new AuthorityFilterHandler();
    }



}
