package cn.com.duiba.sso.api.tool;

import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.exception.SsoException;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.sso.api.remoteservice.RemoteSSOService;
import com.alibaba.ttl.TransmittableThreadLocal;
import com.google.common.base.Objects;
import com.google.common.base.Splitter;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Created by liuyao on 2017/5/11.
 */
public class RequestTool {

    private static ThreadLocal<RequestTool> local = new ThreadLocal<>();
    private static TransmittableThreadLocal<RequestParams> TTL = new TransmittableThreadLocal<>();

    private static List<String> IP_NAME_SET = Lists.newArrayList("X-Real-IP", "Proxy-Client-IP", "WL-Proxy-Client-IP", "Cdn-Src-Ip");
    private static Splitter IP_SPLITTER = Splitter.on(",").trimResults().omitEmptyStrings();

    private static RemoteAdminService remoteAdminService;
    private static RemoteSSOService remoteSSOService;

    private static String LOCALHOST_IP = "127.0.0.1";//NOSONAR

    private HttpServletRequest request;
    private HttpServletResponse response;

    public static void setRequestInThreadLocal(HttpServletRequest request, HttpServletResponse response){
        if(request==null){
            throw new SsoRunTimeException("RequestTool注入request为空");
        }
        if(response==null){
            throw new SsoRunTimeException("RequestTool注入response为空");
        }
        RequestTool tool = new RequestTool();
        tool.request = request;
        tool.response = response;
        TTL.set(createRequestParams(request));
        local.set(tool);
    }

    private static RequestTool get(){
        if(local.get() == null){
            throw new SsoRunTimeException("请在登录拦截器中调用setRequestInThreadLocal,或者请检查是否在异步线程中使用RequestTool,如果有异步场景要获取用户信息,请在异步线程启动之前使用getRequestParams获取当前用户的请求封装信息");
        }
        return local.get();
    }

    public static RequestParams getRequestParams(){
        RequestParams params = TTL.get();
        if(params==null){
            throw new SsoRunTimeException("非用户请求线程，无法获取用户请求数据");
        }
        return params;
    }

    public static HttpServletResponse getResponse(){
        return get().response;
    }

    public static HttpServletRequest getRequest(){
        return get().request;
    }

    public static void removeRequestInThreadLocal(){
        local.remove();
        TTL.remove();
    }

    /**
     * 获取管理员Id
     * @return adminId
     */
    public static Long getAdminId(){
        AdminDto admin = getAdmin();
        return Optional.ofNullable(admin).map(AdminDto::getId).orElse(null);
    }

    public static AdminDto getAdmin() {
        RequestParams param = getRequestParams();
        return param.getAdmin();
    }

    public static void setAdmin(AdminDto admin) {
        RequestParams param = getRequestParams();
        param.setAdmin(admin);
    }

    public static void addCookie(Cookie cookie){
        if(cookie==null){
            return;
        }
        RequestTool tool = RequestTool.get();
        if(tool==null){
            throw new SsoRunTimeException("用户线程已经结束");
        }
        RequestParams params = getRequestParams();
        params.addCookie(cookie.getName(),cookie.getValue());
        tool.response.addCookie(cookie);
    }

    public static String getUrl(){
        RequestParams params = getRequestParams();
        return params.getUrl();
    }

    public static boolean isHttpsRequest(){
        HttpServletRequest request= get().request;
        return Boolean.parseBoolean(request.getHeader("Use-Https"));
    }

    public static String getCookie(String name){
        RequestParams params = getRequestParams();
        return params.getCookie(name);
    }

    /**
     * 提取request中的参数
     * @return 参数列表
     */
    public static Map<String,String> getRequestParamMap(){
        HttpServletRequest request= get().request;
        Map<String, String[]> map=request.getParameterMap();
        Map<String, String> resultMap=new HashMap<>();
        for(Map.Entry<String, String[]> entry :map.entrySet()){
            String[] values = entry.getValue();
            if(values.length!=0){
                resultMap.put(entry.getKey(), entry.getValue()[0]);
            }
        }
        return resultMap;
    }

    public static Boolean isAsynchronousRequests(){
        HttpServletRequest request= get().request;
        String re = request.getHeader("x-requested-with");
        String accept = request.getHeader("Accept");
        return (StringUtils.isNotBlank(re) && StringUtils.equals("XMLHttpRequest",re)) || (StringUtils.isNotBlank(accept) && accept.contains("json"));
    }

    public static String getIp(){
        RequestParams params = getRequestParams();
        return params.getIp();
    }

    public static Boolean isLocalHost(){
        RequestParams params = getRequestParams();
        return Objects.equal(params.getIp(),LOCALHOST_IP);
    }

    public static void googleVerify(String code) throws SsoException{
        Long adminId = getAdminId();
        Boolean result = remoteAdminService.googleCodeVerify(adminId,code);
        if(!result){
            throw new SsoException("动态码验证失败");
        }
    }

    public static String getHomeURL(){
        String scheme = RequestTool.getRequest().getScheme();
        String serverName = RequestTool.getRequest().getServerName();
        int port = RequestTool.getRequest().getServerPort();
        return scheme + "://" + serverName + ":" + port;
    }

    /**
     * 获取令牌
     * @return SSO通行证
     */
    public static String findTicket(){
        String ticketForUrl = getRequestParamMap().get(CookieUtil.LOGIN_COOKIE_NAME);
        if(StringUtils.isNotBlank(ticketForUrl)){
            return ticketForUrl;
        }
        return CookieUtil.getCookie(CookieUtil.LOGIN_COOKIE_NAME);
    }

    /**
     * 获取移动端通行证
     */
    public static String findMobileTicket(){
        return CookieUtil.getCookie(CookieUtil.MOBILE_USER_ID_NAME);
    }

    public static void outLogin(){
        String ticket = RequestTool.findTicket();
        if(StringUtils.isBlank(ticket)){
            return;
        }
        CookieUtil.deleteCookie(CookieUtil.LOGIN_COOKIE_NAME);
        remoteSSOService.outLogin(ticket);
    }

    private static RequestParams createRequestParams(HttpServletRequest request){
        if(request==null) throw new SsoRunTimeException("RequestParams解析时的request不能为空");
        RequestParams rp = new RequestParams();

        Map<String,String> cookieMap = parseCookies(request);
        rp.setUrl(parseUrl(request));
        rp.setCookies(cookieMap);
        rp.setIp(parseIp(request));
        rp.setUri(request.getRequestURI());
        return rp;
    }

    @Autowired
    public void setRemoteAdminService(RemoteAdminService remoteAdminService) {
        RequestTool.remoteAdminService = remoteAdminService;//NOSONAR
    }

    @Autowired
    public void setRemoteSSOService(RemoteSSOService remoteSSOService) {
        RequestTool.remoteSSOService = remoteSSOService;//NOSONAR
    }

    private static String parseUrl(HttpServletRequest request){
        String url = request.getRequestURL().toString();
        String queryString = request.getQueryString();
        StringBuilder sb = new StringBuilder();
        sb.append(url);
        if(org.apache.commons.lang3.StringUtils.isNotBlank(queryString)) sb.append("?").append(queryString);
        return sb.toString();
    }

    private static Map<String,String> parseCookies(HttpServletRequest request){
        Cookie[] cookies=request.getCookies();
        Map<String,String> cookieMap = Maps.newHashMap();
        if(cookies!=null){
            for(Cookie cookie : cookies){
                cookieMap.put(cookie.getName(),cookie.getValue());
            }
        }
        return cookieMap;
    }

    public static String parseIp(HttpServletRequest request){
        List<String> ipList = IP_SPLITTER.splitToList(Optional.ofNullable(request.getHeader("x-forwarded-for")).orElse(""));
        if(!ipList.isEmpty()){
            return ipList.get(0);
        }
        for(String name:IP_NAME_SET){
            String ip = request.getHeader(name);
            if("unknown".equalsIgnoreCase(ip)){
                ip = null;
            }
            if(StringUtils.isNotBlank(ip)){
                return ip;
            }
        }
        return Optional.ofNullable(request.getRemoteAddr()).orElse(LOCALHOST_IP);
    }


}
