package cn.com.duiba.sso.api.web.export;

import cn.com.duiba.sso.api.config.SsoExecutorServiceInitializer;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import com.google.common.base.Joiner;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.File;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;

public class SsoExportFileService{

    private static final String EXPORT_FILE_TESK_NAME = "EXPORT_FILE_TESK_NAME";

    private static final Joiner KEY_JOINER = Joiner.on("_").skipNulls();

    @Resource
    private ApplicationContext applicationContext;

    private static final String FILE_PATH = "tempFiles/";

    @Autowired(required = false)
    private StringRedisTemplate stringRedisTemplate;

    protected StringRedisTemplate getStringRedisTemplate() {
        return stringRedisTemplate;
    }

    @Resource(name= SsoExecutorServiceInitializer.SSO_THREAD_POOL_NAME)
    private ExecutorService executorService;

    public ExportFileInfo getExportFileInfo(Long taskId){
        String key = getTaskKey(taskId);
        ExportFileInfo info = new ExportFileInfo();
        if(Objects.isNull(stringRedisTemplate)){
            throw new SsoRunTimeException("要使用Sso下载组件必须要在IOC中提供StringRedisTemplate");
        }
        if(!stringRedisTemplate.hasKey(key)){
            return info;
        }
        BoundHashOperations<String,String,String> operations = stringRedisTemplate.boundHashOps(key);
        Map<String,String> map = operations.entries();
        info.setTaskId(taskId);
        info.setMessage(map.get("message"));
        info.setCount(Integer.valueOf(map.get("count")));
        info.setSuccessCount(Integer.valueOf(map.get("successCount")));
        info.setFileName(map.get("fileName"));
        info.setSuccess(Boolean.valueOf(map.get("success")));
        info.setUrl(map.get("url"));
        return info;
    }

    public <T extends BaseExportRunnable> T createExportTask(Class<T> teskType){
        if(stringRedisTemplate==null){
            throw new SsoRunTimeException("需要往SsoExportFileService中setStringRedisTemplate才能使用此功能");
        }
        return applicationContext.getBean(teskType);
    }

    public void runTesk(BaseExportRunnable runnable){
        executorService.submit(runnable);
    }


    protected String getTaskKey(Long taskId){
        return KEY_JOINER.join(EXPORT_FILE_TESK_NAME,taskId);
    }


    public File getTempDir(){
        String webRootDir = System.getProperty("user.home");
        File userDir = new File(webRootDir,FILE_PATH);
        if (!userDir.exists()) {
            userDir.mkdirs();
        }
        return userDir;
    }



}
