package cn.com.duiba.sso.api.web.interceptor.handler.impl;

import cn.com.duiba.sso.api.config.SsoExecutorServiceInitializer;
import cn.com.duiba.sso.api.domain.params.AccessLogMultiParams;
import cn.com.duiba.sso.api.domain.params.AccessLogParams;
import cn.com.duiba.sso.api.remoteservice.RemoteManagerLogService;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.sso.api.tool.SystemInfo;
import cn.com.duiba.sso.api.web.interceptor.handler.SsoFilterHandler;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Scheduled;

import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

/**
 * sso访问日志
 */
@Slf4j
public class AccessLogFilterHandler implements SsoFilterHandler {

    private static final int LOG_LIMIT = 20;

    @Resource
    private RemoteManagerLogService remoteManagerLogService;

    @Resource(name= SsoExecutorServiceInitializer.SSO_THREAD_POOL_NAME)
    private ExecutorService executorService;

    //构造发送窗口
    private AtomicReference<Set<AccessLogParams>> reference = new AtomicReference<>(Sets.newConcurrentHashSet());

    @Override
    public Boolean before(Object handler) {
        try{
            AccessLogParams accessLog = new AccessLogParams();
            accessLog.setAppId(SystemInfo.getThisSystemId());
            accessLog.setAdminId(RequestTool.getAdminId());

            HttpServletRequest request = RequestTool.getRequest();
            accessLog.setRequestUri(request.getRequestURI());
            accessLog.setRequestParams(JSONObject.toJSONString(request.getParameterMap()));

            accessLog.setDeviceNo(RequestTool.findDeviceNo());
            accessLog.setIp(RequestTool.getIp());
            accessLog.setUuid(RequestTool.getRequestUUID());

            String userAgent = Optional.ofNullable(request.getHeader("User-Agent")).orElse("未知");
            accessLog.setUserAgent(userAgent.replaceAll(";",""));

            reference.getAndUpdate(accessLogParams -> {
                accessLogParams.add(accessLog);
                return accessLogParams;
            });
            if(reference.get().size()>=LOG_LIMIT){
                flush();
            }
        }catch (Exception e){
            log.error("采集访问日志失败",e);
        }
        return true;
    }

    @PreDestroy
    @Scheduled(cron="3/3 * * * * ?")
    public void scheduleFlush(){
        flush();
    }

    private synchronized void flush(){
        Set<AccessLogParams> logs = reference.getAndSet(Sets.newConcurrentHashSet());
        if(logs.isEmpty()){
            return;
        }
        AccessLogSenderRunnable runnable = new AccessLogSenderRunnable();
        runnable.setLogs(Lists.newArrayList(logs));
        executorService.submit(runnable);

    }

    private class AccessLogSenderRunnable implements Runnable{

        private List<AccessLogParams> logs;

        @Override
        public void run() {
            if(logs.isEmpty()){
                return;
            }
            try{
                AtomicInteger cursor = new AtomicInteger(0);
                BiMap<String,Integer> userAgentIndexMap = HashBiMap.create();

                for(AccessLogParams item:logs){
                    if(userAgentIndexMap.containsKey(item.getUserAgent())){
                        Integer index = userAgentIndexMap.get(item.getUserAgent());
                        item.setUserAgent(null);
                        item.setUserAgentIndex(index);
                    }else {
                        Integer index = cursor.getAndIncrement();
                        userAgentIndexMap.put(item.getUserAgent(),index);
                        item.setUserAgent(null);
                        item.setUserAgentIndex(index);
                    }
                }
                AccessLogMultiParams params = new AccessLogMultiParams();
                params.setLogList(logs);
                params.setUserAgentMap(userAgentIndexMap.inverse());
                remoteManagerLogService.accessMultiLog(params);
            }catch (Exception e){
                log.error("sso access log send error",e);
            }
        }
        void setLogs(List<AccessLogParams> logs) {
            this.logs = logs;
        }
    }



    @Override
    public int getOrder() {
        return 5;
    }
}
