package cn.com.duiba.sso.api.service.power;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.sso.api.common.tree.Tree;
import cn.com.duiba.sso.api.common.tree.TreeFactory;
import cn.com.duiba.sso.api.common.tree.TreeView;
import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemotePermissionService;
import cn.com.duiba.sso.api.service.eventbus.BizEventListener;
import cn.com.duiba.sso.api.tool.SystemInfo;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Function;

@Service
public class PowerTreeService {

    private static Logger logger = LoggerFactory.getLogger(PowerTreeService.class);
    /**
     * 权限树检测更新时间
     */
    private static final Long POWER_FLASH_TIME = 30L;
    //乐观读写锁
    private StampedLock lock = new StampedLock();
    @Autowired
    private RemotePermissionService remotePermissionService;

    private ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();

    /**
     * 权限树
     */
    private volatile Tree<PowerDto> powerTree = TreeFactory.createTree(Collections.emptyList());
    /**
     * URL映射Map,与权限树同时更新
     */
    private static Map<String,Set<Long>> POWER_URL_MAP = Maps.newConcurrentMap();

    /**
     * 权限树版本号
     */
    private AtomicLong powerTreeVersion = new AtomicLong(0L);


    /**
     * 监听系统信息初始化事件
     */
    @EventListener(MainContextRefreshedEvent.class)
    public void SystemInfoInitComplateEventListener(){
        loadPowerTree();
        executorService.scheduleWithFixedDelay(new PowerTreeFlushTask(),POWER_FLASH_TIME,POWER_FLASH_TIME,TimeUnit.SECONDS);
    }

    /**
     * 获取url命中的权限集合
     * @param url 请求链接
     * @return 命中的权限集合
     */
    public Set<Long> getPowerIdsByUrl(String url){
        if(powerTreeVersion.get()==0L){
            throw new SsoRunTimeException("权限系统服务未初始化成功,请稍后...");
        }
        long stamp = lock.tryOptimisticRead();
        Set<Long> powerIds = POWER_URL_MAP.getOrDefault(url,Collections.emptySet());
        if (!lock.validate(stamp)) {
            stamp = lock.readLock();
            try {
                powerIds = POWER_URL_MAP.getOrDefault(url,Collections.emptySet());
            } finally {
                lock.unlockRead(stamp);
            }
        }
        return powerIds;
    }

    /**
     * 获取单个节点
     * @param PowerId
     * @return
     */
    public PowerDto getPower(Long PowerId){
        return powerTree.getNode(PowerId);
    }

    /**
     * 得到一个权限树的副本
     * @return
     */
    public <T extends TreeView<T>> List<T> getPowerTree(Function<PowerDto,T> transform){
        return getPowerTree(Collections.emptySet(),transform);
    }

    /**
     * 获取用户的权限树
     * @param hasPowerIdSet 用户拥有权限Ids,为空时，返回系统完整的资源树
     * @param transform
     * @param <T>
     * @return
     */
    public <T extends TreeView<T>> List<T> getPowerTree(Set<Long> hasPowerIdSet,Function<PowerDto,T> transform){
        List<PowerDto> nodeList = powerTree.getTreeView();
        List<T> list = Lists.newArrayList();
        for(PowerDto power:nodeList){
            if(!hasPowerIdSet.isEmpty() && !hasPowerIdSet.contains(power.getId())){
                continue;
            }
            T node = transform.apply(power);
            if (node.getId()==null || node.getParentId()==null){
                throw new SsoRunTimeException("Function转化方法必须完成Id和parentId的转化");
            }
            list.add(node);
        }
        Tree<T> tree = TreeFactory.createTree(list);
        return TreeFactory.transformTree(tree);
    }

    /**
     * 获取权限在权限树上的遍历路径
     */
    public List<PowerDto> getPowerWay(Long powerId){
        return powerTree.getOneWayView(powerId);
    }

    /**
     * 加载权限树
     */
    private void loadPowerTree(){

        Long systemId = SystemInfo.getThisSystemId();

        List<PowerDto> treeList = remotePermissionService.loadPowerList(systemId);
        Tree<PowerDto> tree = TreeFactory.createTree(treeList);

        HashMultimap<String,Long> urlMap = HashMultimap.create();
        for(PowerDto power:treeList){
            for(String url:power.getUrls()){
                urlMap.put(url,power.getId());
            }
        }
        long stamp = lock.writeLock();
        try {
            powerTree = tree;
            POWER_URL_MAP.clear();
            for(String url:urlMap.keySet()){
                POWER_URL_MAP.put(url,urlMap.get(url));
            }
            if(powerTreeVersion.get()==0L){
                powerTreeVersion.set(new Date().getTime());
            }
        } finally {
            lock.unlockWrite(stamp);
        }
    }

    private class PowerTreeFlushTask implements Runnable{

        @Override
        public void run() {
            try{
                if(powerTreeVersion.get()==0){
                    loadPowerTree();
                    return;
                }
                Long version = remotePermissionService.getSystemPowerVersion(SystemInfo.getThisSystemId());
                if(powerTreeVersion.get()>=version){//如果本地的版本大于远程版本，不执行
                    return;
                }
                loadPowerTree();
                powerTreeVersion.set(version);
            }catch (Exception e){
                logger.error("权限树检测异常",e);
            }

        }
    }
}
