package cn.com.duiba.sso.api.tool;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.sso.api.domain.dto.SystemDto;
import cn.com.duiba.sso.api.domain.params.ApplicationInfoParams;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.properties.SpringApplicationProperties;
import cn.com.duiba.sso.api.properties.SsoBasicProperties;
import cn.com.duiba.sso.api.remoteservice.RemoteSystemService;
import cn.com.duiba.sso.api.service.system.EnvironmentEnum;
import cn.com.duiba.sso.api.service.system.SsoSystemService;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.config.ConfigFileApplicationListener;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.event.EventListener;
import org.springframework.core.env.Environment;

import java.util.Optional;

/**
 * Created by liuyao on 2017/12/5.
 * 本系统的Mate信息
 */
public class SystemInfo implements EnvironmentAware {

    private static Logger logger = LoggerFactory.getLogger(SystemInfo.class);

    public static final String SSO_ENV = "ssoEnv";

    private volatile static String systemAlias;
    private volatile static EnvironmentEnum environment;
    private volatile static String profiles;

    private static SsoSystemService ssoSystemService;

    private static RemoteSystemService remoteSystemService;
    @Autowired
    private SsoBasicProperties ssoBasicProperties;

    @EventListener(MainContextRefreshedEvent.class)
    public void MainContextRefreshedEventListener(){
        try{

            String version = SsoVersion.getVersion();
            ApplicationInfoParams params = new ApplicationInfoParams();
            params.setAppName(systemAlias);
            params.setVersion(version);
            params.setHomeUrl(ssoBasicProperties.getHomeUrl());
            params.setProfiles(profiles);

            remoteSystemService.submitApplicationInfoParams(params);
        }catch (Exception e){
            logger.error("向SSO提交应用原数据失败",e);
        }
    }

    /**
     * 获取本系统的信息
     */
    public static SystemDto getThisSystemInfo(){
        return ssoSystemService.getSystemByAlias(systemAlias);
    }

    /**
     * 获取本系统的Id
     */
    public static Long getThisSystemId(){
        SystemDto system = getThisSystemInfo();
        if(system==null){
            throw new SsoRunTimeException("没有检索到"+systemAlias+"对应的系统配置");
        }
        return system.getId();
    }

    @Autowired
    public void setRemoteSystemService(RemoteSystemService remoteSystemService) {
        SystemInfo.remoteSystemService = remoteSystemService;//NOSONAR
    }
    @Autowired
    public void setSpringApplicationProperties(SpringApplicationProperties springApplicationProperties) {
        SystemInfo.systemAlias = Optional.ofNullable(springApplicationProperties.getPriorName()).orElse(springApplicationProperties.getName());//NOSONAR
        if (StringUtils.isBlank(systemAlias)){
            throw new SsoRunTimeException("Sso客户端启动失败，请配置 spring.application.name 系统别名");
        }
    }

    @Autowired
    public void setSsoSystemService(SsoSystemService ssoSystemService) {
        SystemInfo.ssoSystemService = ssoSystemService;//NOSONAR
    }

    public static String getSystemAlias(){
        return SystemInfo.systemAlias;
    }

    public static String getProfiles(){
        return SystemInfo.profiles;
    }

    public static EnvironmentEnum getEnvironment(){
        return SystemInfo.environment;
    }

    @Override
    public void setEnvironment(Environment environment) {
        EnvironmentEnum environmentEnum = environment.getProperty("duiba.sso.environment",EnvironmentEnum.class);
        if(environmentEnum!=null){
            SystemInfo.environment = environmentEnum;//NOSONAR
            SystemInfo.profiles = environmentEnum.getName();//NOSONAR
            return;
        }
        String profiles = environment.getProperty(ConfigFileApplicationListener.ACTIVE_PROFILES_PROPERTY,"dev");
        SystemInfo.environment = parseEnvironmentEnum(profiles);//NOSONAR
        SystemInfo.profiles = profiles;//NOSONAR
    }

    public static EnvironmentEnum parseEnvironmentEnum(String profiles){
        if(StringUtils.isBlank(profiles)){
            return EnvironmentEnum.PROD;
        }
        for(EnvironmentEnum environmentEnum:EnvironmentEnum.values()){
            if(StringUtils.startsWith(profiles,environmentEnum.getName())){
                return environmentEnum;
            }
        }
        return EnvironmentEnum.PROD;
    }

}