package cn.com.duiba.sso.api.mappingmode;


import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.domain.properties.SsoBasicProperties;
import cn.com.duiba.sso.api.service.system.EnvironmentDecisionMaker;
import cn.com.duiba.sso.api.service.system.EnvironmentEnum;
import cn.com.duiba.sso.api.tool.SignTool;
import cn.com.duiba.sso.api.tool.SystemInfo;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.ttl.TransmittableThreadLocal;
import com.google.common.collect.Maps;
import feign.RequestInterceptor;
import feign.RequestTemplate;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.omg.CORBA.Environment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Map;
import java.util.Optional;

/**
 * 映射模式插件
 * 1.修改请求地址
 * 2.生成请求签名
 */
@Aspect
@Order(1)
public class SsoMappingModePlugin implements RequestInterceptor {

    private Logger logger = LoggerFactory.getLogger(SsoMappingModePlugin.class);

    @Autowired
    private SsoBasicProperties ssoBasicProperties;
    @Autowired
    private EnvironmentDecisionMaker environmentDecisionMaker;

    private static ThreadLocal<Boolean> SSO_INVOKE_FLAG = new TransmittableThreadLocal<>();

    public static final String SSO_RPC_MR = "X-SSO-MR";//映射模式标识符

    public static final String SSO_RPC_SIGN = "X-SSO-SIGN";//映射模式签名

    @Override
    public void apply(RequestTemplate template) {
        try{
            Boolean flag = Optional.ofNullable(SSO_INVOKE_FLAG.get()).orElse(false);
            if(!flag){
                return;
            }
            MappingModeRequestMateInfo mateInfo = new MappingModeRequestMateInfo();
            mateInfo.setSystemName(SystemInfo.getSystemAlias());
            mateInfo.setInterfaceName(template.url());
            mateInfo.setTimestamp(System.currentTimeMillis());

            EnvironmentEnum environment = environmentDecisionMaker.getEnvironmentEnum();
            mateInfo.setProfiles(environment.getName());

            if(StringUtils.isBlank(ssoBasicProperties.getAppSecret())){
                throw new SsoRunTimeException("启动SSO映射模式必须配置duiba.sso.app-secret");
            }
            String sign = createSign(ssoBasicProperties.getAppSecret(),mateInfo);
            mateInfo.setSign(sign);

            String rpcJson = JSONObject.toJSONString(mateInfo);
            template.header(SSO_RPC_MR,"true");
            template.header(SSO_RPC_SIGN, URLEncoder.encode(rpcJson,"UTF-8"));
        } catch (UnsupportedEncodingException e) {
            logger.error("json编码出错",e);
        } finally {
            SSO_INVOKE_FLAG.remove();
        }
    }

    @Before("execution(* cn.com.duiba.sso.api.remoteservice.*.*(..))")
    public void mappingModeRoute(){
        SSO_INVOKE_FLAG.set(ssoBasicProperties.getMappingMode() && !StringUtils.equals("sso-service",SystemInfo.getSystemAlias()));
    }


    public static String createSign(String appSecret,MappingModeRequestMateInfo mateInfo){
        Map<String,String> params = Maps.newHashMap();
        params.put("systemName",mateInfo.getSystemName());
        params.put("interfaceName",mateInfo.getInterfaceName());
        params.put("profiles",mateInfo.getProfiles());
        params.put("timestamp",mateInfo.getTimestamp().toString());
        return SignTool.sign(appSecret,params);

    }



}
