package cn.com.duiba.sso.api.common.interfaces;

import cn.com.duiba.application.boot.api.ApplicationProperties;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.sso.api.config.SsoExecutorServiceInitializer;
import cn.com.duiba.sso.api.domain.params.AppInterfaceInfoParams;
import cn.com.duiba.sso.api.remoteservice.RemoteAppInterfaceInfoService;
import com.google.common.base.Joiner;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.core.MethodParameter;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.AbstractHandlerMethodMapping;
import org.springframework.web.servlet.mvc.condition.PatternsRequestCondition;
import org.springframework.web.servlet.mvc.condition.RequestMethodsRequestCondition;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.stream.Collectors;


@Slf4j
public class AppInterfaceAnalysisService {

    private Set<String> whitePackages = Sets.newHashSet();
    {
        whitePackages.add("org.springframework");
        whitePackages.add("springfox");
        whitePackages.add("cn.com.duiba.sso.ui");
        whitePackages.add("cn.com.duiba.sso.api");
        whitePackages.add("cn.com.duibaboot");
    }

    private static final Integer SUBMIT_LIMIT = 10;

    private Joiner joiner = Joiner.on(",").skipNulls();

    private List<MethodInfoAnalysis> analysisList;

    @Resource(name = SsoExecutorServiceInitializer.SSO_THREAD_POOL_NAME)
    private ExecutorService executorService;
    @Resource
    private ApplicationContext applicationContext;
    @Resource
    private RemoteAppInterfaceInfoService remoteAppInterfaceInfoService;
    @Resource
    private ApplicationProperties applicationProperties;

    public void gatherAppInterfaceInfo(){
        GatherAppInterfaceInfoRunnble runnble = new GatherAppInterfaceInfoRunnble();
        executorService.submit(runnble);
    }


    private class GatherAppInterfaceInfoRunnble implements Runnable{

        @Override
        public void run() {

            String appName = applicationProperties.getApplicationName();
            if(StringUtils.isBlank(appName)){
                return;
            }
            try{
                Thread.sleep(10000);
            }catch (InterruptedException e){
                log.info("信息采集被中断",e);
                Thread.currentThread().interrupt();
            }

            Long version;
            try{
                version = remoteAppInterfaceInfoService.tryBeginSubmitInterfaceInfo(appName);
            }catch (BizException e){
                log.info("信息采集被拒绝:"+e.getMessage());
                return;
            }
            List<AppInterfaceInfoParams> infoParamsList = analysisCurrentAppInterface();

            ArrayListMultimap<Integer,AppInterfaceInfoParams> multimap = ArrayListMultimap.create();
            for(int i=0;i<infoParamsList.size();i++){
                int key = i / SUBMIT_LIMIT;
                multimap.put(key,infoParamsList.get(i));
            }

            int count = 0;

            try{
                for(Integer key:multimap.keySet()){
                    try{
                        List<AppInterfaceInfoParams> list = multimap.get(key);
                        count = count + remoteAppInterfaceInfoService.submitInterfaceInfo(appName,version,list);
                    }catch (BizException e){
                        throw e;
                    }catch (Exception e){
                        log.error("提交接口信息失败",e);
                    }
                }
                log.info("共提交",count);
                remoteAppInterfaceInfoService.endSubmit(appName,version);
            }catch (BizException e){
                log.info("提交接口信息失败",e);
                remoteAppInterfaceInfoService.rollback(appName,version);
            }
        }
    }

    public List<AppInterfaceInfoParams> analysisCurrentAppInterface() {

        Map<String, AbstractHandlerMethodMapping> mappingMap = applicationContext.getBeansOfType(AbstractHandlerMethodMapping.class);

        Collection<AbstractHandlerMethodMapping> methodMappings = mappingMap.values();

        String[] whitePackagesArray = whitePackagesArray();

        List<AppInterfaceInfoParams> infoParams = Lists.newArrayList();
        for(AbstractHandlerMethodMapping mapping:methodMappings){
            @SuppressWarnings("unchecked")
            Map<RequestMappingInfo, HandlerMethod> methods = mapping.getHandlerMethods();
            for(Map.Entry<RequestMappingInfo, HandlerMethod> entry:methods.entrySet()){

                HandlerMethod handlerMethod = entry.getValue();
                RequestMappingInfo requestMappingInfo = entry.getKey();

                String className = handlerMethod.getBeanType().getName();
                if(StringUtils.startsWithAny(className,whitePackagesArray)){
                    continue;
                }
                AppInterfaceInfoParams params = new AppInterfaceInfoParams();
                params.setController(className);
                params.setMethod(buildMethodName(handlerMethod));

                RequestMethodsRequestCondition methodsRequestCondition = requestMappingInfo.getMethodsCondition();
                params.setMethodTypes(methodsRequestCondition.getMethods().stream().map(Objects::toString).collect(Collectors.toSet()));
                PatternsRequestCondition patternsRequestCondition = requestMappingInfo.getPatternsCondition();
                params.setPaths(patternsRequestCondition.getPatterns());

                List<MethodInfo> infos = analysisList.stream().map((analysis)-> analysis.analysis(handlerMethod)).collect(Collectors.toList());

                MethodInfo methodMateDate = mergeMethodInfo(infos);

                params.getMethodTypes().addAll(methodMateDate.getTypeSet());

                params.setInterfaceName(methodMateDate.getName());
                params.setInterfaceComment(methodMateDate.getComment());
                infoParams.add(params);
            }
        }
        return infoParams;
    }

    private MethodInfo mergeMethodInfo(List<MethodInfo> infos){
        MethodInfo methodMateDate = new MethodInfo();
        for(MethodInfo info:infos){
            if(Objects.isNull(info)){
                continue;
            }
            if(StringUtils.isNotBlank(info.getName())){
                methodMateDate.setName(info.getName());
            }
            if(StringUtils.isNotBlank(info.getComment())){
                methodMateDate.setComment(info.getComment());
            }
            methodMateDate.getTypeSet().addAll(info.getTypeSet());
        }
        return methodMateDate;
    }

    private String[] whitePackagesArray(){
        String[] array = new String[whitePackages.size()];
        Lists.newArrayList(whitePackages).toArray(array);
        return array;
    }

    private String buildMethodName(HandlerMethod handlerMethod){
        Method method = handlerMethod.getMethod();

        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append(method.getName());

        MethodParameter[] methodParameters = handlerMethod.getMethodParameters();
        if(methodParameters.length==0){
            return stringBuilder.toString();
        }
        stringBuilder.append("(");

        List<String> paramsInfos = Lists.newArrayList();
        for(MethodParameter parameter:methodParameters){
            paramsInfos.add(parameter.getParameterType().getSimpleName());
        }
        stringBuilder.append(joiner.join(paramsInfos));
        stringBuilder.append(")");
        return stringBuilder.toString();
    }

    protected void setAnalysisList(List<MethodInfoAnalysis> analysisList) {
        this.analysisList = analysisList;
    }

    public void addExcludePackage(Set<String> packages){
        whitePackages.addAll(packages);
    }
}
