package cn.com.duiba.sso.api.service.export;

import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;

import javax.annotation.PostConstruct;
import java.io.File;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

@Scope("prototype")
public abstract class BaseExportRunnable<T> implements Runnable {

    private final Long taskId = new Date().getTime();

    @Autowired
    private SsoExportFileService ssoExportFileService;

    private BoundHashOperations<String,String,String> ops;

    protected Integer count;

    @PostConstruct
    public void init(){

        String key = ssoExportFileService.getTaskKey(taskId);
        StringRedisTemplate stringRedisTemplate = ssoExportFileService.getStringRedisTemplate();
        if(stringRedisTemplate==null){
            throw new SsoRunTimeException("SSO任务依赖StringRedisTemplate");
        }
        ops = stringRedisTemplate.boundHashOps(key);
        ops.put("successCount", "0");
        ops.put("count", "0");
        ops.put("success", "false");
        ops.put("url", "");
        ops.put("taskId", taskId.toString());

        stringRedisTemplate.expire(key,1, TimeUnit.DAYS);//导出结果保留一天
    }

    /**
     * 获取导出任务Id
     * @return
     */
    public Long getTaskId(){
        return taskId;
    }

    protected void setIncrementCount(Integer count){
        Objects.requireNonNull(count);
        ops.increment("successCount",count);
    }

    protected void setCount(Integer count){
        ops.put("count", count.toString());
        this.count = count;
    }

    protected void setMessage(String message){
        ops.put("message", message);
    }

    /**
     * 导出完成之后，下发给客户端一个能够全网访问的文件地址
     * @param url
     */
    protected void complete(String url){
        ops.put("url", url);
        ops.put("success", "true");
    }

    protected void error(String message){
        ops.put("message", message);
        ops.put("success", "true");
    }

    protected File createFile(String fileName,String fileType){
        File tempDir = ssoExportFileService.getTempDir();
        File file =  new File(tempDir, fileName+"-"+System.currentTimeMillis()+"."+fileType);
        ops.put("fileName", file.getName());
        return file;
    }

    /**
     * 提交任务参数，返回预期总记录数
     * @param params 业务参数
     * @return 预期总记录数
     */
    public abstract Integer submitParams(T params);
}
