package cn.com.duiba.sso.api.common.tree;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;

import java.io.Serializable;
import java.lang.reflect.Array;
import java.util.*;

public class TreeNode extends TreeNodeShim implements Serializable{

    protected static final Long VIRTUAL_PARENT_ID = -1L;

    //数据库形态
    private Long id;
    private Long parentId;
    private int weight = 0;

    //Id到index的映射(直接子节点)
    private Map<Long,TreeNode> childMap = Maps.newLinkedHashMap();
    private Map<Long,Set<Long>> allChildId = Maps.newHashMap();

    TreeNode(){}

    TreeNode(Long id, Long parentId){
        this.id = id;
        this.parentId = parentId;
    }

    TreeNode(Long id, Long parentId,int weight){
        this.id = id;
        this.parentId = parentId;
        this.weight = weight;
    }


    /**
     * 在当前和子代范围内，指定父节点进行添加子节点
     */
    public void addChildNode(TreeNode node){
        Long cparentId = node.getParentId();
        if(id.equals(cparentId)){
            //更新线索化的参数，
            node.setLevel(getLevel()+1);
            childMap.put(node.getId(),node);
            putChildValue(node.getId(),node.getId());
            return;
        }
        TreeNode childNode = findChildNode(cparentId);
        if(childNode!=null){//后代查找
            putChildValue(childNode.getId(),node.getId());
            childNode.addChildNode(node);//递归
        }
        //后代都没有，不做处理
    }

    /**
     * 是否存在后代节点
     */
    private boolean containsChildValue(Long destId){
        return findChildNode(destId) != null;
    }

    private void putChildValue(Long childId,Long destId){
        if(!allChildId.containsKey(childId)){
            allChildId.put(childId, Sets.newHashSet());
        }
        allChildId.get(childId).add(destId);
    }

    private void removeChildValue(Long childId,Long destId){
        if(!allChildId.containsKey(childId)){
            return;
        }
        allChildId.get(childId).remove(destId);
    }

    /**
     * 在本节点查找含有destChildId的直接子节点
     */
    private TreeNode findChildNode(Long destId){
        for(Long childId : allChildId.keySet()){
            if(allChildId.get(childId).contains(destId)){
                return childMap.get(childId);
            }
        }
        return null;
    }

    /**
     * 删除目标节点
     */
    public void deleteChildNode(Long destId) throws Exception{
        if(childMap.containsKey(destId)){//当前节点中有目标节点
            if(childMap.get(destId).hasChild()){
                throw new Exception("存在子节点，不能删除");
            }
            removeChildValue(destId,destId);
            childMap.remove(destId);
            return;
        }
        TreeNode childNode = findChildNode(destId);
        if(childNode!=null){//后代
            childNode.deleteChildNode(destId);//递归
            removeChildValue(childNode.getId(),destId);
        }
        //后代都没有，不做处理
    }

    /**
     * 先序遍历当前节点
     */
    public List<TreeNode> getAllNode(){
        List<TreeNode> nodeList = Lists.newLinkedList();
        if(!Objects.equals(this.parentId,VIRTUAL_PARENT_ID)){//不添加虚拟根节点
            nodeList.add(this);//先序遍历
        }
        if(hasChild()){
            for(TreeNode childNode : childMap.values()){
                List<TreeNode> childNodes = childNode.getAllNode();
                Collections.sort(childNodes);
                nodeList.addAll(childNodes);
            }
        }
        return nodeList;
    }

    /**
     * 后代节点的总数
     */
    public int getChildSize(){
        return allChildId.size();
    }

    /**
     * 是否有子节点
     */
    public boolean hasChild(){
        return !childMap.isEmpty();
    }

    /**
     * 包括自己是否存在id节点
     */
    public boolean containsId(Long id){
        return (!Objects.equals(null, this.id) && this.id.equals(id)) || containsChildValue(id);
    }

    /**
     * 记录根到目标节点的路径
     * @param destId 目标节点Id
     * @param nodeWayList 记录从根节点到目标节点的Id序列
     */
    public void findNodeWay(Long destId,List<Long> nodeWayList){
        if(!containsId(destId)){
            return;
        }
        if(!Objects.equals(this.parentId,VIRTUAL_PARENT_ID)){
            nodeWayList.add(this.id);
        }
        TreeNode childNode = findChildNode(destId);
        if(childNode!=null){
            childNode.findNodeWay(destId,nodeWayList);
        }
    }

    @Override
    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }
    @Override
    public Long getParentId() {
        return parentId;
    }

    public void setParentId(Long parentId) {
        this.parentId = parentId;
    }

}
