package cn.com.duiba.sso.api.service.logger.service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.sso.api.service.logger.annotation.SsoLoggerMethod;
import cn.com.duiba.sso.api.service.logger.domain.SsoLoggerStackFrame;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.core.annotation.AnnotationUtils;

import javax.annotation.Resource;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Objects;

/**
 *  日志切面
 * @author liuyao
 */
@Aspect
public class SsoLoggerInterceptor {

    @Resource
    private SsoLoggerService ssoLoggerService;

    public static final String RESPONSE = "response";

    private final LocalVariableTableParameterNameDiscoverer discoverer = new LocalVariableTableParameterNameDiscoverer();

    @Around("@annotation(cn.com.duiba.sso.api.service.logger.annotation.SsoLoggerMethod)")
    public Object logger(ProceedingJoinPoint point) throws Throwable {
        MethodSignature methodSignature = (MethodSignature) point.getSignature();
        Method method = methodSignature.getMethod();
        SsoLoggerMethod loggerMethod = AnnotationUtils.findAnnotation(method, SsoLoggerMethod.class);
        String template = Objects.requireNonNull(loggerMethod).value();
        String group = loggerMethod.group();
        if (StringUtils.isBlank(template)) {
            throw new IllegalArgumentException("操作日志: " + group + ", template不能为空");
        }
        SsoLoggerStackFrame frame = new SsoLoggerStackFrame();
        frame.setTemplate(template);
        frame.setGroup(group);
        ssoLoggerService.initStackContext(frame);

        if (loggerMethod.request()) {
            Object[] arguments = point.getArgs();
            String[] paramNames = discoverer.getParameterNames(method);
            for (int i = 0; i < Objects.requireNonNull(paramNames).length; i++) {
                String paramName = paramNames[i];
                Object value = arguments[i];
                if(value instanceof Serializable){
                    frame.addParam(paramName, value);
                }
            }
        }
        Object result;
        try {
            result = point.proceed();
            if (loggerMethod.response() && (result instanceof Serializable)) {
                frame.addParam(RESPONSE, result);
            }
            return result;
        } catch (Exception e) {
            //记录为异常日志
            Throwable exception = e;
            while (exception instanceof InvocationTargetException) {
                exception = ((InvocationTargetException) exception).getTargetException();
            }
            frame.setErrorLog(!(exception instanceof BizException));
            if (StringUtils.isNotBlank(e.getMessage())) {
                String msg = e.getMessage().length() > 1000 ? e.getMessage().substring(0, 1000) : e.getMessage();
                frame.addParam(RESPONSE, msg);
            }
            throw e;
        } finally {
            ssoLoggerService.flushThreadContext();
        }
    }


}
