package cn.com.duiba.sso.api.service.power;

import cn.com.duiba.sso.api.common.tree.Tree;
import cn.com.duiba.sso.api.common.tree.TreeFactory;
import cn.com.duiba.sso.api.domain.dto.PowerDto;
import cn.com.duiba.sso.api.exception.SsoRunTimeException;
import cn.com.duiba.sso.api.remoteservice.RemotePermissionService;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.HashMultimap;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;

import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

@Slf4j
public class PowerTreeService {

    @Autowired
    private RemotePermissionService remotePermissionService;

    private LoadingCache<Long,PowerTreeCache> treeCache = Caffeine.newBuilder().expireAfterWrite(1, TimeUnit.HOURS).build((systemId)->{
        List<PowerDto> treeList = remotePermissionService.loadPowerList(systemId);
        Long version = remotePermissionService.getSystemPowerVersion(systemId);
        Tree<PowerDto> tree = TreeFactory.createTree(treeList);

        HashMultimap<String,Long> urlMap = HashMultimap.create();
        for(PowerDto power:treeList){
            for(String url:power.getUrls()){
                urlMap.put(url,power.getId());
            }
        }
        PowerTreeCache cache = new PowerTreeCache(tree,urlMap);
        cache.setVersion(version);
        return cache;
    });

    /**
     * 根据系统Id获取的该系统下的资源树
     */
    public PowerTreeCache getPowerTreeCache(Long systemId){
        try{
            return treeCache.get(systemId);
        }catch (Exception e){
            log.error("加载systemId:"+systemId+"的资源异常",e);
            throw new SsoRunTimeException("资源加载失败");
        }
    }

    /**
     * 任务调度
     */
    @Scheduled(cron = "0/15 0/1 * * * ?")
    public void powerTreeFlushTask() {
        Set<Long> systemIds = treeCache.asMap().keySet();
        for(Long systemId:systemIds){
            try{
                Long version = remotePermissionService.getSystemPowerVersion(systemId);
                PowerTreeCache cache = treeCache.getIfPresent(systemId);
                if(cache==null){
                    continue;
                }
                if(cache.getVersion()>=version){//如果本地的版本大于远程版本，不执行
                    continue;
                }
                treeCache.invalidate(systemId);
            }catch (Exception e){
                log.error("权限资源检查失败",e);
            }
        }
    }
}
