package cn.com.duiba.sso.ui.resource;

import cn.com.duiba.sso.api.remoteservice.RemoteSystemService;
import cn.com.duiba.sso.api.tool.JsonRender;
import cn.com.duiba.sso.api.tool.RequestTool;
import cn.com.duiba.sso.api.tool.SystemInfo;
import cn.com.duiba.sso.api.web.interceptor.annotation.AuthCanAccess;
import cn.com.duiba.sso.api.web.interceptor.annotation.CanAccess;
import cn.com.duiba.sso.ui.mume.MumeLoader;
import cn.com.duiba.sso.ui.mume.SsoUIMume;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.http.CacheControl;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import static org.springframework.web.bind.annotation.RequestMethod.GET;

@Controller
public class SsoUIController {

    private Logger logger = LoggerFactory.getLogger(SsoUIController.class);

    @Autowired
    private MumeLoader mumeLoader;
    @Autowired
    private SsoUIResManager ssoUIResManager;
    @Autowired
    private RemoteSystemService remoteSystemService;

    @Value("classpath:ui/index.html")
    Resource indexHtml;

    @CanAccess
    @ResponseBody
    @RequestMapping(value = "/sso/ui/uiMume",method = RequestMethod.GET)
    public JsonRender uiMume(){

        try{
            List<SsoUIMume> ssoUIMume =  mumeLoader.loadUIMume();

            fixMumeUrl(ssoUIMume);

            JsonRender render = JsonRender.successResult();
            render.put("mumes",ssoUIMume);
            render.put("webTitle",ssoUIResManager.getWebTitle());
            return render;
        }catch (Exception e){
            return JsonRender.failResult(e);
        }
    }

    @AuthCanAccess
    @RequestMapping(value = "/ui/index.html", method = GET)
    public ResponseEntity serveIndex(){
        ResponseEntity.BodyBuilder result = ResponseEntity.ok()
                .cacheControl(CacheControl.maxAge(1, TimeUnit.MINUTES))
                .contentType(MediaType.TEXT_HTML);
        return result.body(indexHtml);
    }

    @AuthCanAccess
    @RequestMapping(value = {"/ui/","/ui"},method = GET)
    public ModelAndView forwardUiEndpoints() {
        return new ModelAndView("forward:/ui/index.html");
    }

    @AuthCanAccess
    @RequestMapping(value = "/ui/main.js", method = GET)
    public ResponseEntity mainJs() throws IOException {
        ResponseEntity.BodyBuilder result = ResponseEntity.ok()
                .cacheControl(CacheControl.maxAge(1, TimeUnit.MINUTES))
                .contentType(MediaType.parseMediaType("application/javascript"));
        return result.body(ssoUIResManager.getMainJs());
    }

    private void fixMumeUrl(List<SsoUIMume> ssoUIMumes){

        List<SsoUIMume> mumes = traversal(ssoUIMumes);
        Set<String> appNames = Sets.newHashSet();


        List<SsoUIMume> canFixList = Lists.newArrayList();
        for(SsoUIMume mume:mumes){
            if(StringUtils.isNotBlank(mume.getSystem())){
                appNames.add(mume.getSystem());
                canFixList.add(mume);
            }
        }
        Map<String,String> homeUrlMap = remoteSystemService.findAppHomeUrlMap(SystemInfo.getProfiles(),appNames);

        for(SsoUIMume mume:canFixList){
            try{
                if(!homeUrlMap.containsKey(mume.getSystem())){
                    continue;
                }
                String homeUrl = homeUrlMap.get(mume.getSystem());

                if(StringUtils.startsWith(mume.getUrl(),"/")){
                    mume.setUrl(homeUrl+mume.getUrl());
                }
                if(StringUtils.startsWith(mume.getUrl(),"http")){
                    String url = mume.getUrl();
                    URL url1 = new URL(homeUrl);
                    URL url2 = new URL(mume.getUrl());

                    url = url.replace(url2.getAuthority(),url1.getAuthority());
                    mume.setUrl(url);
                }
            }catch (Exception e){
                logger.debug("url:"+mume.getUrl()+"解析失败",e);
            }
        }
    }

    private List<SsoUIMume> traversal(List<SsoUIMume> ssoUIMumes){
        List<SsoUIMume> returnList = Lists.newArrayList();
        for(SsoUIMume mume:ssoUIMumes){
            returnList.add(mume);
            if(mume.getChild()!=null && !mume.getChild().isEmpty()){
                returnList.addAll(traversal(mume.getChild()));
            }
        }
        return returnList;
    }

}
