package cn.com.duiba.sso.ui.resource;

import cn.com.duiba.application.boot.api.component.environment.Environment;
import cn.com.duiba.application.boot.api.component.environment.EnvironmentService;
import cn.com.duiba.application.boot.api.component.environment.RequestEnvironmentDecisionMaker;
import cn.com.duiba.application.boot.api.domain.dto.ApplicationDto;
import cn.com.duiba.application.boot.api.domain.dto.ClusterDto;
import cn.com.duiba.application.boot.api.remoteservice.RemoteClusterService;
import cn.com.duiba.sso.api.remoteservice.RemoteSSOService;
import cn.com.duiba.sso.api.service.HostEnvironment;
import cn.com.duiba.sso.api.service.HostEnvironmentMapping;
import cn.com.duiba.sso.api.tool.JsonRender;
import cn.com.duiba.sso.api.tool.SystemInfo;
import cn.com.duiba.sso.api.web.interceptor.annotation.AuthCanAccess;
import cn.com.duiba.sso.api.web.interceptor.annotation.CanAccess;
import cn.com.duiba.sso.ui.SsoUIProperties;
import com.alibaba.fastjson.JSONObject;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.CacheControl;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.Resource;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static org.springframework.web.bind.annotation.RequestMethod.GET;

@Controller
public class SsoUIController {

    @Resource
    private SsoUIResManager ssoUIResManager;
    @Resource
    private RequestEnvironmentDecisionMaker requestEnvironmentDecisionMaker;
    @Resource
    private HostEnvironmentMapping hostEnvironmentMapping;
    @Resource
    private RemoteSSOService remoteSSOService;
    @Resource
    private EnvironmentService environmentService;
    @Resource
    private RemoteClusterService remoteClusterService;
    @Resource
    private SsoUIProperties ssoUIProperties;

    private LoadingCache<HostEnvironment, ClusterDto> clusterCache = Caffeine.newBuilder().expireAfterWrite(1,TimeUnit.MINUTES).build(new CacheLoader<HostEnvironment, ClusterDto>() {
        @Nullable
        @Override
        public ClusterDto load(@Nonnull HostEnvironment hostEnvironment){
            return remoteClusterService.findByUK(hostEnvironment.getEnvironment(),hostEnvironment.getCluster());
        }
    });

    @CanAccess
    @ResponseBody
    @RequestMapping(value = "${duiba.sso.root-path:'/sso'}/ui/systemInfo")
    public JsonRender systemInfo(){

        JsonRender render = JsonRender.successResult();
        render.put("ssoHomeURL",remoteSSOService.findSsoHomeUrl());
        ApplicationDto systemDto = SystemInfo.getThisSystemInfo();
        render.put("system",systemDto);

        Environment currentEnvironment = requestEnvironmentDecisionMaker.getRequestEnvironment();

        List<HostEnvironment> hostEnvironments = hostEnvironmentMapping.getHostEnvironmentMappings();
        String clusterKey = environmentService.currentClusterKey();

        List<JSONObject> environments = hostEnvironments.stream().map((item)->{

            ClusterDto clusterDto = clusterCache.get(item);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("host",item.getHost());
            Environment environment = item.getEnvironment();
            jsonObject.put("clusterName", Optional.ofNullable(clusterDto).map(ClusterDto::getClusterName).orElse("未知"));
            jsonObject.put("currentCluster",environment==currentEnvironment && StringUtils.equals(clusterKey,item.getCluster()));
            return jsonObject;
        }).collect(Collectors.toList());

        render.put("hostEnvironmentMapping",environments);
        return render;
    }

    @AuthCanAccess
    @RequestMapping(value = "/ui/index.html", method = GET)
    public ResponseEntity serveIndex() throws IOException {
        ResponseEntity.BodyBuilder result = ResponseEntity.ok()
                .cacheControl(CacheControl.maxAge(1, TimeUnit.MINUTES))
                .contentType(MediaType.TEXT_HTML);
        return result.body(ssoUIResManager.indexHtml());
    }

    @AuthCanAccess
    @RequestMapping(value = {"/ui","${duiba.sso.root-path:'/sso'}/","${duiba.sso.root-path:'/sso'}"},method = GET)
    public ModelAndView forwardUiEndpoints() {
        return new ModelAndView("forward:"+ssoUIProperties.getIndexFilePath());
    }

    @AuthCanAccess
    @RequestMapping(value = "${duiba.sso.root-path:'/sso'}/ui/{version}/main.js", method = GET)
    public ResponseEntity mainJs() throws IOException {
        ResponseEntity.BodyBuilder result = ResponseEntity.ok()
                .header("Cache-Control","no-cache,no-store")
                .contentType(MediaType.parseMediaType("application/javascript"));
        return result.body(ssoUIResManager.getMainJs());
    }

}
