package cn.com.duiba.stock.service.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.stock.service.api.dto.NewStockReqDto;
import cn.com.duiba.stock.service.api.dto.NewStockRespDto;
import cn.com.duiba.stock.service.api.dto.StockDto;
import cn.com.duiba.stock.service.api.dto.UpdateStockReqDto;
import cn.com.duiba.wolf.dubbo.DubboResult;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by xuezhaoming on 16/5/16.
 */
@AdvancedFeignClient
public interface RemoteStockBackendService {

    /**
     * 增加count 个指定库存ID的库存
     *
     * @param stockId 库存ID
     * @param count   库存个数
     * @return true :成功 flase : 失败
     */
    DubboResult<Boolean> increaseItemStock(long stockId, long count);

    /**
     * 减少count 个指定库存ID的库存
     *
     * @param stockId 库存ID
     * @param count   减少库存个数
     * @return true :成功 flase : 失败
     */
    DubboResult<Boolean> decreaseItemStock(long stockId, long count);

    /**
     * 批量增加库存
     * bug：result只取决于最后一条数据的成功或失败。后续请使用updateItemStockBatch
     *
     * @param reqs 库存ID，库存数集合
     * @return
     */
    @Deprecated
    DubboResult<Boolean> increaseItemStockBatch(List<UpdateStockReqDto> reqs);

    /**
     * 批量减少库存
     * bug：result只取决于最后一条数据的成功或失败。后续请使用updateItemStockBatch
     *
     * @param reqs 库存ID，库存数集合
     * @return
     */
    @Deprecated
    DubboResult<Boolean> decreaseItemStockBatch(List<UpdateStockReqDto> reqs);


    /**
     * 批量减少库存
     * 1：可通过正负判断增减库存
     * 2：修复increaseItemStockBatch和decreaseItemStockBatch执行时只要最后一条数据成功，result就为true的bug
     *
     * @param reqs 库存ID，库存数集合;由于可通过正负处理数据，请自行合并数据
     * @return
     */
    DubboResult<Boolean> updateItemStockBatch(List<UpdateStockReqDto> reqs);

    /**
     * 新增商品库存
     *
     * @param gid   商品ID
     * @param stock 库存量
     * @return 库存ID
     */
    DubboResult<Long> newStock(long gid, long stock);

    /**
     * 新增商品库存
     *
     * @param gid          商品ID
     * @param currentStock 当前库存
     * @param totalStock   库存总量
     * @return 库存ID
     */
    DubboResult<Long> newTotalStock(long gid, long currentStock, long totalStock);

    /**
     * 批量创建库存
     *
     * @param reqs List<NewStockReqDto> 商品ID初始化多少库存集合
     * @return List<NewStockRespDto> 商品ID对应库存ID集合
     */
    DubboResult<List<NewStockRespDto>> newStockBatch(List<NewStockReqDto> reqs);


    /**
     * 根据库存ID 查询库存
     *
     * @param stockID 库存ID
     * @return 库存数据
     */
    DubboResult<Long> find(long stockID);

    /**
     * 获取总库存
     *
     * @param stockId 库存ID
     * @return 总库存
     */
    DubboResult<Long> findTotalStock(long stockId);


    /**
     * 批量查询库存数量<br/>
     * 没有缓存
     *
     * @param stockIds 库存ID集合, 最大只允许1000个
     * @return 集合 key :库存ID value : 库存量
     */
    Map<Long, Long> findBatch(List<Long> stockIds) throws BizException;

    /**
     * 批量查询库存DTO<br/>
     * 无缓存
     *
     * @param stockIds 库存ID集合
     * @return 库存stockDto集合
     * @return
     */
    List<StockDto> findBatchByIds(List<Long> stockIds);
}
