package cn.com.duiba.supplier.channel.service.api.enums.channel;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.Getter;

/**
 * 企迈券状态变更类型枚举
 * @author yangyukang
 * @date 2024/12/19 6:00 PM
 */
@Getter
@AllArgsConstructor
public enum QiMaiCouponChangeTypeEnum {

    VERIFICATION(2, "核销"),
    REVERSAL_VERIFICATION(3, "冲正-核销"),
    RECOVERY(7, "回收");

    /**
     * 变更类型码
     */
    private final Integer code;

    /**
     * 变更类型描述
     */
    private final String desc;

    private static final Map<Integer, QiMaiCouponChangeTypeEnum> CODE_MAP = Collections.unmodifiableMap(
            Arrays.stream(values()).collect(Collectors.toMap(QiMaiCouponChangeTypeEnum::getCode, Function.identity(), (v1, v2) -> v2))
    );

    /**
     * 根据变更类型码获取枚举
     * @param code 变更类型码
     * @return 对应的枚举，如果不存在则返回null
     */
    public static QiMaiCouponChangeTypeEnum getByCode(Integer code) {
        if (code == null) {
            return null;
        }
        return CODE_MAP.get(code);
    }

    /**
     * 根据变更类型码获取描述
     * @param code 变更类型码
     * @return 对应的描述，如果不存在则返回"未知变更类型"
     */
    public static String getDescByCode(Integer code) {
        QiMaiCouponChangeTypeEnum changeTypeEnum = getByCode(code);
        return changeTypeEnum != null ? changeTypeEnum.getDesc() : "未知变更类型";
    }

    /**
     * 判断是否为核销类型
     * @param code 变更类型码
     * @return true 如果是核销类型，false 否则
     */
    public static boolean isVerification(Integer code) {
        return VERIFICATION.getCode().equals(code);
    }

    /**
     * 判断是否为冲正核销类型
     * @param code 变更类型码
     * @return true 如果是冲正核销类型，false 否则
     */
    public static boolean isReversalVerification(Integer code) {
        return REVERSAL_VERIFICATION.getCode().equals(code);
    }

    /**
     * 判断是否为回收类型
     * @param code 变更类型码
     * @return true 如果是回收类型，false 否则
     */
    public static boolean isRecovery(Integer code) {
        return RECOVERY.getCode().equals(code);
    }
}
