package cn.com.duiba.biz.phoneflow.supplier;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.domain.MessageUniqueCheckDO;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.domain.SupplierResponse;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.service.MessageService;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.tool.JsonTool;

/**
 * 魔品流量接口请求
 */
@Service
public class MoPinPhoneflowSupplier {

	private static Logger log = LoggerFactory.getLogger(MoPinPhoneflowSupplier.class);

	@Autowired
	private MessageService messageService;
	@Autowired
	private HttpAsyncClientPool httpAsyncClientPool;

	/**
	 * 提交到HTTP请求池
	 * @param message
	 * @param callbackTopic
	 * @param msgTag
	 * @param msgKey
	 */
	public void asyncSubmit(final SupplierRequest message, final String callbackTopic, final String msgTag, final String msgKey) {
		if (!messageService.messageUniqueCheck(message.getSupplierOrderId(), MessageUniqueCheckDO.TYPE_PHONEFLOW)) {
			return;
		}
		HttpPost post = new HttpPost(message.getHttpUrl());
		post.setHeader("Content-Type", "application/octet-stream");
		post.setHeader("charset", "UTF-8");
		post.setHeader("d", message.getParams().get("digest"));
		post.setHeader("s", message.getParams().get("signature"));
		HttpEntity entity = new StringEntity(message.getParams().get("ciphertext"), "UTF-8");
		post.setEntity(entity);

		HttpRequestLog.logUrl("[action phoneflow] [tag request] [bizId " + message.getOrderId() + "] [url " + message.getHttpUrl() + "]");
		httpAsyncClientPool.submit(message.getAppId(), post, new FutureCallback<HttpResponse>() {

			@Override
			public void completed(HttpResponse response) {
				SupplierResponse resp = new SupplierResponse();
				try {
					resp.setOrderId(message.getOrderId());
					resp.setSupplierOrderId(message.getSupplierOrderId());
					resp.setSupplierName(message.getSupplierName());
					resp.setUrl(message.getHttpUrl());
					resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_COMPLETED);
					resp.setAppId(message.getAppId());
					resp.setConsumerId(message.getConsumerId());
					String body = EntityUtils.toString(response.getEntity());
					resp.setBody(body);
				} catch (Exception e) {
					log.error("MoPinPhoneflowSupplier completed", e);
				} finally {
					finallyBlock(message, resp, callbackTopic, msgTag, msgKey);
				}
			}

			@Override
			public void failed(Exception ex) {
				log.error("MoPinPhoneflowSupplier failed orderId:" + message.getOrderId(), ex);
				SupplierResponse resp = new SupplierResponse();
				try {
					resp.setOrderId(message.getOrderId());
					resp.setSupplierOrderId(message.getSupplierOrderId());
					resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_FAILED);
					resp.setAppId(message.getAppId());
					resp.setConsumerId(message.getConsumerId());
					resp.setErrorMessage(ex.getClass().getName() + ":" + ex.getMessage());
				} catch (Exception e) {
					log.error("MoPinPhoneflowSupplier completed", e);
				} finally {
					finallyBlock(message, resp, callbackTopic, msgTag, msgKey);
				}
			}

			@Override
			public void cancelled() {
				SupplierResponse resp = new SupplierResponse();
				try {
					resp.setOrderId(message.getOrderId());
					resp.setSupplierOrderId(message.getSupplierOrderId());
					resp.setCallbackType(SupplierResponse.CALLBACK_TYPE_CANCELLED);
					resp.setAppId(message.getAppId());
					resp.setConsumerId(message.getConsumerId());
				} catch (Exception e) {
					log.error("MoPinPhoneflowSupplier cancelled", e);
				} finally {
					finallyBlock(message, resp, callbackTopic, msgTag, msgKey);
				}
			}

		});

	}
	
	private void finallyBlock(SupplierRequest message, SupplierResponse response, String callbackTopic, String msgTag, String msgKey) {
		try {
			response.setUrl(message.getHttpUrl());
			response.setSupplierName(message.getSupplierName());
			response.setSupplierOrderId(message.getSupplierOrderId());
			response.setOrderId(message.getOrderId());
			response.setAppId(message.getAppId());
			response.setConsumerId(message.getConsumerId());
			String body = JsonTool.objectToJson(response);
			HttpRequestLog.logUrl("[action phoneflow] [tag response] [bizId " + message.getOrderId() + "] [body " + response.getBody() + "] [callback " + response.getCallbackType() + "]");
			if(StringUtils.isEmpty(msgTag) && StringUtils.isEmpty(msgKey)){
				messageService.sendMsg(callbackTopic, body);
			}else{
				messageService.sendMsg(callbackTopic, msgTag, msgKey, body);
			}
		} catch (Exception e) {
			log.error("VirtualSupplier callback:" + message.getOrderId(), e);
		}
	}

}
