package cn.com.duiba.crecord.message;

import cn.com.duiba.service.KafkaClient;
import cn.com.duiba.service.ThreadPoolService;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.Collections;
import java.util.Properties;

/**
 * 兑换记录消费者<br/>
 */
@Service
public class CrecordConsumer implements InitializingBean {

	private static Logger log = LoggerFactory.getLogger(CrecordConsumer.class);

	@Value("${notify.topic.crecord.requset}")
	private String crecordRequest;

	@Autowired
	private KafkaClient kafkaClient;
	@Autowired
	private CrecordAsyncHttp crecordAsyncHttp;

	private static int pool = 3;

	@Autowired
	private ThreadPoolService threadPoolService;

	/**
	 * 初始化启动
	 */
	@Override
	public void afterPropertiesSet() {
		start();
	}

	/**
	 * 启动兑换记录同步消费者
	 */
	public synchronized void start() {
		for (int i = 0; i < pool; i++) {
			threadPoolService.submit(new Runnable() {
				@Override
				public void run() {
					whilePoll();
				}
			});
		}
		log.info("CrecordConsumer started");
	}

	private void whilePoll() {
		Properties props = new Properties();
		props.put("bootstrap.servers", kafkaClient.getBootstrapServers());
		props.put("group.id", "notify-center");
		props.put("enable.auto.commit", "false");
		props.put("session.timeout.ms", "30000");
		props.put("auto.offset.reset", "earliest");
		props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
		props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");

		try (KafkaConsumer<String, String> consumer = new KafkaConsumer<>(props)){
			consumer.subscribe(Arrays.asList(crecordRequest));
			while (true) {
				ConsumerRecords<String, String> records = consumer.poll(100);
				for (TopicPartition pt : records.partitions()) {
					commit(consumer, records, pt);
				}
			}
		} catch (Exception e) {
			log.error("error:", e);
		}
	}

	private void commit(KafkaConsumer<String, String> consumer, ConsumerRecords<String, String> records, TopicPartition pt) {
		for (ConsumerRecord<String, String> record : records.records(pt)) {
            try {
                crecordAsyncHttp.asyncSubmit(record.value());
                long lastOffset = record.offset();
                consumer.commitSync(Collections.singletonMap(pt, new OffsetAndMetadata(lastOffset + 1)));
            } catch (Exception e) {
                log.error("CrecordConsumer error:", e);
            }
        }
	}

}
