package cn.com.duiba.notifycenter.service;

import cn.com.duiba.service.ThreadPoolService;
import com.alibaba.fastjson.JSONObject;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 通知服务HTTP连接池
 */
@Service
public class NotifyHttpClientPool implements InitializingBean, DisposableBean {

	private static final Logger log = LoggerFactory.getLogger(NotifyHttpClientPool.class);

	/**
	 * 连接超时
	 */
	public static final int CONNECT_TIMEOUT = 30 * 1000;

	/**
	 * 长链接空闲时间
	 */
	public static final int KEEPALIVE_TIMEOUT = 30 * 1000;

	/**
	 * 处理超时
	 */
	public static final int SOCKET_TIMEOUT = 30 * 1000;

	/**
	 * 最大总连接值
	 */
	public static final int MAX_CONNECT = 1024;

	/**
	 * 每个路由最大连接{并发}值
	 */
	public static final int MAX_ROUTE_CONNECT = 128;

	private CloseableHttpAsyncClient httpClient;

	private Map<Long, AtomicInteger> runningStat = new ConcurrentHashMap<>();

	@Autowired
	private ThreadPoolService threadPoolService;

	/**
	 * 执行HTTP异步请求
	 * 
	 * @param appId
	 * @param request
	 * @param callback
	 */
	public void execute(Long appId, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
		try {
			long s = System.currentTimeMillis();
			AtomicInteger count = runningStat.get(appId);
			if (count == null) {
				count = new AtomicInteger(1);
				runningStat.put(appId, count);
			} else {
				count.incrementAndGet();
			}
            request.setHeader("User-Agent","Mozilla/5.0 (iPhone; CPU iPhone OS 9_1 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Version/9.0 Mobile/13B143 Safari/601.1");
            CallbackProcesser process = new CallbackProcesser(appId, request, callback);
			if (!httpClient.isRunning()) {
				start();
			}
            httpClient.execute(process.getRequest(), process);
			long e = System.currentTimeMillis();
			long t = e - s;
			if (t > 1000) {
				log.warn("notify httpClent.execute > {} ms", t);
			}
		} catch (Exception e) {
			callback.failed(e);
			log.error("execute:", e);
		}
	}

	/**
	 * 提交HTTP异步请求
	 * 
	 * @param appId
	 * @param request
	 * @param callback
	 */
	public void submit(final Long appId, final HttpUriRequest request, final FutureCallback<HttpResponse> callback) {
		if (threadPoolService.canSubmit()) {
			threadPoolService.submit(new Runnable() {
				@Override
				public void run() {
					execute(appId, request, callback);
				}
			});
		} else {
			execute(appId, request, callback);
		}
	}

	/**
	 * 连接池详细信息
	 * 
	 * @return
	 */
	public JSONObject dumpDetail() {
		Map<Long, Object> map = new HashMap<>();
		for (Map.Entry<Long, AtomicInteger> entry : runningStat.entrySet()) {
			if (entry.getValue().intValue() > 0) {
				map.put(entry.getKey(), entry.getValue());
			}
		}
		JSONObject o = new JSONObject();
		o.put("runningStat", map);
		return o;
	}

	/**
	 * 连接池大小
	 * 
	 * @return
	 */
	public int dumpSize() {
		int total = 0;
		for (AtomicInteger a : runningStat.values()) {
			total += a.get();
		}
		return total;
	}

	/**
	 * 通知请求回调类
	 */
	class CallbackProcesser implements FutureCallback<HttpResponse> {
		private HttpUriRequest request;
		private FutureCallback<HttpResponse> callback;
		private Long appId;

		/**
		 * CallbackProcesser
		 * 
		 * @param appId
		 * @param request
		 * @param callback
		 */
		public CallbackProcesser(Long appId, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
			this.request = request;
			this.callback = callback;
			this.appId = appId;
		}

		@Override
		public void completed(HttpResponse result) {
			try {
				callback.completed(result);
			} finally {
				runningStat.get(appId).decrementAndGet();
			}
		}

		@Override
		public void failed(Exception ex) {
			try {
				callback.failed(ex);
			} finally {
				runningStat.get(appId).decrementAndGet();
			}
		}

		@Override
		public void cancelled() {
			try {
				callback.cancelled();
			} finally {
				runningStat.get(appId).decrementAndGet();
			}
		}

		/**
		 * getRequest
		 * 
		 * @return
		 */
		public HttpUriRequest getRequest() {
			return request;
		}

	}

	@Override
	public void afterPropertiesSet() throws Exception {
		start();
	}

	@Override
	public void destroy() throws Exception {
		if (httpClient != null && httpClient.isRunning()) {
			httpClient.close();
			log.info("Notify httpClient closed");
		}
	}

	/**
	 * 启动HTTP服务
	 */
	private synchronized void start() {
		if (httpClient != null && httpClient.isRunning()) {
			return;
		}
		RequestConfig config = RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setConnectionRequestTimeout(CONNECT_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
		httpClient = HttpAsyncClients.custom().setDefaultRequestConfig(config).setMaxConnTotal(MAX_CONNECT).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy()).build();
		httpClient.start();
		log.info("Notify httpClient started");
	}

	/**
	 * 长连接
	 * 
	 * @return
	 */
	private DefaultConnectionKeepAliveStrategy getKeepAliveStrategy() {
		return new DefaultConnectionKeepAliveStrategy() {
			@Override
			public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
				long duration = super.getKeepAliveDuration(response, context);
				if (duration == -1) {
					return KEEPALIVE_TIMEOUT;
				}
				return duration;
			}
		};
	}

}
