package cn.com.duiba.service.impl;

import cn.com.duiba.biz.credits.*;
import cn.com.duiba.credits.sdk.CreditNotifyParams;
import cn.com.duiba.credits.sdk.SignTool;
import cn.com.duiba.dao.ThirdpartyRelationDAO;
import cn.com.duiba.domain.AppDO;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.service.CustomService;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.CodeException;
import cn.com.duiba.tool.HttpRequestLog;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 接口自定义
 */
@Service
public class CustomServiceImpl implements CustomService {

    private static Logger log = LoggerFactory.getLogger(CustomServiceImpl.class);

    private static final List<Long> NOTIFY_WITH_TRANSFER = Arrays.asList(24202L, 1l);//24202:沪江开心词场

    /**
     * 平安健康的appID集合
     */
    private static final List<Long> pajkAppIds = Arrays.asList(27271L, 30703L, 31294L);

    private static final List<Long> mobikeIds=Arrays.asList(33888L,28651L,35830L);

    private static final List<Long> iqiyiIds=Arrays.asList(31819L,33713L);


    @Value("${mogujie.appId}")
    private Long mogujieAppId;

    @Value("${taojinbi.appId}")
    private Long taojinbiAppId;

    //凤金积分商城
    @Value("${fengjin.appId}")
    private String fengJinAppId;

    @Autowired
    private MogujieApi mogujieApi;
    @Autowired
    private PajkApi pajkApi;
    @Autowired
    private TaojinbiApi taojinbiApi;
    @Autowired
    private ThirdpartyRelationDAO thirdpartyRelationDAO;
    @Autowired
    private MobikeApi mobikeApi;

    @Autowired
    private FengJinApi fengJinApi;

    @Autowired
    private IqiyiApi iqiyiApi;

    public boolean isFengJin(Long appId){
        return fengJinAppId != null && Arrays.asList(fengJinAppId.replace(" ", "").split(",")).contains(appId + "");
    }

    @Override
    public boolean isMogujie(Long appId) {
        return mogujieAppId != null && mogujieAppId.equals(appId);
    }

    @Override
    public boolean isPajk(Long appId) {
        return appId != null && pajkAppIds.contains(appId);
    }

    @Override
    public boolean isTaojinbi(Long appId) {
        return taojinbiAppId != null && taojinbiAppId.equals(appId);
    }
    @Override
    public boolean isMobike(Long appId){
        return mobikeIds.contains(appId);
    }

    @Override
    public boolean isIqiyi(Long appId) {
        return appId != null && iqiyiIds.contains(appId);
    };

    @Override
    public CreditsMessage getRequestCredits(CreditsMessage request) {
        if (request.getAppId() == null) {
            return request;
        }
        Long appId = Long.valueOf(request.getAppId());
        if (isMogujie(appId)) {
            return mogujieApi.getMogujieRequestCredits(request);
        }
        if (isPajk(appId)) {
            return pajkApi.getPajkRequestCredits(request);
        }
        if (isTaojinbi(appId)) {
            return taojinbiApi.getCreditsMessage(request);
        }
        if(isMobike(appId)){
            return mobikeApi.buildSubCreditsRequest(request);
        }
        if (isFengJin(appId)) {
            return fengJinApi.getSubCreditsMessage(request);
        }
        if (isIqiyi(appId)) {
            return iqiyiApi.getSubCreditsMessage(request);
        }
        return request;
    }

    @Override
    public String getResponseCredits(String appId, String body) {
        if (appId == null) {
            return body;
        }
        Long appIdLong = Long.valueOf(appId);
        if (isMogujie(appIdLong)) {
            HttpRequestLog.logUrl("[action subCredits] [tag response] [appId " + appId + "] [type completed] [body " + body + "]");
            return mogujieApi.getMogujieResponseCredis(body);
        }
        if (isPajk(appIdLong)) {
            HttpRequestLog.logUrl("[action subCredits] [tag response] [appId " + appId + "] [type completed] [body " + body + "]");
            return pajkApi.getPajkResponseCredits(appIdLong, body);
        }
        if (isTaojinbi(appIdLong)) {
            HttpRequestLog.logUrl("[action subCredits] [tag response] [appId " + appId + "] [type completed] [body " + body + "]");
            return taojinbiApi.parseCreditsRsp(body);
        }
        return body;
    }

    @Override
    public String getRequestNotify(String notifyUrl, NotifyQueueDO record,String time, AppDO appDO) {
        try {
            if (isMogujie(record.getAppId())) {
                return mogujieApi.getMogujieRequestNotify(notifyUrl, record);
            }
            if (isPajk(record.getAppId())) {
                return pajkApi.getPajkRequestNotify(notifyUrl, record);
            }
            if(isMobike(record.getAppId())){
                return mobikeApi.getMobikeNotifyUrl(notifyUrl,record,time);
            }
            if (isFengJin(record.getAppId())) {
                return fengJinApi.getRequestNotify(notifyUrl, record, appDO);
            }
            if (isIqiyi(record.getAppId())) {
                return iqiyiApi.getRequestNotify(notifyUrl, record);
            }
        } catch (Exception e) {
            log.error("getRequestNotify", e);
        }
        return null;
    }

    @Override
    public String getResponseNotify(Long appId, String body) {
        try {
            if (isMogujie(appId)) {
                HttpRequestLog.logUrl("[action notify] [tag response] [appId " + appId + "] [type completed] [body " + body + "]");
                return mogujieApi.getMogujieResponseNotify(body);
            }
            if (isPajk(appId)) {
                HttpRequestLog.logUrl("[action notify] [tag response] [appId " + appId + "] [type completed] [body " + body + "]");
                return pajkApi.getPajkResponseNotify(body);
            }
        } catch (Exception e) {
            log.error("getResponseNotify", e);
        }
        return body;
    }

    @Override
    public Long getLongOrderNum(String duibaOrderNum) throws CodeException {
        if (StringUtils.isEmpty(duibaOrderNum)) {
            throw new CodeException("0", "订单号不能为空");
        }
        Long lid = thirdpartyRelationDAO.findByOrderNum(duibaOrderNum);
        if (lid == null) {
            try {
                return thirdpartyRelationDAO.insert(duibaOrderNum);
            } catch (DuplicateKeyException e) {
                log.warn("thirdpartyRelationDAO.insert DuplicateKeyException", e);
                lid = thirdpartyRelationDAO.findByOrderNum(duibaOrderNum);
            }
        }
        return lid;
    }

    @Override
    public boolean isNotifyWithTransfer(Long appId) {
        return NOTIFY_WITH_TRANSFER.contains(appId);
    }

    @Override
    public Map<String, String> toRequestMapWithTransfer(String appSecret, CreditNotifyParams p, String transfer, String uid) {//uid没有get方法
        HashMap map = new HashMap();
        map.put("success", Boolean.toString(p.isSuccess()));
        map.put("errorMessage", getString(p.getErrorMessage()));
        map.put("bizId", getString(p.getBizId()));
        map.put("appKey", getString(p.getAppKey()));
        map.put("appSecret", getString(appSecret));
        map.put("timestamp", getString(Long.valueOf(p.getTimestamp().getTime())));
        map.put("uid", getString(uid));
        map.put("orderNum", getString(p.getOrderNum()));
        map.put("transfer", getString(transfer));
        String sign = SignTool.sign(map);
        map.remove("appSecret");
        map.put("sign", sign);
        return map;
    }

    private String getString(Object o) {
        return o == null ? "" : o.toString();
    }

	@Override
	public CreditsMessageDto getRequestAddCredits(CreditsMessageDto creditsMessageDto) {
        if (creditsMessageDto.getAppId() == null) {
            return creditsMessageDto;
        }

        if (StringUtils.isNotBlank(creditsMessageDto.getAppId()) && isFengJin(Long.valueOf(creditsMessageDto.getAppId()))) {
            return fengJinApi.getAddCreditsMessageDto(creditsMessageDto);
        }
        return creditsMessageDto;
	}

	@Override
	public SupplierRequest getVirturalRequest(SupplierRequest request) {
		if(request.getAppId() == null){
			return request;
		}
		//如果是蘑菇街的魔豆
        Long appId=Long.valueOf(request.getAppId());
		if(isMogujie(appId)){
			return mogujieApi.getMogujieRequestSendCredits(request);
		}
		if(isMobike(appId)){
            return mobikeApi.getVirtualRequest(request);
        }
		return request;
	}

	@Override
	public String getVirturalResponse(SupplierRequest message, String body) {
		if(message.getAppId() == null){
			return body;
		}
		//蘑菇街魔豆,以后有扩展还需要进一步划分
		if(isMogujie(Long.valueOf(message.getAppId()))){
			return mogujieApi.getVirtualRespone(body);
		}
		return body;
	}


}
