package cn.com.duiba.notifycenter.service.impl;

import cn.com.duiba.notifycenter.dao.NotifyQueueDAO;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.notifycenter.service.BussinessTypesService;
import cn.com.duiba.notifycenter.service.NotifyHttpClientPool;
import cn.com.duiba.notifycenter.service.NotifyService;
import cn.com.duiba.service.CustomService;
import cn.com.duiba.thirdparty.dto.NotifyQueueDto;
import cn.com.duiba.thirdparty.enums.NotifyTypeEnum;
import cn.com.duiba.tool.CaiNiaoTool;
import cn.com.duiba.tool.HttpRequestLog;
import com.alibaba.fastjson.JSON;
import com.dangdang.ddframe.job.api.JobExecutionMultipleShardingContext;
import com.dangdang.ddframe.job.plugin.job.type.simple.AbstractSimpleElasticJob;
import org.apache.commons.lang.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.entity.GzipDecompressingEntity;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 通知服务
 */
@Service("notifyService")
public class NotifyServiceImpl extends AbstractSimpleElasticJob implements NotifyService {

	private static Logger log = LoggerFactory.getLogger(NotifyServiceImpl.class);

	@Autowired
	private NotifyHttpClientPool notifyHttpClientPool;
	@Autowired
	private NotifyQueueDAO notifyQueueDAO;
	@Autowired
	private BussinessTypesService bussinessTypesService;
	@Autowired
	private CustomService customService;

	@Value("${notifycenter.timer.enable}")
	private boolean timerEnable;

	private static Map<Integer, Long> timeMap = new HashMap<>();
	static {
		timeMap.put(0, 2 * 60 * 1000L);
		timeMap.put(1, 10 * 60 * 1000L);
		timeMap.put(2, 10 * 60 * 1000L);
		timeMap.put(3, 1 * 60 * 60 * 1000L);
		timeMap.put(4, 2 * 60 * 60 * 1000L);
		timeMap.put(5, 6 * 60 * 60 * 1000L);
		timeMap.put(6, 15 * 60 * 60 * 1000L);
	}

	/**
	 * 定时扫描每分钟执行次方法<br/>
	 * 0 0/1 * * * ?
	 */
	@Override
	public void process(JobExecutionMultipleShardingContext shardingContext) {
		scan();
	}

	@Override
	public void scan() {
		if (!timerEnable) {
			return;
		}
		long start = System.currentTimeMillis();
		List<NotifyQueueDO> list = notifyQueueDAO.findNeedNotifyList();
		int index = 0;
		for (NotifyQueueDO queue : list) {
			if(index++ % 100 == 0){
				try{
					TimeUnit.MILLISECONDS.sleep(1000);
				}catch (InterruptedException e){
					//ignore
				}
			}
			notify(queue, "scan");
		}
		long end = System.currentTimeMillis();
		log.info("notify scan size: {}, time: {} ms", list.size(), (end - start));
	}

	@Override
	public void notify(NotifyQueueDO queue, String fromSource) {
		try {
			FutureCallback callback =  getFutureCallback(queue);
			HttpUriRequest request = bussinessTypesService.getRequest(queue);
			if (request != null) {
				NotifyTypeEnum notify = NotifyTypeEnum.getByCode(queue.getNotifyType());
				HttpRequestLog.logUrl("[action notify]] [notifyType = "+notify.getDescription()+"] [tag request] [bizId " + queue.getDuibaOrderNum() + "] [url " + String.valueOf(request.getURI()) + "] [fromSource " + fromSource + "]");
				notifyHttpClientPool.submit(queue.getAppId(), request, callback);
			} else {
				log.warn("notifyQueueId={}, can't gen notify url,delete. {}" , queue.getId(), JSON.toJSONString(queue));
				notifyQueueDAO.finish(queue.getId());
			}
		} catch (Exception e) {
			notifyQueueDAO.finish(queue.getId());
			log.error("notifyQueueId=" + queue.getId() + ",submit error:" + JSON.toJSONString(queue), e);
		}
	}

	@Override
	public Date getNextTime(NotifyQueueDO notifyQueue) {
		return new Date(System.currentTimeMillis() + timeMap.get(notifyQueue.getTimes()));
	}

	/**
	 * 根据relationType获取不同的Callback
	 * @param queue
	 * @return
	 */
	private FutureCallback<HttpResponse> getFutureCallback(NotifyQueueDO queue){
		NotifyFutureCallback callback;
		if(NotifyQueueDto.RT_RE_SIGN_CARD.equals(queue.getRelationType())){
			callback = new NotifyFutureCallback(queue);
		}else {
			callback = new CreditsNotifyFutureCallback(queue);
		}
		return callback;
	}

	/**
	 * 通用通知回调
	 */
	class NotifyFutureCallback implements FutureCallback<HttpResponse> {

		protected NotifyQueueDO notifyQueue;

		/**
		 * 构造方法
		 * @param notifyQueue
		 */
		public NotifyFutureCallback(NotifyQueueDO notifyQueue) {
			this.notifyQueue = notifyQueue;
		}

		@Override
		public void completed(HttpResponse response) {
			String body = null;
			//兑换成功的返回结果有些开发者包含双引号，这里处理一下
			//符合条件的： {status: ok} 或 {"status":"ok"} 或 {"result":"ok"} 或 {"success":"true","errorMessage":"","bizId":"1943088"} 或 {"status":"ok","errorMessage":""} 或 okok
			String rulePattern = "([:]|\\b)[\"|\\s]{0,1}(OK|ok|true)[\"]{0,1}";
			NotifyTypeEnum notify = NotifyTypeEnum.getByCode(notifyQueue.getNotifyType());
			try {
				Header header = response.getEntity().getContentEncoding();
				String entity;
				if(header != null && header.toString().contains(CaiNiaoTool.CONTENT_ENCODING_GZIP)){
					entity = EntityUtils.toString(new GzipDecompressingEntity(response.getEntity()), CaiNiaoTool.CHARSET_UTF8);
				}else{
					entity = EntityUtils.toString(response.getEntity(), CaiNiaoTool.CHARSET_UTF8);
				}
				body = convertBody4Dcustom(entity);
				if (body != null) {
					body = body.trim();
				}

				if(StringUtils.isBlank(body) || body.length() > 500){
					fail();
					return;
				}

				Pattern pattern = Pattern.compile(rulePattern);
				Matcher matcher = pattern.matcher(body);
				if (matcher.find( )) {
					success();
				} else {
					//排除掉很长的body，避免浪费磁盘空间
					log.info("[appId {}] [type failed] [notifyType = {}] [body= {}]", notifyQueue.getAppId(),notify.getDescription(), body);
					fail();
				}
			} catch (Exception e) {
				log.error("completed", e);
				fail();
			} finally {
				HttpRequestLog.logUrl("[action notify] [notifyType = "+notify.getDescription()+"] [tag response] [bizId " + notifyQueue.getDuibaOrderNum() + "] [type completed] [body " + body + "]");
			}
		}

		@Override
		public void failed(Exception ex) {
			NotifyTypeEnum notify = NotifyTypeEnum.getByCode(notifyQueue.getNotifyType());
			HttpRequestLog.logUrl("[action notify] [notifyType = "+notify.getDescription()+"] [tag response] [bizId " + notifyQueue.getDuibaOrderNum() + "] [type failed] [ex " + ex.getMessage() + "]");
			fail();
		}

		@Override
		public void cancelled() {
			NotifyTypeEnum notify = NotifyTypeEnum.getByCode(notifyQueue.getNotifyType());
			HttpRequestLog.logUrl("[action notify] [notifyType = "+notify.getDescription()+"] [tag response] [type cancelled] [bizId " + notifyQueue.getDuibaOrderNum() + "]");
			fail();
		}

		/**
		 * 通知成功，删除待通知记录
		 */
		protected void success() {
			notifyQueueDAO.finish(notifyQueue.getId());
		}

		/**
		 * 通知失败，更新下次通知时间
		 */
		protected void fail() {
			Calendar cal = Calendar.getInstance();
			cal.setTime(new Date());
			cal.add(Calendar.DATE, -1);
			if (notifyQueue.getTimes() > 6) {
				notifyQueueDAO.finish(notifyQueue.getId());
				log.info("appId: {},relationType={},relationId={}, notify:{} fail,drop", notifyQueue.getAppId(), notifyQueue.getRelationType(), notifyQueue.getRelationId(), notifyQueue.getTimes());
			} else if (notifyQueue.getTimes() > 1 && notifyQueue.getGmtCreate().before(cal.getTime())) {
				notifyQueueDAO.finish(notifyQueue.getId());
				log.info("appId: {},relationType={},relationId={} notify > 24h,drop",notifyQueue.getAppId(), notifyQueue.getRelationType(), notifyQueue.getRelationId());
			} else {
				Date nexttime = getNextTime(notifyQueue);
				notifyQueueDAO.updateNextTime(notifyQueue.getId(), notifyQueue.getTimes() + 1, nexttime);
			}
		}

		/**
		 * 子类可以重载以适配不同的开发者响应格式
		 * @param entity
		 * @return
		 */
		protected String convertBody4Dcustom(String entity){
			return entity;
		}

	}

	/**
	 * 订单通知HTTP请求回调类
	 */
	class CreditsNotifyFutureCallback extends NotifyFutureCallback {

		/**
		 * 构造方法
		 * @param notifyQueue
		 */
		public CreditsNotifyFutureCallback(NotifyQueueDO notifyQueue) {
			super(notifyQueue);
		}

		@Override
		protected String convertBody4Dcustom(String entity){
			return customService.getResponseNotify(notifyQueue.getAppId(), entity);
		}
	}

}
