package cn.com.duiba.service;

import cn.com.duiba.tool.UrlUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * HTTP异步请求池
 */
@Service
public class HttpAsyncClientPool implements InitializingBean, DisposableBean {

	private static final Logger log = LoggerFactory.getLogger(HttpAsyncClientPool.class);

	/**
	 * APP最大处理队列
	 */
	public static final int MAX_APP_QUEUE = 500;

	private CloseableHttpAsyncClient httpClient;

	private Map<String, AtomicInteger> runningStat = new ConcurrentHashMap<>();

	@Autowired
	private ThreadPoolService threadPoolService;
	@Autowired
	private SlowRequestHandler slowRequestHandler;

	private static final int MAX_QUEUE_SIZE = 50000;
	private final BlockingQueue<CallbackProcesser> latterQueue = new LinkedBlockingQueue<>(MAX_QUEUE_SIZE);
	private final ExecutorService executorService = Executors.newSingleThreadExecutor(new ThreadFactory() {
		@Override
		public Thread newThread(Runnable r) {
			return new Thread(r, "LatterQueueProcess");
		}
	});
	private volatile boolean status = true;

	/**
	 * 执行HTTP异步请求
	 *
	 * @param queueKey
	 * @param request
	 * @param callback
	 */
	public void execute(String queueKey, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
		try {
			long s = System.currentTimeMillis();
			AtomicInteger running = runningStat.get(queueKey);
			if (running == null) {
				runningStat.put(queueKey, new AtomicInteger(1));
			} else {
				running.incrementAndGet();
			}
			request.setHeader("User-Agent","Mozilla/5.0 (iPhone; CPU iPhone OS 9_1 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Version/9.0 Mobile/13B143 Safari/601.1");

			CallbackProcesser process = new CallbackProcesser(queueKey, request, callback);

			String reqUrl = UrlUtils.buildUrl(request.getURI());
			if(slowRequestHandler.isBlackHost(reqUrl)){
				log.info("{} is black host", reqUrl);
				return;
			}
			if (slowRequestHandler.isSlow(reqUrl)) {
				log.info("url [{}], add to queue, appId={}", reqUrl, queueKey);
				slowRequestHandler.addQueue(process);
				return;
			}
			httpClient.execute(request, process);
			long e = System.currentTimeMillis();
			long t = e - s;
			if (t > 1000) {
				log.warn("http get pool time > {} ms for {}, appId={}", t, request.getURI(), queueKey);
				if (t>2000) {
					slowRequestHandler.addSlow(reqUrl, (int)(t/1000L));
				}
			}
		} catch (Exception e) {
			callback.failed(e);
			log.error("execute:", e);
		}
	}

	/**
	 * 提交HTTP异步请求
	 *
	 * @param queueKey
	 * @param request
	 * @param callback
	 */
	public void submit(final String queueKey, final HttpUriRequest request, final FutureCallback<HttpResponse> callback) {
		if (threadPoolService.canSubmit()) {
			threadPoolService.submit(new Runnable() {
				@Override
				public void run() {
					execute(queueKey, request, callback);
				}
			});
		} else {
			try {
				latterQueue.add(new CallbackProcesser(queueKey, request, callback));
			} catch (IllegalStateException e) {
				// 队列满了,任务取消
				log.warn("latterQueue is temporarily full",e);
				callback.cancelled();
			}

		}
	}

	/**
	 * 判断APP能否提交任务到执行队列中
	 *
	 * @param queueKey
	 * @return
	 */
	public boolean canSubmitToAppPool(String queueKey) {
		int queueCount = 0;
		AtomicInteger running = runningStat.get(queueKey);
		if (running != null) {
			queueCount = running.intValue();
		}
		if (queueCount < MAX_APP_QUEUE) {
			return true;
		}
		return false;
	}

	/**
	 * 队列详细信息
	 *
	 * @return
	 */
	public Map<String, Object> dumpDetail() {
		Map<String, Object> map = new HashMap<>();
		for (Map.Entry<String, AtomicInteger> entry : runningStat.entrySet()) {
			if (entry.getValue().intValue() > 0) {
				map.put(entry.getKey(), entry.getValue());
			}
		}
		Map<String, Object> ret = new HashMap<>();
		ret.put("runningStat", map);
		return ret;
	}

	/**
	 * HTTP处理中的个数
	 *
	 * @return
	 */
	public int dumpSize() {
		int running = 0;
		for (AtomicInteger a : runningStat.values()) {
			running += a.get();
		}
		return running;
	}

	/**
	 * 自定义回调
	 */
	public class CallbackProcesser implements FutureCallback<HttpResponse> {

		private FutureCallback<HttpResponse> callback;
		private String queueKey;
		private HttpUriRequest request;

		public CallbackProcesser(String queueKey, HttpUriRequest request, FutureCallback<HttpResponse> callback) {
			this.callback = callback;
			this.queueKey = queueKey;
			this.request = request;
		}

		@Override
		public void completed(HttpResponse result) {
			try {
				callback.completed(result);
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

		@Override
		public void failed(Exception ex) {
			try {
				callback.failed(ex);
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

		@Override
		public void cancelled() {
			try {
				callback.cancelled();
			} finally {
				runningStat.get(queueKey).decrementAndGet();
			}
		}

		public HttpUriRequest getRequest() {
			return request;
		}

		public String getQueueKey() {
			return queueKey;
		}
	}

	/**
	 * 销毁关闭
	 */
	@Override
	public void destroy() throws Exception {
		status = false;
		executorService.shutdown();
		if (httpClient != null && httpClient.isRunning()) {
			httpClient.close();
			log.info("HttpAsyncClientPool closed");
		}
	}

	/**
	 * 初始化启动
	 */
	@Override
		public void afterPropertiesSet() throws Exception {
		start();
		executorService.execute(new Runnable() {
			@Override
			public void run() {
				while (status) {
					try{
						if (!threadPoolService.canSubmit()) {
							TimeUnit.MILLISECONDS.sleep(200);
							continue;
						}
						CallbackProcesser callbackProcesser =latterQueue.take();
						if (callbackProcesser == null) {
							continue;
						}
						final String queueKey = callbackProcesser.getQueueKey();
						final HttpUriRequest request = callbackProcesser.getRequest();
						final FutureCallback<HttpResponse> callback = callbackProcesser.callback;
						threadPoolService.submit(new Runnable() {
							@Override
							public void run() {
								execute(queueKey, request, callback);
							}
						});
					}catch (InterruptedException e) {
						log.warn("Interrupted",e);
					}catch (Exception e) {
						log.warn("",e);
					}
				}
			}
		});
	}

	/**
	 * 启动HTTP服务
	 */
	private synchronized void start() {
		if (httpClient != null && httpClient.isRunning()) {
			return;
		}
		httpClient = HttpClientFactory.newCloseableHttpAsyncClient();
		log.info("HttpAsyncClientPool started");
	}

}
