package cn.com.duiba.mq;

import cn.com.duiba.boot.profiler.DBTimeProfiler;
import cn.com.duiba.wolf.utils.GZIPUtils;
import com.alibaba.ttl.threadpool.TtlExecutors;
import org.apache.commons.lang.StringUtils;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.client.producer.SendStatus;
import org.apache.rocketmq.common.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

/**
 * Created by zzy on 2017/7/17.
 */
@Service
public class RocketMQMsgProducer {
    private static final Logger LOG = LoggerFactory.getLogger(RocketMQMsgProducer.class);
    @Autowired
    private DefaultMQProducer rocketMqProducer;
    private static final int MAX_RETRY_TIMES = 3;

    @Autowired
    private ExecutorService executorService;

    // 专门用来发消息和重试的线程池
    private final ExecutorService mqExecutorService = TtlExecutors.getTtlExecutorService(Executors.newFixedThreadPool(4, new ThreadFactory() {
        private int index = 0;

        @Override
        public Thread newThread(Runnable r) {
            return new Thread(r, "rocketmq-producer-" + index++);
        }
    }));

    /**
     * 发送消息rocketmq
     *
     * @param topic   队列名单
     * @param tag     标签
     * @param key     消息key
     * @param message 消息
     * @return
     * @see <a>http://cf.dui88.com/pages/viewpage.action?pageId=5256828</a>
     */
    @DBTimeProfiler
    public void sendMsg(String topic, String tag, String key, String message, boolean useGzip, MqSentCallback callback) {
        MqResult ret = doSendMsg(topic, tag, key, message, useGzip, callback);
        // 失败
        if (!ret.success && ret.msg != null) {
            mqExecutorService.execute(() -> {
                retry(ret);
            });
        } else {
            if (callback == null) {
                return;
            }
            executorService.execute(() -> {
                try {
                    callback.onSuccess();
                } catch (Exception e) {
                    LOG.error("", e);
                }
            });
        }
    }

    @DBTimeProfiler
    private MqResult doSendMsg(String topic, String tag, String key, String message, boolean useGzip, MqSentCallback callback) {
        Message msg = null;
        try {
            byte[] body = useGzip ? GZIPUtils.gzip(message) : message.getBytes("utf-8");
            msg = new Message(topic, tag, body);
            if (StringUtils.isNotEmpty(key)) {
                msg.setKeys(key);
            }
            SendResult ret = rocketMqProducer.send(msg);
            boolean success = SendStatus.SEND_OK == ret.getSendStatus();
            MqResult mqResult = new MqResult();
            mqResult.success = success;
            mqResult.msg = msg;
            mqResult.callback = callback;
            return mqResult;
        } catch (Exception e) {
            LOG.error("Send mq failed, topic={}, tag={}, key={}, msg={}", topic, tag, key, message, e);
            MqResult mqResult = new MqResult();
            mqResult.success = false;
            mqResult.msg = msg;
            mqResult.callback = callback;
            return mqResult;
        }
    }

    /**
     * 重试逻辑
     *
     * @param mqResult
     */
    @DBTimeProfiler
    void retry(MqResult mqResult) {
        mqResult.retryTimes++;
        try {
            // 超过重试次数，按失败处理
            if (mqResult.retryTimes > MAX_RETRY_TIMES && mqResult.callback != null) {
                mqResult.callback.onFail();
                return;
            }
            // 线程休眠时间第一次重试200ms之后，第二次400ms，第三次600ms
            TimeUnit.MILLISECONDS.sleep(200 * mqResult.retryTimes);
            SendResult sendResult = rocketMqProducer.send(mqResult.msg);
            if (SendStatus.SEND_OK == sendResult.getSendStatus() && mqResult.callback != null) {
                mqResult.callback.onSuccess();
            } else {
                retry(mqResult);
            }
        } catch (InterruptedException e) {
            LOG.warn("", e);
        } catch (Exception e) {
            LOG.warn("", e);
            retry(mqResult);
        }
    }

    private class MqResult {
        private boolean success;
        private Message msg;
        private int retryTimes = 0;
        private MqSentCallback callback;
    }

}
