package cn.com.duiba.service;

import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * Created by zzy on 2017/11/20.
 */
@Configuration
public class HttpClientFactory {
    private static final Logger log = LoggerFactory.getLogger(HttpClientFactory.class);

    /**
     * 连接超时
     */
    public static final int CONNECT_TIMEOUT = 30 * 1000;

    //从连接池中获取连接的超时时间
    public static final int DEFAULT_REQUEST_TIMEOUT = 3000;

    /**
     * 长链接空闲时间
     */
    public static final int KEEPALIVE_TIMEOUT = 5 * 1000;

    /**
     * 处理超时
     */
    public static final int SOCKET_TIMEOUT = 16 * 1000;

    /**
     * 最大总连接值
     */
    public static final int MAX_CONNECT = 6000;

    /**
     * 每个路由最大连接{并发}值
     */
    public static final int MAX_ROUTE_CONNECT = 500;

    /**
     * 通知连接超时
     */
    public static final int NOTIFY_CONNECT_TIMEOUT = 10 * 1000;

    /**
     * 通知长链接空闲时间
     */
    public static final int NOTIFY_KEEPALIVE_TIMEOUT = 5 * 1000;

    /**
     * 通知长链接空闲时间
     */
    public static final int NOTIFY_SOCKET_TIMEOUT = 5 * 1000;

    @Bean(initMethod="start", destroyMethod = "close", name="httpAsyncClient")
    public CloseableHttpAsyncClient httpAsyncClient(){
        RequestConfig config = RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom().setDefaultRequestConfig(config).setMaxConnTotal(MAX_CONNECT).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy(false)).build();
        log.info("HttpAsyncClientPool started");
        return httpClient;
    }

    /**
     * 慢请求单独的http异步连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="slowHttpAsyncClient")
    public CloseableHttpAsyncClient slowHttpAsyncClient(){
        RequestConfig config = RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom().setDefaultRequestConfig(config).setMaxConnTotal(1000).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy(false)).build();
        log.info("slowHttpAsyncClient started");
        return httpClient;
    }

    /**
     * 虚拟商品单独的http异步连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="virtualHttpAsyncClient")
    public CloseableHttpAsyncClient virtualHttpAsyncClient(){
        RequestConfig config = RequestConfig.custom().setConnectTimeout(CONNECT_TIMEOUT).setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT).setSocketTimeout(SOCKET_TIMEOUT).build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom().setDefaultRequestConfig(config).setMaxConnTotal(3000).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy(false)).build();
        log.info("virtualHttpAsyncClient started");
        return httpClient;
    }

    /**
     * 通知连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="httpAsyncNotifyClient")
    public CloseableHttpAsyncClient httpAsyncNotifyClient(){
        RequestConfig config = RequestConfig.custom().setConnectTimeout(NOTIFY_CONNECT_TIMEOUT).setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT).setSocketTimeout(NOTIFY_SOCKET_TIMEOUT).build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom().setDefaultRequestConfig(config).setMaxConnTotal(MAX_CONNECT).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy(true)).build();
        log.info("httpAsyncNotifyClient started");
        return httpClient;
    }

    /**
     * 慢通知单独的连接池
     * @return
     */
    @Bean(initMethod="start", destroyMethod = "close", name="slowHttpAsyncNotifyClient")
    public CloseableHttpAsyncClient slowHttpAsyncNotifyClient(){
        RequestConfig config = RequestConfig.custom().setConnectTimeout(NOTIFY_CONNECT_TIMEOUT).setConnectionRequestTimeout(DEFAULT_REQUEST_TIMEOUT).setSocketTimeout(NOTIFY_SOCKET_TIMEOUT).build();
        CloseableHttpAsyncClient httpClient = HttpAsyncClients.custom().setDefaultRequestConfig(config).setMaxConnTotal(1000).setMaxConnPerRoute(MAX_ROUTE_CONNECT).setKeepAliveStrategy(getKeepAliveStrategy(true)).build();
        log.info("slowHttpAsyncNotifyClient started");
        return httpClient;
    }

    /**
     * 长连接
     *
     * @return
     */
    private static DefaultConnectionKeepAliveStrategy getKeepAliveStrategy(boolean isNotification) {
        return new DefaultConnectionKeepAliveStrategy() {
            @Override
            public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                long duration = super.getKeepAliveDuration(response, context);
                if (duration == -1) {
                    if(isNotification){
                        return NOTIFY_KEEPALIVE_TIMEOUT;
                    }else{
                        return KEEPALIVE_TIMEOUT;
                    }
                }
                return duration;
            }
        };
    }

}