package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.constant.ReconciliationConfig;
import cn.com.duiba.credits.sdk.CreditConsumeParams;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.service.reconciliation.ReconciliationRecordService;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.thirdparty.dto.reconciliation.ReconciliationRecordDto;
import cn.com.duiba.thirdparty.enums.reconciliation.ReconciliationLogTypeEnum;
import cn.com.duiba.thirdparty.enums.reconciliation.ReconciliationRespStatusEnum;
import cn.com.duiba.tool.AssembleTool;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;

/**
 * @program: thirdparty-all
 * @description: 对账定制api
 * @author: Simba
 * @create: 2020-02-18 15:49
 **/
@Service
public class ReconciliationBizService {

    private static final Logger logger = LoggerFactory.getLogger(ReconciliationBizService.class);

    @Autowired
    private ReconciliationConfig reconciliationConfig;

    @Resource(name = "httpCallbackExecutorService")
    private ExecutorService executorService;

    @Autowired
    private ReconciliationRecordService reconciliationRecordService;

    private boolean isReconcile(Long appId) {
    	if(appId == null || CollectionUtils.isEmpty(reconciliationConfig.getAppIdSet())){
    		return false;
		}

        return reconciliationConfig.getAppIdSet().contains(appId);
    }

    /**
     * 保存对账记录 入口是rpc
     * @param request
     * @return
     */
	public void saveRpcSubCreditsMsg(CreditsMessage request) {
        if(!isReconcile(Long.valueOf(request.getAppId()))){
            return;
        }

        executorService.execute(() -> {
            try{
                String url = request.getHttpUrl();
                String orderNum = request.getRelationId();

                Map<String, String> authParamMap;
                if(Objects.equals(request.getHttpType(), CreditsMessage.HTTP_POST)){
                    authParamMap = request.getAuthParams();
                    if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
                        orderNum = authParamMap.get("orderNum");
                    }
                }else{
                    String authParams = url.substring(url.indexOf('?') + 1);
                    authParamMap = AssembleTool.getUrlParams(authParams);
                }

                ReconciliationRecordDto dto = wrapReconciliationLog(Long.valueOf(request.getAppId()), orderNum, request.getRelationType(), authParamMap, request.getParams(), ReconciliationLogTypeEnum.SUB_CREDITS);
                reconciliationRecordService.saveOrUpdate(dto);
            }catch (Exception e){
                logger.error("save对账记录出错, appId:{}, relationId:{}, relationType:{}, ", request.getAppId(), request.getRelationId(), request.getRelationType(), e);
            }
        });
	}

    /**
     * 保存对账记录 入口是rocketmq
     * @param subCreditsMsgWrapper
     * @return
     */
    public void saveMqSubCreditsMsg(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        if(!isReconcile(subCreditsMsgWrapper.getSubCreditsMsg().getAppId())){
            return;
        }

        executorService.execute(() -> {
            try{
                ReconciliationRecordDto dto = wrapReconciliationLog(subCreditsMsgWrapper);
                reconciliationRecordService.saveOrUpdate(dto);
            }catch (Exception e){
                logger.error("save对账记录出错, appId:{}, relationId:{}, relationType:{}, ", subCreditsMsgWrapper.getSubCreditsMsg().getAppId(), subCreditsMsgWrapper.getSubCreditsMsg().getRelationId(), subCreditsMsgWrapper.getSubCreditsMsg().getRelationType(), e);
            }
        });
    }

    /**
     * 构造扣积分请求参数-入口是rocketmq
     * 加/扣积分：兑吧订单号、用户id、操作类型（加积分、扣积分）、时间、活动ID、状态（成功、失败）、积分值
     */
    private ReconciliationRecordDto wrapReconciliationLog(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        SubCreditsMsgDto subCreditsMsgDto = subCreditsMsgWrapper.getSubCreditsMsg();
        CreditConsumeParams creditConsumeParams = subCreditsMsgDto.getCreditConsumeParams();
	    Map<String, String> paramMap = subCreditsMsgDto.getParams();

        ReconciliationRecordDto reconciliationRecordDto = new ReconciliationRecordDto();
        reconciliationRecordDto.setOrderNum(subCreditsMsgDto.getRelationId());
        reconciliationRecordDto.setPartnerUserId(creditConsumeParams.getUid());
        reconciliationRecordDto.setLogType(ReconciliationLogTypeEnum.SUB_CREDITS.getCode());
        reconciliationRecordDto.setSendTime(new Date());
        if(MapUtils.isNotEmpty(paramMap)){
            reconciliationRecordDto.setOptId(paramMap.get("opId"));
        }

        if(subCreditsMsgDto.getRelationType() != null){
            reconciliationRecordDto.setOptType(subCreditsMsgDto.getRelationType().getMsg());
        }
        reconciliationRecordDto.setCredits(creditConsumeParams.getCredits());
        reconciliationRecordDto.setAppId(subCreditsMsgDto.getAppId());

        return reconciliationRecordDto;
    }

    /**
     *
     * 构造扣积分请求参数-入口是rpc
     * 加/扣积分：兑吧订单号、用户id、操作类型（加积分、扣积分）、时间、活动ID、状态（成功、失败）、积分值
     */
    private ReconciliationRecordDto wrapReconciliationLog(Long appId, String relationId, String relationType, Map<String, String> authParamMap, Map<String, String> paramMap, ReconciliationLogTypeEnum logTypeEnum) {
        ReconciliationRecordDto reconciliationRecordDto = new ReconciliationRecordDto();
        reconciliationRecordDto.setOrderNum(relationId);
        reconciliationRecordDto.setPartnerUserId(authParamMap.get("uid"));
        reconciliationRecordDto.setLogType(logTypeEnum.getCode());
        reconciliationRecordDto.setSendTime(new Date());

        if(MapUtils.isNotEmpty(paramMap)){
            reconciliationRecordDto.setOptId(paramMap.get("opId"));
        }

        if(StringUtils.isNotBlank(relationType)){
            reconciliationRecordDto.setOptType(relationType);
        }
        String credits = authParamMap.get("credits");
        if(StringUtils.isNotBlank(credits)){
            reconciliationRecordDto.setCredits(Long.valueOf(credits));
        }

        reconciliationRecordDto.setAppId(appId);

        return reconciliationRecordDto;
    }

    /**
     * 构造加积分请求
     */
    public void saveAddCreditsMsg(CreditsMessageDto request){
        if(!isReconcile(Long.valueOf(request.getAppId()))){
            return;
        }

        executorService.execute(() -> {
            try{
                String url = request.getHttpUrl();
                String orderNum = request.getRelationId();
                Map<String, String> authParamMap;
                if(Objects.equals(request.getHttpType(), CreditsMessage.HTTP_POST)){
                    authParamMap = request.getAuthParams();
                    if(MapUtils.isNotEmpty(authParamMap) && StringUtils.isNotBlank(authParamMap.get("orderNum"))){
                        orderNum = authParamMap.get("orderNum");
                    }
                }else{
                    String authParams = url.substring(url.indexOf('?') + 1);
                    authParamMap = AssembleTool.getUrlParams(authParams);
                }

                ReconciliationRecordDto dto = wrapReconciliationLog(Long.valueOf(request.getAppId()), orderNum, request.getRelationType(), authParamMap, request.getParams(), ReconciliationLogTypeEnum.ADD_CREDITS);
                reconciliationRecordService.saveOrUpdate(dto);
            }catch (Exception e){
                logger.error("save对账记录出错, appId:{}, relationId:{}, relationType:{}, ", request.getAppId(), request.getRelationId(), request.getRelationType(), e);
            }
        });
    }

    public void mallExchangeNotify(ReconciliationRecordDto dto){
        if(!isReconcile(dto.getAppId())){
            return;
        }
        reconciliationRecordService.saveOrUpdate(dto);
    }

    /**
     * 解析积分响应
     * @param appId
     * @param body
     * @param addCredits
     * @param relationId
     * @param relationType
     * @return
     */
    public void updateCreditsRsp(Long appId, String body, Boolean addCredits, String relationId, String relationType) {
        if(!isReconcile(appId)){
            return;
        }

        executorService.execute(() -> {
            try{
                ReconciliationRecordDto reconciliationRecordDto = new ReconciliationRecordDto();
                reconciliationRecordDto.setOrderNum(relationId);
                reconciliationRecordDto.setLogType(ReconciliationLogTypeEnum.SUB_CREDITS.getCode());
                reconciliationRecordDto.setAppId(appId);
                reconciliationRecordDto.setOptType(relationType);
                if(addCredits){
                    reconciliationRecordDto.setLogType(ReconciliationLogTypeEnum.ADD_CREDITS.getCode());
                }
                parseRespBody(body, reconciliationRecordDto);

                reconciliationRecordService.update(reconciliationRecordDto);
            }catch (Exception e){
                logger.error("update响应结果出错, appId:{}, relationId:{}, relationType:{}, body:{}, ", appId, relationId, relationType, body, e);
            }
        });
    }

    private void parseRespBody(String body, ReconciliationRecordDto dto){
        String respBody = body;
        if(StringUtils.isNotBlank(body)){
            dto.setResponseStatus(ReconciliationRespStatusEnum.FAIL.getCode());
            JSONObject json = JSON.parseObject(body);
            if (json != null && ("success".equalsIgnoreCase(json.getString("status")) || "ok".equalsIgnoreCase(json.getString("status")))) {
                dto.setResponseStatus(ReconciliationRespStatusEnum.SUCCESS.getCode());
            }

            if(respBody.length() > 255){
                respBody = respBody.substring(0, 255);
            }
            dto.setResponseBody(respBody);
        }
    }
}

