package cn.com.duiba.biz.credits;

import cn.com.duiba.constant.HuaweiConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.huawei.SignUtils;
import cn.com.duiba.tool.huawei.UrlUtils;
import cn.com.duiba.wolf.utils.UrlUtils2;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.http.Header;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

/**
 * 华为商城定制api
 * Created by xuwei on 2020/12/02.
 **/
@Service
public class HuaweiApi {

    private static final Logger LOG = LoggerFactory.getLogger(HuaweiApi.class);

    private static final String VIRTUAL_TYPE_KEY = "virtualType";

    @Autowired
    private HuaweiConfig huaweiConfig;

    public boolean isHuaweiApp(Long appId) {
        return huaweiConfig.isHuaweiApp(appId);
    }

    public HttpRequestBase getSubCreditsHttpRequest(CreditsMessage message) {
        try {
            String url = UrlUtils2.extractUrl(message.getHttpUrl());
            Map<String, String> requestParams = buildConsumeCreditsParams(message.getHttpUrl());
            HttpRequestBase result = buildJsonRequest(url, requestParams);
            message.setHttpUrl(url);
            message.setAuthParams(requestParams);
            LOG.info("huawei 扣积分请求\n\nurl:\n{}\n\nparams:\n{}\n\nheaders:\n{}", message.getHttpUrl(), JSON.toJSONString(requestParams), JSON.toJSONString(headerMap(result.getAllHeaders())));
            return result;
        } catch (Exception e) {
            LOG.error("huawei 创建扣积分请求异常 message={}", JSON.toJSONString(message), e);
            throw new IllegalStateException(e);
        }
    }

    public HttpRequestBase getMqSubCreditsHttpRequest(SubCreditsMsgWrapper message) {
        try {
            String url = UrlUtils2.extractUrl(message.getHttpUrl());
            Map<String, String> requestParams = buildConsumeCreditsParams(message.getHttpUrl());
            HttpRequestBase result = buildJsonRequest(url, requestParams);
            message.setHttpUrl(url);
            message.getSubCreditsMsg().setAuthParams(requestParams);
            LOG.info("huawei 扣积分请求\n\nurl:\n{}\n\nparams:\n{}\n\nheaders:\n{}", message.getHttpUrl(), JSON.toJSONString(requestParams), JSON.toJSONString(headerMap(result.getAllHeaders())));
            return result;
        } catch (Exception e) {
            LOG.error("huawei 创建扣积分请求异常 message={}", JSON.toJSONString(message), e);
            throw new IllegalStateException(e);
        }
    }

    public HttpRequestBase getAddCreditsMessageRequest(CreditsMessageDto message) {
        try {
            String url = UrlUtils2.extractUrl(message.getHttpUrl());
            Map<String, String> requestParams = buildAddCreditsParams(message.getHttpUrl());
            HttpRequestBase result = buildJsonRequest(url, requestParams);
            message.setHttpUrl(url);
            message.setAuthParams(requestParams);
            LOG.info("huawei 加积分请求\n\nurl:\n{}\n\nparams:\n{}\n\nheaders:\n{}", message.getHttpUrl(), JSON.toJSONString(requestParams), JSON.toJSONString(headerMap(result.getAllHeaders())));
            return result;
        } catch (Exception e) {
            LOG.error("huawei 创建加积分请求异常 message={}", JSON.toJSONString(message), e);
            throw new IllegalStateException(e);
        }
    }

    public String parseCreditsResponse(String body, Boolean addCredits) {
        LOG.info("huawei {}积分响应结果\n{}", addCredits ? "加" : "扣", body);
        JSONObject result = new JSONObject();
        try {
            JSONObject json = JSON.parseObject(body);
            if (json.getBooleanValue("success") && "0".equals(json.getString("code"))) {
                result.put("status", "ok");
                // 开发者不返回订单号，自定义
                result.put("bizId", System.currentTimeMillis() + RandomStringUtils.randomNumeric(6));
            } else {
                result.put("status", "fail");
                result.put("errorMessage", json.getString("msg"));
            }
        } catch (Exception e) {
            LOG.error("huawei 解析{}积分响应结果异常 response={}", addCredits ? "加":"扣", body, e);
            result.put("status", "fail");
            result.put("errorMessage", "解析响应结果异常");
        }
        return result.toJSONString();
    }

    public HttpRequestBase getVirtualRequest(SupplierRequest request) {
        try {
            Map<String, String> params = UrlUtils.extractUrlParamsFromUrl(request.getHttpUrl());
            // Pair<类型,值>
            Pair<String, String> thirdCode = parseThirdCode(params.get("params"));
            // 由于虚拟商品包含多种类型，每种类型的接口都不同，
            // 所以虚拟商品充值接口只配置一个域名(如: https://www.baidu.com)，
            // 实际路径根据类型从配置中读取，和域名进行拼接
            String domain = UrlUtils2.extractUrl(request.getHttpUrl());
            HttpRequestBase result = buildVirtualRequestByType(request, domain, params, thirdCode.getLeft(), thirdCode.getRight());
            LOG.info("huawei 虚拟商品充值请求\n\nurl:\n{}\n\nparams:\n{}\n\nheaders:\n{}", request.getHttpUrl(), JSON.toJSONString(request.getAuthParams()), JSON.toJSONString(headerMap(result.getAllHeaders())));
            return result;
        } catch (Exception e) {
            LOG.error("huawei 创建虚拟商品请求参数异常 request={}", JSON.toJSONString(request), e);
            throw new IllegalStateException(e);
        }

    }

    public String getVirtualResponse(SupplierRequest request, String body) {
        try {
            String type = request.getParams().get(VIRTUAL_TYPE_KEY);
            LOG.info("huawei 虚拟商品充值响应结果\n\ntype:\n{}\n\nresponse:\n{}", type, body);
            return parseVirtualResponseByType(type, body);
        } catch (Exception e) {
            JSONObject result = new JSONObject();
            LOG.error("huawei 解析虚拟商品响应结果异常 response={}", body, e);
            result.put("status", "fail");
            result.put("errorMessage", "解析响应结果异常");
            return result.toJSONString();
        }
    }

    private String parseVirtualResponseByType(String type, String body) {
        if (huaweiConfig.getVirtualTypeCredits().equals(type)) {
            return parseAddCreditsVirtualResponse(body);
        }
        // 领取优惠券（批次方式）
        if (huaweiConfig.getVirtualTypeCouponBatch().equals(type)) {
            return parseCouponBatchVirtualResponse(body);
        }
        // 领取优惠券（券码方式）
        if (huaweiConfig.getVirtualTypeCouponCode().equals(type)) {
            return parseCouponCodeVirtualResponse(body);
        }
        throw new UnsupportedOperationException("不支持该虚拟商品类型：" + type);
    }

    private String parseCouponCodeVirtualResponse(String body) {
        return parseCouponBatchVirtualResponse(body);
    }

    private String parseCouponBatchVirtualResponse(String body) {
        JSONObject result = new JSONObject();
        JSONObject json = JSON.parseObject(body);
        // code为空也表示成功
        String code = StringUtils.defaultIfBlank(json.getString("code"), "0");
        if (json.getBooleanValue("success") && "0".equals(code)) {
            result.put("status", "success");
            // 开发者不返回订单号，自定义
            result.put("supplierBizId", System.currentTimeMillis() + RandomStringUtils.randomNumeric(6));
        } else {
            result.put("status", "fail");
            result.put("errorMessage", StringUtils.defaultIfBlank(json.getString("errorTip"), json.getString("msg")));
        }
        return result.toJSONString();
    }

    private String parseAddCreditsVirtualResponse(String body) {
        JSONObject result = new JSONObject();
        JSONObject json = JSON.parseObject(body);
        if (json.getBooleanValue("success") && "0".equals(json.getString("code"))) {
            result.put("status", "success");
            // 开发者不返回订单号，自定义
            result.put("supplierBizId", System.currentTimeMillis() + RandomStringUtils.randomNumeric(6));
        } else {
            result.put("status", "fail");
            result.put("errorMessage", json.getString("msg"));
        }
        return result.toJSONString();
    }

    private HttpRequestBase buildVirtualRequestByType(SupplierRequest request, String domain, Map<String, String> params, String type, String value) throws Exception {
        // 将虚拟商品类型设置进虚拟商品扩展参数中，以便回调时判断类型解析
        markVirtualType(request, type);
        // 积分类型虚拟商品，执行加积分逻辑
        if (huaweiConfig.getVirtualTypeCredits().equals(type)) {
            return buildAddCreditsVirtualRequest(request, domain, params, value);
        }
        // 领取优惠券（批次方式）
        if (huaweiConfig.getVirtualTypeCouponBatch().equals(type)) {
            return buildCouponBatchVirtualRequest(request, domain, params, value);
        }
        // 领取优惠券（券码方式）
        if (huaweiConfig.getVirtualTypeCouponCode().equals(type)) {
            return buildCouponCodeVirtualRequest(request, domain, params, value);
        }
        throw new UnsupportedOperationException("不支持该虚拟商品类型：" + type);
    }

    private HttpRequestBase buildCouponCodeVirtualRequest(SupplierRequest request, String domain, Map<String, String> params, String value) throws Exception {
        Map<String, String> requestParams = buildCommonRequestParams(params);
        requestParams.put("busiCode", value);

        String url = domain + huaweiConfig.getReceiveCouponCodeUrl();
        HttpRequestBase result = buildJsonRequest(url, requestParams);
        request.setHttpUrl(url);
        request.setAuthParams(requestParams);
        return result;
    }

    private HttpRequestBase buildCouponBatchVirtualRequest(SupplierRequest request, String domain, Map<String, String> params, String value) throws Exception {
        Map<String, String> requestParams = buildCommonRequestParams(params);
        // 优惠券批次值格式：活动编码,批次编码
        String[] arr = StringUtils.split(value, ',');
        if (arr.length != 2) {
            throw new IllegalStateException("优惠券批次值配置有误");
        }
        requestParams.put("activityCode", arr[0].trim());
        requestParams.put("batchCode", arr[1].trim());
        requestParams.put("receiveChannel", "1");

        String url = domain + huaweiConfig.getReceiveCouponBatchUrl();
        HttpRequestBase result = buildJsonRequest(url, requestParams);
        request.setHttpUrl(url);
        request.setAuthParams(requestParams);
        return result;
    }

    /**
     * 优惠券通用请求参数
     * @param params
     * @return
     * @throws Exception
     */
    private Map<String, String> buildCommonRequestParams(Map<String, String> params) {
        Map<String, String> result = new HashMap<>();

        result.put("appId", huaweiConfig.getHuaweiAppId());
        result.put("openId", params.get("uid"));
        result.put("portal", "72");
        result.put("lang", "zh-CN");
        result.put("country", "CN");
//        requestParams.put("beCode", huaweiConfig.getBeCode());

        return result;
    }

    /**
     * 标记虚拟商品类型
     * @param request
     * @param type
     */
    private void markVirtualType(SupplierRequest request, String type) {
        Map<String, String> params = request.getParams();
        if (params == null) {
            params = new HashMap<>();
            request.setParams(params);
        }
        params.put(VIRTUAL_TYPE_KEY, type);
    }

    /**
     * 解析商家编码 编码格式 ： 类型-值
     * @param thirdCode
     * @return Pair<类型, 值>
     */
    private Pair<String, String> parseThirdCode(String thirdCode) {
        String[] arr = StringUtils.split(thirdCode, '-');
        if (arr.length != 2) {
            throw new IllegalStateException("虚拟商品商家编码格式错误");
        }
        return Pair.of(arr[0].trim().toUpperCase(), arr[1].trim());
    }

    /**
     * 积分类型虚拟商品加积分请求
     * @param request
     * @param domain
     * @param params
     * @param value
     * @return
     */
    private HttpRequestBase buildAddCreditsVirtualRequest(SupplierRequest request, String domain, Map<String, String> params, String value) throws Exception {
        params.put("credits", value);
        Map<String, String> requestParams = buildCommonCreditsParams(params, huaweiConfig.getAddCreditsActCode());
        String url = domain + huaweiConfig.getAddCreditsPath();
        HttpRequestBase result = buildJsonRequest(url, requestParams);
        request.setHttpUrl(url);
        request.setAuthParams(requestParams);
        return result;
    }

    /**
     * 加积分请求参数
     * @param httpUrl
     * @return
     * @throws Exception
     */
    private Map<String, String> buildAddCreditsParams(String httpUrl) throws Exception {
        return buildCommonCreditsParams(UrlUtils.extractUrlParamsFromUrl(httpUrl), huaweiConfig.getAddCreditsActCode());
    }

    /**
     * 扣积分请求参数
     * @param httpUrl
     * @return
     * @throws Exception
     */
    private Map<String, String> buildConsumeCreditsParams(String httpUrl) throws Exception {
        return buildCommonCreditsParams(UrlUtils.extractUrlParamsFromUrl(httpUrl), huaweiConfig.getConsumeCreditsActCode());
    }

    /**
     * 扣/加积分请求通用请求参数
     * @param params
     * @return
     * @throws Exception
     */
    private Map<String, String> buildCommonCreditsParams(Map<String, String> params, String actCode) throws Exception {
        Map<String, String> result = buildCommonRequestParams(params);

        String transId = generateTransId();
        result.put("transId", transId);
        result.put("pointValue", params.get("credits"));
        result.put("serviceUnit", huaweiConfig.getServiceUnit());
        result.put("actCode", actCode);
        result.put("timeStamp", params.get("timestamp"));
        result.put("actTime", generateActTime());

        TreeMap<String, String> signParams = new TreeMap<>();
        signParams.put("transId", transId);
        signParams.put("appId", huaweiConfig.getHuaweiAppId());
        signParams.put("openId", params.get("uid"));
        signParams.put("pointValue", params.get("credits"));
        signParams.put("serviceUnit", huaweiConfig.getServiceUnit());
        signParams.put("actCode", actCode);
        signParams.put("timeStamp", params.get("timestamp"));
        result.put("sign", sign(signParams));

        return result;
    }

    private HttpRequestBase buildJsonRequest(String url, Map<String, String> params) throws Exception {
        HttpPost post = new HttpPost(url);
        post.setHeaders(createCustomHeaders());
        post.setEntity(new StringEntity(JSON.toJSONString(params), ContentType.APPLICATION_JSON));
        return post;
    }

    /**
     * 接口签名，保证参数顺序传入
     * @param params
     * @return
     * @throws Exception
     */
    private String sign(TreeMap<String, String> params) throws Exception {
        String signStr = UrlUtils2.buildUrlParams(params);
        LOG.info("huawei 签名字符串 {}", signStr);
        return SignUtils.sign(signStr, huaweiConfig.getPrivateKeyInstance());
    }

    public Map<String, String> headerMap(Header[] headers) {
        Map<String, String> map = new HashMap<>();
        for (Header header : headers) {
            map.put(header.getName(), header.getValue());
        }
        return map;
    }

    private Header[] createCustomHeaders() throws Exception {
        Header[] headers = new Header[4];
        headers[0] = new BasicHeader("VmallDeviceType", huaweiConfig.getVmallDeviceType());
        headers[1] = new BasicHeader("VmallDeviceAccount", huaweiConfig.getVmallDeviceAccount());
        String vmallTimeStamp = generateVmallTimeStamp();
        headers[2] = new BasicHeader("VmallDevicePassword", getVmallDevicePassword(vmallTimeStamp));
        headers[3] = new BasicHeader("VmallTimeStamp", vmallTimeStamp);
        return headers;
    }

    /**
     * 接入设备密码，密码规则：HmacSha256（密码 + Base64（VmallDeviceType）+  VmallTimeStamp）
     * 密钥：Sha256(密码)
     * @return
     * @param vmallTimeStamp
     */
    private String getVmallDevicePassword(String vmallTimeStamp) throws Exception {
        String data = StringUtils.join(
                huaweiConfig.getVmallDevicePassword(),
                Base64.toBase64String(huaweiConfig.getVmallDeviceType().getBytes(StandardCharsets.UTF_8)),
                vmallTimeStamp
        );
        return SignUtils.hmacSHA256(data, SignUtils.sha256(huaweiConfig.getEncryptSecret()));
    }

    /**
     * 格式：YYYY-MM-DD HH:MM:SS + 4位时区
     * 例如：2013-08-30 12:00:00 +0800
     * @return
     */
    private String generateVmallTimeStamp() {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss ZZZ").format(new Date());
    }

    /**
     * 格式：YYYY-MM-DD HH:MM:SS + 4位时区
     * 例如：2013-08-30 12:00:00+0800
     * @return
     */
    private String generateActTime() {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ssZZZ").format(new Date());
    }

    private String generateTransId() {
        String formatTime = new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date());
        return huaweiConfig.getServiceUnit() + formatTime + RandomStringUtils.randomNumeric(4);
    }

    public static void main(String[] args) {
        System.out.println(new SimpleDateFormat("yyyyMMddHHmmssSSS").format(new Date()));
    }
}
