package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.KunshanRCBConfig;
import cn.com.duiba.credits.sdk.CreditConsumeParams;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.JsonTool;
import cn.com.duiba.tool.kunshanRCB.EncryptAPI;
import cn.com.duiba.tool.kunshanRCB.RadixConvertUtils;
import cn.com.duiba.vo.KunshanRCBCreditsChangeVo;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;

/**
 * @description:
 * @author: liupu1
 * @create: 2020/3/17 4:11 PM
 **/
@Service
public class KunshanRCBApi {


    private static final Logger LOGGER = LoggerFactory.getLogger(KunshanRCBApi.class);

    private static DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    @Resource
    private KunshanRCBConfig bankConfig;

    public boolean isKunshanRCBApp(Long appId) {
        return appId != null && appId.equals(bankConfig.getAppId());
    }


    /**
     * mq 扣积分
     *
     * @param subCreditsMsgWrapper
     * @return
     */
    public SubCreditsMsgWrapper getSubCreditsMessage(SubCreditsMsgWrapper subCreditsMsgWrapper) throws BizException {

        LOGGER.info("kunshanRCB消息队列方式扣积分，接收到SubCreditsMsgWrapper[{}]", JSON.toJSONString(subCreditsMsgWrapper));

        subCreditsMsgWrapper.setHttpUrl(bankConfig.getCreditsUrl());
        SubCreditsMsgDto msg = subCreditsMsgWrapper.getSubCreditsMsg();

        CreditConsumeParams consumeParams = msg.getCreditConsumeParams();

        KunshanRCBCreditsChangeVo creditsChangeVo = new KunshanRCBCreditsChangeVo();

        creditsChangeVo.setVersionNo("V1.0");
        creditsChangeVo.setPointsChannelType("010");
        creditsChangeVo.setFrontDateTime(DATE_TIME_FORMATTER.format(LocalDateTime.now()));
        creditsChangeVo.setAuthoCode("123456");
        String fixedOrderNum = getKunshanOrderNum(consumeParams.getOrderNum());
        creditsChangeVo.setReqSeqNo(fixedOrderNum);
        creditsChangeVo.setBackServiceMethod("000002");
        creditsChangeVo.setPointsCertiType("HCNO");
        creditsChangeVo.setCertiNo(consumeParams.getUid());
        creditsChangeVo.setPointsTypeNo("0001");
        creditsChangeVo.setDebcreFlag("D");
        creditsChangeVo.setPointsTranType("0009");
        creditsChangeVo.setTranAmt(BigDecimal.valueOf(consumeParams.getCredits()));
        creditsChangeVo.setSummary(consumeParams.getDescription());
        creditsChangeVo.setSystemTraceNo(fixedOrderNum);
        creditsChangeVo.setTransId("CUF00001");
        creditsChangeVo.setSceneId("A01");
        String data = JSON.toJSONString(creditsChangeVo);
        LOGGER.info("kunshanRCB消息队列方式扣积分,加密前请求明文:[{}]", data);

        String[] env;
        try {
            env = EncryptAPI.encryptDataEnvelop(bankConfig.getKunshanPublicKey(), data);
        } catch (Exception e) {
            throw new BizException("kunshanRCB消息队列方式扣积分,加密信息报错");
        }

        Map<String, String> authParams = Maps.newHashMap();
        authParams.put("key", env[0]);
        authParams.put("data", env[1]);
        authParams.put("postType", "raw");
        msg.setAuthParams(authParams);
        msg.setHttpType(SubCreditsMsgDto.HTTP_POST);

        LOGGER.info("kunshanRCB消息队列方式扣积分,加密后请求体为:[{}]", JSON.toJSONString(authParams));

        return subCreditsMsgWrapper;
    }

    //兑吧32位订单号转化为昆山农商行订单号(30位以内，实际28位)
    private String getKunshanOrderNum(String orderNum) {
        if (StringUtils.isBlank(orderNum)) return orderNum;
        return RadixConvertUtils.transRadix(orderNum, 36, 62);
    }

    private String getDuibaOrderNum(String orderNum) {
        if (StringUtils.isBlank(orderNum)) return orderNum;
        return StringUtils.leftPad(RadixConvertUtils.transRadix(orderNum, 36, 62), 32, '0');
    }


    public String parseCreditsRsp(String body) {
        LOGGER.info("kunshanRCB消息队列方式加减积分,解密前响应体为:[{}]", body);
        try {
            JSONObject encryptResponse = JSONObject.parseObject(body);
            String key = encryptResponse.getString("key");
            String data = encryptResponse.getString("data");

            String[] decryptData = EncryptAPI.decryptDataEnvelop(bankConfig.getDuibaPrivateKey(), key, data);

            JSONObject responseBody = JSONObject.parseObject(decryptData[1]);

            LOGGER.info("kunshanRCB消息队列方式加减积分,解密后响应体为:[{}]", responseBody.toJSONString());


            String status = responseBody.getString("ResCode");
            String message = responseBody.getString("ErrorProm");
            Map<String, String> duibaDoc = new HashMap<>();
            if (status != null && "0000".equals(status)) {
                duibaDoc.put("status", "ok");
                duibaDoc.put("bizId", responseBody.getString("HostSeqNo"));
            } else {
                duibaDoc.put("status", "fail");
                if (message != null) {
                    duibaDoc.put("errorMessage", message);
                }
            }
            return JsonTool.objectToJson(duibaDoc);
        } catch (Exception e) {
            LOGGER.error("kunshanRCB消息队列方式加减积分,处理响应消息出错", e);
            return body;
        }

    }

    public CreditsMessageDto getAddCreditsMessage(CreditsMessageDto messageDto) throws BizException {

        LOGGER.info("kunshanRCB消息队列方式加积分，接收到CreditsMessageDto[{}]", JSON.toJSONString(messageDto));

        messageDto.setHttpType(CreditsMessageDto.HTTP_POST);
        messageDto.setHttpUrl(bankConfig.getCreditsUrl());

        Map<String, String> originAuthParams = messageDto.getAuthParams();

        KunshanRCBCreditsChangeVo creditsChangeVo = new KunshanRCBCreditsChangeVo();

        creditsChangeVo.setVersionNo("V1.0");
        creditsChangeVo.setPointsChannelType("010");
        creditsChangeVo.setFrontDateTime(DATE_TIME_FORMATTER.format(LocalDateTime.now()));
        creditsChangeVo.setAuthoCode("123456");
        creditsChangeVo.setReqSeqNo(originAuthParams.get("orderNum"));
        creditsChangeVo.setBackServiceMethod("000002");
        creditsChangeVo.setPointsCertiType("HCNO");
        creditsChangeVo.setCertiNo(originAuthParams.get("uid"));
        creditsChangeVo.setPointsTypeNo("0001");
        creditsChangeVo.setDebcreFlag("C");
        creditsChangeVo.setPointsTranType("0009");
        creditsChangeVo.setTranAmt(new BigDecimal(originAuthParams.get("credits")));
        creditsChangeVo.setSummary(originAuthParams.get("description"));
        creditsChangeVo.setSystemTraceNo(originAuthParams.get("orderNum"));
        creditsChangeVo.setPointsDueDateTime(bankConfig.getPointsDueDateTime());
        creditsChangeVo.setBrcNo("0000000");
        creditsChangeVo.setBelgtoDept("01000101");

        creditsChangeVo.setTransId("CUF00001");
        creditsChangeVo.setSceneId("A01");
        String data = JSON.toJSONString(creditsChangeVo);
        LOGGER.info("kunshanRCB消息队列方式加积分,加密前请求明文:[{}]", data);

        String[] env;
        try {
            env = EncryptAPI.encryptDataEnvelop(bankConfig.getKunshanPublicKey(), data);
        } catch (Exception e) {
            throw new BizException("kunshanRCB消息队列方式加积分,加密信息报错");
        }

        Map<String, String> authParams = Maps.newHashMap();
        authParams.put("key", env[0]);
        authParams.put("data", env[1]);
        authParams.put("postType", "raw");

        messageDto.setAuthParams(authParams);
        LOGGER.info("kunshanRCB消息队列方式加积分,加密后请求体为:[{}]", JSON.toJSONString(messageDto));

        return messageDto;
    }
}
