package cn.com.duiba.service.virtualsupplierprocess;

import cn.com.duiba.constant.nongzonghang.FuLuConfig;
import cn.com.duiba.constant.nongzonghang.NongZongHangConfig;
import cn.com.duiba.domain.SupplierResponse;
import cn.com.duiba.domain.VirtualSupplierProcessingOrderLogEntity;
import cn.com.duiba.dto.fulu.FuLuBaseReq;
import cn.com.duiba.dto.fulu.FuLuBaseResp;
import cn.com.duiba.dto.fulu.req.OrderInfoGetReq;
import cn.com.duiba.dto.fulu.resp.OrderInfoGetResp;
import cn.com.duiba.dto.virtualsupplierprocess.MqInfo;
import cn.com.duiba.job.AbstractDuibaSimpleElasticJob;
import cn.com.duiba.mq.RocketMQMsgProducer;
import cn.com.duiba.service.HttpAsyncClientPool;
import cn.com.duiba.service.MessageService;
import cn.com.duiba.tool.JsonTool;
import cn.com.duiba.tool.fulu.FuLuUtils;
import com.alibaba.fastjson.JSON;
import io.elasticjob.autoconfigure.annotation.ElasticJob;
import io.elasticjob.lite.api.ShardingContext;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 虚拟兑换中间状态订单重复查询任务
 *
 * @author haojiahong
 * @date 2020-07-24
 */
@ElasticJob(name = "virtualSupplierProcessingOrderRepeatQueryJob", cron = "0 0/1 * * * ?", shardingTotalCount = 1, overwrite = true)
public class VirtualSupplierProcessingOrderRepeatQueryJob extends AbstractDuibaSimpleElasticJob {
    private static final Logger LOGGER = LoggerFactory.getLogger(VirtualSupplierProcessingOrderRepeatQueryJob.class);

    @Autowired
    private VirtualSupplierProcessOrderService virtualSupplierProcessOrderService;
    @Autowired
    private NongZongHangConfig nongZongHangConfig;
    @Autowired
    private FuLuConfig fuLuConfig;
    @Autowired
    private HttpAsyncClientPool httpAsyncClientPool;
    @Autowired
    private MessageService messageService;
    @Autowired
    private RocketMQMsgProducer rocketMQMsgProducer;

    @Override
    protected void doProcess(ShardingContext shardingContext) {
        if (CollectionUtils.isEmpty(nongZongHangConfig.getAppIds())) {
            return;
        }
        long s = System.currentTimeMillis();
        int size = 0;
        for (Long appId : nongZongHangConfig.getAppIds()) {
            int count = virtualSupplierProcessOrderService.countNeedRepeatQuery(appId);
            if (count <= 0) {
                continue;
            }
            if (count > 100) {
                //每次最多处理100条，防止http请求池中提交过多请求。（有开发者回调通知接口兜底，所以可以增加限制条件）
                count = 100;
            }
            int pageSize = 100;
            int pageNum = count / pageSize + 1;
            for (int i = 1; i <= pageNum; i++) {
                int offset = (i - 1) * pageSize;
                List<VirtualSupplierProcessingOrderLogEntity> list = virtualSupplierProcessOrderService.selectNeedRepeatQueryList(appId, offset, pageSize);
                size = size + list.size();
                for (VirtualSupplierProcessingOrderLogEntity entity : list) {
                    HttpPost httpPost = getHttpPost(entity);
                    httpAsyncClientPool.submit(String.valueOf(appId), httpPost, new FutureCallback<HttpResponse>() {
                        @Override
                        public void completed(HttpResponse response) {
                            try {
                                doCompleted(response, entity);
                            } catch (Exception e) {
                                LOGGER.warn("fulu query error", e);
                            } finally {
                                increaseHasQueryTimes(entity);
                                deleteDoneRecord(entity);
                            }
                        }

                        @Override
                        public void failed(Exception e) {
                            increaseHasQueryTimes(entity);
                        }

                        @Override
                        public void cancelled() {
                            increaseHasQueryTimes(entity);
                        }
                    });
                }
            }

        }
        long e = System.currentTimeMillis();
        LOGGER.info("virtualSupplierProcessingOrderRepeatQueryJob scan size:{} time:{} ms", size, (e - s));
    }

    private void doCompleted(HttpResponse response, VirtualSupplierProcessingOrderLogEntity entity) throws IOException {
        String result = EntityUtils.toString(response.getEntity());
        FuLuBaseResp fuLuBaseResp = JSON.parseObject(result, FuLuBaseResp.class);
        if (fuLuBaseResp.getCode() == 0) {
            OrderInfoGetResp orderInfoGetResp = JSON.parseObject(fuLuBaseResp.getResult(), OrderInfoGetResp.class);
            Map<String, String> duibaDoc = new HashMap<>();
            //下单成功
            if (StringUtils.equals(orderInfoGetResp.getOrderState(), "success")) {
                duibaDoc.put("status", "success");
                duibaDoc.put("supplierBizId", orderInfoGetResp.getOrderId());
                //更新中间状态为完成
                doneRecord(entity);
                //发送消息
                sendMq(entity, SupplierResponse.CALLBACK_TYPE_COMPLETED, duibaDoc, null);
            }
            //下单失败
            if (StringUtils.equals(orderInfoGetResp.getOrderState(), "failed")) {
                LOGGER.warn("fulu，下单失败，resp=={}", JSON.toJSONString(fuLuBaseResp));
                duibaDoc.put("status", "fail");
                duibaDoc.put("errorMessage", "出了点小问题，请重新下单");
                doneRecord(entity);
                sendMq(entity, SupplierResponse.CALLBACK_TYPE_COMPLETED, duibaDoc, null);
            }
            //订单处理中或其他情况，走finally即可
        }
    }


    private HttpPost getHttpPost(VirtualSupplierProcessingOrderLogEntity entity) {
        HttpPost httpPost = new HttpPost(fuLuConfig.getUrl());
        //拼装请求
        FuLuBaseReq fuLuBaseReq = new FuLuBaseReq(fuLuConfig, FuLuConfig.Api.ORDER_INFO_GET.getMethodName());
        OrderInfoGetReq orderInfoGetReq = new OrderInfoGetReq();
        orderInfoGetReq.setCustomerOrderNo(entity.getOrderNum());
        fuLuBaseReq.setBizContent(FuLuUtils.toJSON(orderInfoGetReq));
        fuLuBaseReq.setSign(null);
        fuLuBaseReq.setSign(FuLuUtils.sign(fuLuConfig.getSysSecret(), fuLuBaseReq));
        StringEntity stringEntity = new StringEntity(FuLuUtils.toJSON(fuLuBaseReq), "UTF-8");
        stringEntity.setContentEncoding("UTF-8");
        stringEntity.setContentType("application/json");
        httpPost.setEntity(stringEntity);
        return httpPost;
    }


    private void sendMq(VirtualSupplierProcessingOrderLogEntity entity, String callbackType, Map<String, String> duibaDoc, String errorMsg) {
        SupplierResponse response = new SupplierResponse();
        response.setUrl(entity.getHttpUrl());
        response.setSupplierName(entity.getSupplierName());
        response.setSupplierOrderId(String.valueOf(entity.getSupplierOrderId()));
        response.setOrderId(String.valueOf(entity.getOrderId()));
        response.setAppId(String.valueOf(entity.getAppId()));
        response.setConsumerId(String.valueOf(entity.getConsumerId()));
        response.setCallbackType(callbackType);
        if (MapUtils.isNotEmpty(duibaDoc)) {
            response.setBody(JSON.toJSONString(duibaDoc));
        }
        if (StringUtils.isNotBlank(errorMsg)) {
            response.setErrorMessage(errorMsg);
        }
        String body = JsonTool.objectToJson(response);
        MqInfo mqInfo = JSON.parseObject(entity.getMqInfo(), MqInfo.class);
        if (mqInfo.getMqCallbackTopic().contains("MQ")) {
            rocketMQMsgProducer.sendMsg(mqInfo.getMqCallbackTopic(), mqInfo.getMqTag(), mqInfo.getMqMsgKey(), body, false, null);
        } else {
            messageService.sendMsg(mqInfo.getMqCallbackTopic(), mqInfo.getMqTag(), mqInfo.getMqMsgKey(), body);
        }
    }

    private void doneRecord(VirtualSupplierProcessingOrderLogEntity entity) {
        VirtualSupplierProcessingOrderLogEntity newEntity = new VirtualSupplierProcessingOrderLogEntity();
        newEntity.setId(entity.getId());
        newEntity.setProcessStatus(1);
        virtualSupplierProcessOrderService.updateById(newEntity);
    }

    private void deleteDoneRecord(VirtualSupplierProcessingOrderLogEntity entity) {
        VirtualSupplierProcessingOrderLogEntity entity1 = virtualSupplierProcessOrderService.getById(entity.getId());
        if (entity1.getProcessStatus() == 1) {
            virtualSupplierProcessOrderService.deleteById(entity1.getId());
        }
    }

    private void increaseHasQueryTimes(VirtualSupplierProcessingOrderLogEntity entity) {
        int hasQueryTimes = entity.getHasQueryTimes() + 1;
        VirtualSupplierProcessingOrderLogEntity newEntity = new VirtualSupplierProcessingOrderLogEntity();
        newEntity.setId(entity.getId());
        newEntity.setHasQueryTimes(hasQueryTimes);
        newEntity.setLatestQueryTime(new Date());
        virtualSupplierProcessOrderService.updateById(newEntity);

        if (hasQueryTimes >= 3 && entity.getProcessStatus() == 0) {
            VirtualSupplierProcessingOrderLogEntity updateEntity = new VirtualSupplierProcessingOrderLogEntity();
            updateEntity.setId(entity.getId());
            //查询超过3次仍是处理中，记录状态为2，且发送订单到异常订单
            updateEntity.setProcessStatus(2);
            virtualSupplierProcessOrderService.updateById(updateEntity);

            sendMq(entity, SupplierResponse.CALLBACK_TYPE_FAILED, null, "出了点小问题，请重新下单");
        }
    }
}
