package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.api.bo.subcredits.SubCreditsResultMsgDto;
import cn.com.duiba.api.enums.HttpRequestResultType;
import cn.com.duiba.api.enums.RedisKeySpace;
import cn.com.duiba.api.tools.RandomCodeUtil;
import cn.com.duiba.api.tools.abc.SHAEncrypt4ABCUtils;
import cn.com.duiba.constant.AbchinaConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.tool.JsonTool;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author qianjue
 * @date 2019/9/23
 * @description: 农行定制
 */
@Service
public class AbchinaApi {

    private Logger logger = LoggerFactory.getLogger(AbchinaApi.class);

    public static final String APP_ID = "appid";
    public static final String FRIEND_SEQNO = "friend_seqno";
    public static final String SEQNO = "seqno";
    public static final String ID = "id";
    public static final String CHANNEL = "channel";
    public static final String APP_SECRET = "appsecret";
    public static final String SIGN = "sign";
    public static final String ACTNCOD = "actncod";
    public static final Long  EXPIRE_TIME = 7L;
    public static final String  UIDDECLIMIT_TYPE = "#NEED_AUTH_CHECK#";

    @Autowired
    private AbchinaConfig abchinaConfig;

    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;


    public SubCreditsMsgWrapper getSubCreditsMessage(SubCreditsMsgWrapper subCreditsMsgWrapper) {
        try {
            SubCreditsMsgDto subCreditsMsgDto = subCreditsMsgWrapper.getSubCreditsMsg();
            subCreditsMsgDto.setHttpType(SubCreditsMsgDto.HTTP_POST);
            String url = subCreditsMsgWrapper.getHttpUrl();
            String host = url.substring(0, url.indexOf('?'));
            subCreditsMsgWrapper.setHttpUrl(host);
            Map<String, String> originAuthParams = subCreditsMsgDto.getAuthParams();
            Map<String, String> authParams = Maps.newHashMap();
            if(!subCreditsMsgDto.getParams().isEmpty()){
                authParams.put(ACTNCOD,subCreditsMsgDto.getParams().get(ACTNCOD));
            }
            //兼容星速台扣积分逻辑，定制参数放在description中
            if (authParams.get(ACTNCOD) == null) {
                authParams.put(ACTNCOD, subCreditsMsgDto.getCreditConsumeParams().getDescription());
            }
            authParams.put(APP_ID, abchinaConfig.getDuibaAppId());
            authParams.put(FRIEND_SEQNO, originAuthParams.get("orderNum"));
            authParams.put(ID, originAuthParams.get("uid"));
            authParams.put(APP_SECRET, abchinaConfig.getDuibaAppSecret());
            String transfer = originAuthParams.get("transfer");
            if (StringUtils.isNotBlank(transfer)) {
                String[] str = transfer.split("@");
                authParams.put(SEQNO, str[0]);
                authParams.put(CHANNEL, str[1]);
            }
            authParams.put(SIGN, SHAEncrypt4ABCUtils.encrypt(authParams));
            authParams.remove(APP_SECRET);
            subCreditsMsgWrapper.getSubCreditsMsg().setAuthParams(authParams);
            return subCreditsMsgWrapper;
        } catch (Exception e) {
            logger.info("AbchinaApi call getSubCreditsMessage error , subCreditsMsgWrapper = {}", JSONObject.toJSONString(subCreditsMsgWrapper), e);
            return subCreditsMsgWrapper;
        }
    }

    public String parseCreditsRsp(String body) {
        JSONObject jsonBody;
        try {
            jsonBody = JSON.parseObject(body);
        } catch (Exception e) {
            logger.error("JSON.parseObject:", e);
            return body;
        }
        JSONObject json = JsonTool.getAllJson(jsonBody);
        Map<String, String> duibaDoc = new HashMap<>();
        String status = json.getString("returncode");
        if ("000000".equals(status)) {
            duibaDoc.put("status", "ok");
            //为防止重复，bizId使用当前时间戳+随机4位数字拼接的方式组成
            duibaDoc.put("bizId", getBizId());
        } else {
            duibaDoc.put("status", "fail");
            //弱实名扣豆失败判断条件：returncode的值为UIDDECLIMITEXCEPTION。
            //errorMessage字段会展示给用户，额外添加一个字段记录用户扣积分失败原因为用户为弱实名用户
            String msg = StringUtils.defaultIfBlank(json.getString("returnmsg"),"");
            if("UIDDECLIMITEXCEPTION".equals(status)) {
                msg =UIDDECLIMIT_TYPE + msg;
            }
            duibaDoc.put("errorMessage",msg);
            duibaDoc.put("returnCode", json.getString("returncode"));
        }
        duibaDoc.put("credits", json.getString("numBean"));
        return JsonTool.objectToJson(duibaDoc);
    }

    private String getBizId(){
        return String.valueOf(System.currentTimeMillis())  + "T" + RandomCodeUtil.getNumCode(4);
    }

    public boolean isAbchinaApp(Long appId){
        return abchinaConfig.getAppIds().contains(appId);
    }

    public void addCreditsNum(SubCreditsMsgWrapper req, SubCreditsResultMsgDto resp) {
        if (req.getSubCreditsMsg().getAppId() == null) {
            return ;
        }
        Long appId = req.getSubCreditsMsg().getAppId();
        //不是农行app,不做处理
        if(!isAbchinaApp(appId)){
            return;
        }
        //扣积分是不成功状态，忽略
        if ( !(HttpRequestResultType.COMPLETED.equals(resp.getResultType())
                && SubCreditsResultMsgDto.CODE_SUCCESS == resp.getCode())) {
            return;
        }
        //获取actncod的值
        String actncod = getActncod(req);
        if(StringUtils.isBlank(actncod)){
            return;
        }
        //获取积分值
        Long credits = req.getSubCreditsMsg().getCreditConsumeParams().getCredits();
        if(null == credits){
            credits = 0L;
        }
        redisTemplate.boundHashOps(getBeanNumDateKey()).increment(actncod,1L);
        redisTemplate.boundHashOps(getBeanNumDateKey()).expire(EXPIRE_TIME, TimeUnit.DAYS);
        redisTemplate.boundHashOps(getBeanCountDateKey()).increment(actncod,credits);
        redisTemplate.boundHashOps(getBeanCountDateKey()).expire(EXPIRE_TIME, TimeUnit.DAYS);
    }

    private String getActncod(SubCreditsMsgWrapper req) {
        if(null==req || null == req.getSubCreditsMsg() || req.getSubCreditsMsg().getParams().isEmpty()){
            return StringUtils.EMPTY;
        }
        return req.getSubCreditsMsg().getParams().get(ACTNCOD);
    }

    private String getBeanNumDateKey(){
        return RedisKeySpace.K150.toString() + DateUtils.getDayNumber(new Date()) + "_NUM";
    }

    private String getBeanCountDateKey(){
        return RedisKeySpace.K150.toString() + DateUtils.getDayNumber(new Date()) + "_COUNT";
    }
}
