package cn.com.duiba.biz.credits;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.UnionPayConstants;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.tool.AssembleTool;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Random;

/**
 * 云闪付定制接口
 *
 * @author XuJing
 * @since 2020/10/26 10:04 上午
 */
@Service
public class UnionPayApi {

	private static final Logger LOGGER = LoggerFactory.getLogger(UnionPayApi.class);

	@Autowired
	private UnionPayConstants unionPayConstants;

	/**
	 * 判断当前app是否是云闪付
	 *
	 * @param appId
	 * @return
	 */
	public boolean isUnionPayApp(Long appId) {
		return unionPayConstants.getAppConfig(appId) != null;
	}

	/**
	 * 拼接云闪付积分兑换参数
	 *
	 * @param request
	 * @return
	 */
	public HttpRequestBase getVirtualRequest(SupplierRequest request) {
		String url = request.getHttpUrl();
		String authParams = url.substring(url.indexOf('?') + 1);
		Map<String, String> authParamMap = AssembleTool.getUrlParams(authParams);
		String goodsNum = authParamMap.get("params");

		if (StringUtils.isBlank(goodsNum) || !goodsNum.startsWith(unionPayConstants.getPrefix())) {
			return new HttpGet(url);
		}
		Map<String, String> requestParams = buildVirtualRequestParams(authParamMap, unionPayConstants.getAppConfig(Long.valueOf(request.getAppId())));
		HttpPost post = new HttpPost(unionPayConstants.getVirtualUrl());
		post.setEntity(new StringEntity(JSON.toJSONString(requestParams), ContentType.APPLICATION_JSON));
		return post;
	}

	/**
	 * 构造虚拟商品请求参数
	 */
	private Map<String, String> buildVirtualRequestParams(Map<String, String> authParamMap, UnionPayConstants.AppConfig config) {
		Map<String, String> requestParams = new HashMap<>();
		requestParams.put("appId", config.getAppId());
		requestParams.put("transSeqId", authParamMap.get("orderNum"));
		requestParams.put("transTs", DateUtil.format(new Date(), "yyyyMMdd"));
		requestParams.put("couponId", authParamMap.get("params").replaceAll(unionPayConstants.getPrefix(), ""));
		requestParams.put("openId", authParamMap.get("uid"));
		requestParams.put("acctEntityTp", "03");
		requestParams.put("couponNum", "1");
		requestParams.put("nonceStr", createNonceStr());
		requestParams.put("timestamp", Long.toString(System.currentTimeMillis() / 1000));
		requestParams.put("signature", sign(requestParams, unionPayConstants.getPrivateKey()));
		return requestParams;
	}

	/**
	 * 解析云闪付积分兑换请求结果
	 *
	 * @param message
	 * @param body
	 * @return
	 */
	public String getVirtualResponse(SupplierRequest message, String body) {
		String url = message.getHttpUrl();
		String authParams = url.substring(url.indexOf('?') + 1);
		Map<String, String> authParamMap = AssembleTool.getUrlParams(authParams);
		String goodsNum = authParamMap.get("params");

		if (StringUtils.isBlank(goodsNum) || !goodsNum.startsWith(unionPayConstants.getPrefix())) {
			return body;
		}
		Map<String, String> map = new HashMap<>();
		if (StringUtils.isBlank(body)) {
			throw new ThirdpatyException("云闪付，虚拟商品定制，返回结果为空");
		}
		LOGGER.info("云闪付定制返回结果为:{}", body.length() > 500 ? body.substring(0, 500): body);
		try {
			JSONObject jsonObject = JSON.parseObject(body);
			map.put("status", Objects.equals("00", jsonObject.getString("resp")) ? "success" : "fail");
			map.put("errorMessage", StringEscapeUtils.unescapeHtml4(jsonObject.getString("msg")));
			map.put("supplierBizId", jsonObject.getString("transSeqId"));
		} catch (Exception e) {
			LOGGER.error("云闪付，虚拟商品定制，结果解析错误:{}", body, e);
			map.put("status", "fail");
			map.put("errorMessage", "虚拟商品充值接口响应解析错误");
		}
		return JSON.toJSONString(map);
	}

	/**
	 * 生成签名随机值
	 *
	 * @return
	 */
	public String createNonceStr() {
		String sl = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < 16; i++) {
			sb.append(sl.charAt(new Random().nextInt(sl.length())));
		}
		return sb.toString();
	}


	/**
	 * 云闪付RSA签名
	 */
	public String sign(Map<String, String> param, String signKey) {
		try {
			String value = sortMap(param);
			LOGGER.info("云闪付定制签名原串为:{}", value);
			byte[] keyBytes = Base64.decodeBase64(signKey);
			PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyf = KeyFactory.getInstance("RSA");
			PrivateKey priKey = keyf.generatePrivate(keySpec);
			Signature signature = Signature.getInstance("SHA256WithRSA");
			signature.initSign(priKey);
			signature.update(value.getBytes());
			byte[] signed = signature.sign();
			return Base64.encodeBase64String(signed);
		} catch (Exception e) {
			LOGGER.error("云闪付定制签名失败", e);
			throw new ThirdpatyException("云闪付定制签名失败");
		}
	}


	/**
	 * 参数名按ASCII码从小到大排序（字典序）
	 *
	 * @return
	 */
	public static String sortMap(Map<String, String> param) {
		StringBuilder result = new StringBuilder();
		Collection<String> keySet = param.keySet();
		List<String> list = new ArrayList<String>(keySet);
		Collections.sort(list);
		for (int i = 0; i < list.size(); ++i) {
			String key = list.get(i);
			if ("symmetricKey".equals(key)) {
				continue;
			}
			result.append(key).append("=").append(param.get(key)).append("&");
		}
		String sortMap = result.substring(0, result.length() - 1);
		return sortMap;
	}
}
