package cn.com.duiba.service.impl;

import cn.com.duiba.service.DingService;
import cn.com.duiba.sso.api.domain.dto.AdminDto;
import cn.com.duiba.sso.api.remoteservice.RemoteAdminService;
import cn.com.duiba.tool.CodeException;
import cn.com.duiba.tool.ErrorCode;
import com.dingHelper.auth.AuthHelper;
import com.dingtalk.api.DefaultDingTalkClient;
import com.dingtalk.api.DingTalkClient;
import com.dingtalk.api.request.CorpMessageCorpconversationAsyncsendRequest;
import com.dingtalk.api.response.CorpMessageCorpconversationAsyncsendResponse;
import com.dingtalk.chatbot.DingtalkChatbotClient;
import com.dingtalk.chatbot.SendResult;
import com.dingtalk.chatbot.message.TextMessage;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.RateLimiter;
import com.taobao.api.ApiException;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

@Service
public class DingServiceImpl implements DingService {
    private static final Logger LOGGER = LoggerFactory.getLogger(DingServiceImpl.class);
    private static Cache<String, String> cache = CacheBuilder.newBuilder().expireAfterWrite(5, TimeUnit.MINUTES).build();
    private static Map<String, RateLimiter> rateLimiterMap = new ConcurrentHashMap<>();
    private static final String DING_MESSAGE_URL = "https://oapi.dingtalk.com/robot/send?access_token=";

    @Value("${ding.app.secret}")
    private String secret;

    @Value("${ding.app.appId}")
    private String appId;

    @Value("${ding.app.agentId}")
    private String dingAgentId;

    @Value("${ding.app.restApi}")
    private String dingRestAPi;

    @Autowired
    private RemoteAdminService remoteAdminService;

    @Override
    public void sendPersonalDingTextMsgByEmail(List<String> emails, String text) {
        //根据用户邮箱去sso那边获取钉钉id，后期应该考虑让sso提供批量接口
        List<String> dingUserIds = Lists.newArrayList();
        emails.forEach( email -> {
            AdminDto adminDto = remoteAdminService.findByEmail(email);
            if (adminDto != null && StringUtils.isNotEmpty(adminDto.getDingUserId())) {
                dingUserIds.add(adminDto.getDingUserId());
            }
        });
        this.sendDingTextMsg(dingUserIds, text);
    }

    @Override
    public void sendPersonalDingTextMsgByDingId(List<String> dingIds, String text) {
        this.sendDingTextMsg(dingIds, text);
    }

    /**
     * 发送文本消息到钉钉个人
     * @param dingUserIds 钉钉用户id列表
     * @param text 文本消息
     * @return
     */
    public boolean sendDingTextMsg(List<String> dingUserIds, String text){
        String userIds = String.join(",", dingUserIds);
        CorpMessageCorpconversationAsyncsendRequest req = new CorpMessageCorpconversationAsyncsendRequest();
        req.setMsgtype("text");
        req.setAgentId(Long.valueOf(dingAgentId));
        req.setUseridList(userIds);
        req.setToAllUser(false);
        req.setMsgcontent("{\"content\": \"" + text +  "\"}");
        return getResult(req);
    }

    private boolean getResult(CorpMessageCorpconversationAsyncsendRequest req) {
        DingTalkClient client = new DefaultDingTalkClient(dingRestAPi);
        try {
            CorpMessageCorpconversationAsyncsendResponse rsp = client.execute(req, this.getAccessToken());
            if (rsp == null) {
                throw new RuntimeException("send" + req.getMsgcontent() + " , result: null");
            }
            return rsp.getResult().getSuccess();
        } catch (ApiException e) {
            throw new CodeException(ErrorCode.E9999999, e);
        }
    }

    private String getAccessToken() {
        final String cacheKey = "Access_Token";
        try {
            return cache.get(cacheKey, () -> AuthHelper.getAccessToken(appId,secret));
        } catch (ExecutionException e) {
            throw new CodeException(ErrorCode.E9999999, e);
        }
    }

    /**
     * 每个机器人每分钟只能发送20条消息，超过这个限制消息会被丢弃，要按照钉钉机器人限流量，每分钟每个机器人只能发20条消息
     * 发送钉钉消息到群
     *
     * @param dingTalkToken
     * @param message
     */
    @Override
    public boolean sendDingQunMessage(String dingTalkToken,String message,boolean isAtAll){
        RateLimiter rateLimiter = rateLimiterMap.get(dingTalkToken);
        if (rateLimiter == null) {
            rateLimiter = RateLimiter.create(0.3);
            RateLimiter temp = rateLimiterMap.putIfAbsent(dingTalkToken, rateLimiter);
            if (temp != null) {
                rateLimiter = temp;
            }
        }
        rateLimiter.acquire(1);
        TextMessage textMessage = new TextMessage(message);
        textMessage.setIsAtAll(isAtAll);
        DingtalkChatbotClient client = new DingtalkChatbotClient();
        SendResult sendResult = null;
        try {
            sendResult = client.send(DING_MESSAGE_URL + dingTalkToken, textMessage);
        } catch (IOException e) {
            LOGGER.error(e.getMessage(), e);
            return false;
        }

        return sendResult.isSuccess();
    }
}
