package cn.com.duiba.tool;


import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.constant.HaidilaoSdkConstant;
import cn.com.duiba.constant.HttpConstant;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.collect.Maps;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;




/**
 *
 * 海底捞定制积分接口 签名专用工具类
 * */
public class HaidilaoSignTool {

    private static final Charset CLOUDAPI_ENCODING = Charset.forName("UTF-8");

    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(HaidilaoSignTool.class);

    /**
     * 签名方法
     * 本方法将Request中的httpMethod、headers、path、queryParam、formParam合成一个字符串用hmacSha256算法双向加密进行签名
     *
     * @param method                http method
     * @param appSecret             your app secret
     * @param headerParams          http headers
     * @param pathWithParams        params builted in http path
     * @param queryParams           http query params
     * @param formParams            form params
     * @return signResults
     */
    public static String sign(String method, String appSecret, Map<String, String> headerParams, String pathWithParams, Map<String, String> queryParams, Map<String, String> formParams) {
        try {
            Mac hmacSha256 = Mac.getInstance("HmacSHA256");
            byte[] keyBytes = appSecret.getBytes(HaidilaoSdkConstant.CLOUDAPI_ENCODING);
            hmacSha256.init(new SecretKeySpec(keyBytes, 0, keyBytes.length, "HmacSHA256"));

            //将Request中的httpMethod、headers、path、queryParam、formParam合成一个字符串
            String signString = combineParamsTogether(method, headerParams, pathWithParams, queryParams, formParams);
            //对字符串进行hmacSha256加密，然后再进行BASE64编码
            byte[] signResult = hmacSha256.doFinal(signString.getBytes(HaidilaoSdkConstant.CLOUDAPI_ENCODING));
            return Base64.encodeBase64String(signResult);
        } catch (Exception e) {
            logger.info("海底捞接口签名失败，",e);
            throw new ThirdpatyException("海底捞接口签名失败",e);
        }
    }

    /**
     * 将Request中的httpMethod、headers、path、queryParam、formParam合成一个字符串
     *
     */
    private static String combineParamsTogether(String method, Map<String, String> headerParams, String pathWithParams, Map<String, String> queryParams, Map<String, String> formParams) {

        StringBuilder sb = new StringBuilder();
        sb.append(method).append(HaidilaoSdkConstant.CLOUDAPI_LF);

        //如果有@"Accept"头，这个头需要参与签名
        if (headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_ACCEPT) != null) {
            sb.append(headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_ACCEPT));
        }
        sb.append(HaidilaoSdkConstant.CLOUDAPI_LF);

        //如果有@"Content-MD5"头，这个头需要参与签名
        if (headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_CONTENT_MD5) != null) {
            sb.append(headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_CONTENT_MD5));
        }
        sb.append(HaidilaoSdkConstant.CLOUDAPI_LF);

        //如果有@"Content-Type"头，这个头需要参与签名
        if (headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_CONTENT_TYPE) != null) {
            sb.append(headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_CONTENT_TYPE));
        }
        sb.append(HaidilaoSdkConstant.CLOUDAPI_LF);

        //签名优先读取HTTP_CA_HEADER_DATE，因为通过浏览器过来的请求不允许自定义Date（会被浏览器认为是篡改攻击）
        if (headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_DATE) != null) {
            sb.append(headerParams.get(HttpConstant.CLOUDAPI_HTTP_HEADER_DATE));
        }
        sb.append(HaidilaoSdkConstant.CLOUDAPI_LF);

        //将headers合成一个字符串
        sb.append(buildHeaders(headerParams));
        sb.append(HaidilaoSdkConstant.CLOUDAPI_LF);

        //将path、queryParam、formParam合成一个字符串
        sb.append(buildResource(pathWithParams, queryParams, formParams));
        return sb.toString();
    }

    /**
     * 将path、queryParam、formParam合成一个字符串
     *
     */
    private static String buildResource(String pathWithParams, Map<String, String> queryParams, Map<String, String> formParams) {
        StringBuilder result = new StringBuilder();
        result.append(pathWithParams);

        //使用TreeMap,默认按照字母排序
        TreeMap<String, String> parameter = new TreeMap<String, String>();
        if (MapUtils.isNotEmpty(queryParams)) {
            parameter.putAll(queryParams);
        }
        if (MapUtils.isNotEmpty(formParams)) {
            parameter.putAll(formParams);
        }

        if (parameter.size() > 0) {
            result.append("?");

            // bugfix by VK.Gao@2017-05-03: "kv separator should be ignored while value is empty, ex. k1=v1&k2&k3=v3&k4"
            List<String> comboMap = new ArrayList<String>();
            for(Map.Entry<String, String> entry : parameter.entrySet()){
                String comboResult = entry.getKey() + (StringUtils.isNotEmpty(entry.getValue())? "="+entry.getValue() : StringUtils.EMPTY);
                comboMap.add(comboResult);
            }
            Joiner joiner = Joiner.on("&");
            result.append(joiner.join(comboMap));
        }
        return result.toString();
    }

    /**
     * 将headers合成一个字符串
     * 需要注意的是，HTTP头需要按照字母排序加入签名字符串
     * 同时所有加入签名的头的列表，需要用逗号分隔形成一个字符串，加入一个新HTTP头@"X-Ca-Signature-Headers"
     *
     */
    private static String buildHeaders(Map<String, String> headers) {

        if (MapUtils.isNotEmpty(headers)) {
            // 筛选出需要签名的key
            Predicate<String> signFilter = new Predicate<String>() {
                @Override
                public boolean apply(String input) {
                    return input.startsWith(HaidilaoSdkConstant.CLOUDAPI_CA_HEADER_TO_SIGN_PREFIX_SYSTEM);
                }
            };

            // 使用TreeMap,默认按照字母排序
            Map<String, String> headersToSign = new TreeMap<String, String>(Maps.filterKeys(headers, signFilter));

            // 所有加入签名的头的列表，需要用逗号分隔形成一个字符串，加入一个新HTTP头@"X-Ca-Signature-Headers"
            String signHeaders = Joiner.on(',').join(headersToSign.keySet());
            headers.put(HaidilaoSdkConstant.CLOUDAPI_X_CA_SIGNATURE_HEADERS, signHeaders);

            // 拼装签名内容
            Joiner.MapJoiner joiner = Joiner.on(HaidilaoSdkConstant.CLOUDAPI_LF).withKeyValueSeparator(":");
            return joiner.join(headersToSign);

        } else {
            return StringUtils.EMPTY;
        }


    }

    /**
     * 获取 Content-MD5
     * */
    public static String getContentMd5(String bodyStream){
        try {
            byte[] bytes = bodyStream.getBytes("UTF-8");
            if (bytes == null)
                throw new IllegalArgumentException("bytes can not be null");

            final MessageDigest md = MessageDigest.getInstance("MD5");
            md.reset();
            md.update(bytes);
            byte[] md5Result = md.digest();
            String base64Result = Base64.encodeBase64String(md5Result);
            /*
             * 正常情况下，base64的结果为24位，因与服务器有约定，在超过24位的情况下，截取前24位
             */
            return base64Result.length() > 24 ? base64Result.substring(0, 23) : base64Result;

        }catch (Exception e){
            logger.info("海底捞定制接口处理 "+ HaidilaoSdkConstant.CLOUDAPI_CONTENT_MD5 +" 异常，-{}",e);
            throw new ThirdpatyException("海底捞定制接口处理 "+ HaidilaoSdkConstant.CLOUDAPI_CONTENT_MD5 +" 异常",e);
        }

    }

}
