package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.constant.livat.LivatConfig;
import cn.com.duiba.constant.livat.LivatConstant;
import cn.com.duiba.dao.custom.LivatRollbackRecordDAO;
import cn.com.duiba.domain.LivatRollbackRecordDO;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.SupplierRequest;
import cn.com.duiba.enums.livat.LivatRollbackRecordStatusEnum;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.HttpUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.ExecutorService;

/**
 * Created by fangdong on 2020/07/10
 */
@Service
public class LivatApi {
    private static final Integer SCORE_EVENT = 2000;
    private static final Logger log = LoggerFactory.getLogger(LivatApi.class);
    private static final Integer SUCCESS_CODE = 1;
    private static final String VIRTUAL_CREDITS_PRE = "HJJF-";

    @Resource
    private LivatConstant livatConstant;
    @Resource(name = "httpClient")
    // HTTP超时已修复
    private CloseableHttpClient httpClient;
    @Resource
    private LivatRollbackRecordDAO livatRollbackRecordDAO;
    @Resource(name = "httpCallbackExecutorService")
    private ExecutorService httpCallbackExecutorService;

    /**
     * 是否是改应用
     */
    public boolean isLivat(Long appId) {
        return livatConstant.getAppId().contains(appId);
    }

    /**
     * rpc减积分
     */
    public HttpRequestBase getSubCreditsHttpRequest(CreditsMessage message, Long appId) {
        String url = message.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> authParams = AssembleTool.getUrlParams(paramsStr);
        LivatConfig livatConfig = livatConstant.getConfigParamsMap().get(appId);

        // body
        Map<String, String> body = this.getCreditsHttpBody(authParams);

        // headers
        Map<String, String> headers = this.getHttpRequestHeaders(JSON.toJSONString(body), Long.parseLong(authParams.get("timestamp")),livatConfig);

        // newAuthParams
        Map<String, String> newAuthParams = Maps.newHashMap();
        newAuthParams.putAll(body);
        newAuthParams.putAll(headers);

        message.setHttpType(CreditsMessageDto.HTTP_POST);
        message.setHttpUrl(newUrl);
        message.setAuthParams(newAuthParams);

        return this.getHttpPost(message.getHttpUrl(), JSON.toJSONString(body), headers);
    }

    /**
     * mq减积分
     */
    public HttpRequestBase getMqSubCreditsHttpRequest(SubCreditsMsgWrapper message, Long appId) {
        String url = message.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> authParams = AssembleTool.getUrlParams(paramsStr);


        LivatConfig livatConfig = livatConstant.getConfigParamsMap().get(appId);
        // body
        Map<String, String> body = this.getCreditsHttpBody(authParams);

        // headers
        Map<String, String> headers = this.getHttpRequestHeaders(JSON.toJSONString(body), Long.parseLong(authParams.get("timestamp")), livatConfig);

        // newAuthParams
        Map<String, String> newAuthParams = Maps.newHashMap();
        newAuthParams.putAll(body);
        newAuthParams.putAll(headers);

        message.getSubCreditsMsg().setHttpType(SubCreditsMsgDto.HTTP_POST);
        message.setHttpUrl(newUrl);
        message.getSubCreditsMsg().setAuthParams(newAuthParams);

        return this.getHttpPost(message.getHttpUrl(), JSON.toJSONString(body), headers);
    }

    /**
     * 加积分
     */
    public HttpRequestBase getAddCreditsHttpRequest(CreditsMessageDto message, Long appId) {
        String url = message.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> authParams = AssembleTool.getUrlParams(paramsStr);
        LivatConfig livatConfig = livatConstant.getConfigParamsMap().get(appId);

        // body
        Map<String, String> body = this.getCreditsHttpBody(authParams);

        // headers
        Map<String, String> headers = this.getHttpRequestHeaders(JSON.toJSONString(body), Long.parseLong(authParams.get("timestamp")), livatConfig);

        // newAuthParams
        Map<String, String> newAuthParams = Maps.newHashMap();
        newAuthParams.putAll(body);
        newAuthParams.putAll(headers);

        message.setHttpType(CreditsMessageDto.HTTP_POST);
        message.setHttpUrl(newUrl);
        message.setAuthParams(newAuthParams);

        return this.getHttpPost(message.getHttpUrl(), JSON.toJSONString(body), headers);
    }

    /**
     * 解析积分响应
     */
    public String parseCreditsResponse(String body, Boolean addCredits, Map<String, String> authParams, Long appId) {
        log.info("livat加减积分response, add={}, body={}", addCredits, body);

        JSONObject jsonBody = JSON.parseObject(body);

        JSONObject result = new JSONObject();
        if (SUCCESS_CODE.equals(jsonBody.getInteger("Code"))) {
            result.put("status", "ok");
            // 开发者不返回订单号，自定义
            result.put("bizId", System.currentTimeMillis() + RandomStringUtils.randomNumeric(6));
            result.put("credits", String.valueOf(this.getCredits(authParams.get("Mobile"),appId)));
            if(!addCredits){
                // 创建回滚记录
                LivatRollbackRecordDO rollbackRecord = new LivatRollbackRecordDO();
                rollbackRecord.setPartnerUserId(authParams.get("Mobile"));
                rollbackRecord.setOrderNum(authParams.get("TransID"));
                rollbackRecord.setRecordStatus(LivatRollbackRecordStatusEnum.NEED_ROLLBACK.getCode());
                rollbackRecord.setCredits(Long.valueOf(authParams.get("Score")));
                rollbackRecord.setAppId(appId);
                livatRollbackRecordDAO.insert(rollbackRecord);
            }
        } else {
            result.put("status", "fail");
            result.put("errorMessage", jsonBody.getString("Message"));
        }

        return result.toString();
    }

    /**
     * 回滚积分
     */
    public void rollbackCredits(NotifyQueueDO notify) {
        httpCallbackExecutorService.execute(() -> this.doRollbackCredits(notify));
    }

    private void doRollbackCredits(NotifyQueueDO notify) {
        LivatRollbackRecordDO rollbackRecord = livatRollbackRecordDAO.selectByOrderNum(notify.getDuibaOrderNum());
        if (rollbackRecord == null) {
            log.error("livat积分回滚记录不存在{}", notify.getDuibaOrderNum());
            return;
        }
        if (LivatRollbackRecordStatusEnum.NEED_NOT_ROLLBACK.getCode().equals(rollbackRecord.getRecordStatus())) {
            log.error("livat该状态不允许回滚积分{}", notify.getDuibaOrderNum());
            return;
        }

        try {
            // 更新回滚记录状态，不需要回滚 -> 需要回滚
            LivatRollbackRecordDO update = new LivatRollbackRecordDO();
            update.setId(rollbackRecord.getId());
            update.setRollbackMessage(notify.getError4developer());
            update.setRecordStatus(LivatRollbackRecordStatusEnum.NEED_ROLLBACK.getCode());

            livatRollbackRecordDAO.updateById(update);
            // 增加积分
            this.addCredits(notify.getPartnerUserId(), rollbackRecord.getCredits(), notify.getDuibaOrderNum() + "-rollback",notify.getAppId());

            // 更新回滚成功
            update.setRecordStatus(LivatRollbackRecordStatusEnum.ALREADY_ROLLBACK.getCode());
            livatRollbackRecordDAO.updateById(update);
        } catch (Exception e) {
            log.error("livat积分回滚失败{}, detail={}", notify.getDuibaOrderNum(), JSON.toJSONString(notify), e);

            // 更新回滚记录状态 -> 回滚失败
            LivatRollbackRecordDO update = new LivatRollbackRecordDO();
            update.setId(rollbackRecord.getId());
            update.setRollbackMessage(notify.getError4developer());
            update.setRecordStatus(LivatRollbackRecordStatusEnum.ROLLBACK_FAIL.getCode());

            livatRollbackRecordDAO.updateById(update);
            return;
        }

        log.info("livat积分回滚成功{}, detail={}", notify.getDuibaOrderNum(), JSON.toJSONString(notify));
    }

    /**
     * 虚拟商品请求
     */
    public HttpRequestBase getVirtualRequest(SupplierRequest request, Long appId) {
        String url = request.getHttpUrl();
        String newUrl = url.substring(0, url.indexOf('?'));
        String paramsStr = url.substring(url.indexOf('?') + 1);
        Map<String, String> authParams = AssembleTool.getUrlParams(paramsStr);
        LivatConfig livatConfig = livatConstant.getConfigParamsMap().get(appId);
        String goodsNum = authParams.get("params");
        //如果商品编码是HJJF-开头，则虚拟商品兑换走增加用户会员积分接口
        if (StringUtils.isNotBlank(goodsNum) && goodsNum.startsWith(VIRTUAL_CREDITS_PRE)) {
            if (request.getAuthParams() == null) {
                request.setAuthParams(authParams);
            } else {
                request.getAuthParams().putAll(authParams);
            }
            authParams.put("credits", goodsNum.replace(VIRTUAL_CREDITS_PRE, ""));
            return getAddCreditsVirtualRequest(authParams,livatConfig);
        }

        Map<String, String> item = Maps.newHashMap();
        item.put("BussinessID", authParams.get("orderNum"));
        item.put("TraceID", authParams.get("orderNum") + System.currentTimeMillis() + livatConfig.getDeveloperAppId());
        item.put("PICMID", authParams.get("params"));
        item.put("Mobile", authParams.get("uid"));

        JSONArray items = new JSONArray();
        items.add(JSON.parseObject(JSON.toJSONString(item)));
        JSONObject body = new JSONObject();
        body.put("UserList", items);

        Map<String, String> headers = this.getHttpRequestHeaders(body.toJSONString(), Long.parseLong(authParams.get("timestamp")), livatConfig);

        // newAuthParams
        Map<String, String> newAuthParams = Maps.newHashMap();
        newAuthParams.putAll(item);
        newAuthParams.putAll(headers);

        request.setHttpUrl(newUrl);
        request.setAuthParams(newAuthParams);

        return this.getHttpPost(newUrl, body.toJSONString(), headers);
    }

    private HttpRequestBase getAddCreditsVirtualRequest(Map<String, String> authParams, LivatConfig livatConfig) {
        Map<String, String> body = Maps.newHashMap();
        body.put("Mobile", authParams.get("uid"));
        body.put("ScoreEvent", SCORE_EVENT + "");
        body.put("Score", authParams.get("credits"));
        //不能只用orderNum，以为前面有加减积分使用了orderNum了，加入时间戳
        body.put("TransID", authParams.get("orderNum") + System.currentTimeMillis() + livatConfig.getDeveloperAppId());
        HttpRequestBase request = getHttpPost(livatConstant.getCreditsAddUrl(), JSON.toJSONString(body),
                getHttpRequestHeaders(JSON.toJSONString(body), System.currentTimeMillis(), livatConfig));
        return request;
    }

    /**
     * 解析虚拟商品充值响应结果
     */
    public String getVirtualResponse(SupplierRequest request, String body) {
        log.info("livat虚拟商品充值响应结果response, body={}", body);

        Map<String, String> params = request.getAuthParams();
        String goodsNum = "";
        if (params != null) {
            goodsNum = params.get("params");
        }
        if (StringUtils.isNotBlank(goodsNum) && goodsNum.startsWith(VIRTUAL_CREDITS_PRE)) {
            return getAddCreditsVirtualResponse(body);
        }

        JSONObject jsonBody = JSON.parseObject(body);

        JSONObject result = new JSONObject();
        if (!SUCCESS_CODE.equals(jsonBody.getInteger("Code"))) {
            result.put("status", "fail");
            result.put("errorMessage", jsonBody.getString("Message"));

            return result.toString();
        }

        JSONObject jsonData = jsonBody.getJSONArray("Data").getJSONObject(0);
        if (Boolean.FALSE.equals(jsonData.getBoolean("IsSuccess")) || !SUCCESS_CODE.equals(jsonData.getInteger("Code"))) {
            result.put("status", "fail");
            result.put("errorMessage", jsonData.getString("FailReason"));

            return result.toString();
        }

        result.put("status", "success");
        return result.toString();
    }

    private String getAddCreditsVirtualResponse(String body) {
        JSONObject jsonBody = JSON.parseObject(body);
        JSONObject duibaDoc = new JSONObject();
        if (!SUCCESS_CODE.equals(jsonBody.getInteger("Code"))) {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", jsonBody.getString("Message"));
            return duibaDoc.toString();
        }
        duibaDoc.put("status", "success");
        return duibaDoc.toJSONString();
    }

    /**
     * 获取用户积分
     */
    public Long getCredits(String uid, Long appId) {
        String datetime = DateTime.now().toString("yyyyMMddHHmmss");

        Map<String, String> body = Maps.newHashMap();
        body.put("Mobile", uid);

        HttpPost request = new HttpPost(livatConstant.getVipInfoApiUrl());
        request.setEntity(new StringEntity(JSON.toJSONString(body), ContentType.APPLICATION_JSON));
        LivatConfig livatConfig = livatConstant.getConfigParamsMap().get(appId);
        request.setHeader("AppID", livatConfig.getDeveloperAppId());
        request.setHeader("PublicKey", livatConfig.getDeveloperPublicKey());
        request.setHeader("TimeStamp", datetime);
        request.setHeader("Sign", this.getSign(JSON.toJSONString(body), datetime, livatConfig));

        String response = null;
        HttpUtils.resetTimeOut(request);
        try (CloseableHttpResponse resp = httpClient.execute(request)) {
            response = EntityUtils.toString(resp.getEntity());
            JSONObject jsonResult = JSON.parseObject(response);
            Integer code = jsonResult.getInteger("Code");
            if (!SUCCESS_CODE.equals(code)) {
                String error = jsonResult.getString("Message");
                log.error("livat获取会员信息接口调用失败, request={}, params={}, response={}, code={}, msg={}", request, body, response, code, error);
                return null;
            }

            return jsonResult.getJSONObject("Data").getLong("Score");
        } catch (Exception e) {
            log.error("livat获取会员信息接口调用失败, request={}, params={}, response={}", request, body, response, e);
            return null;
        }
    }

    /**
     * 加积分
     */
    private void addCredits(String uid, Long credits, String orderNum, Long appId) {
        Map<String, String> body = Maps.newHashMap();
        body.put("Mobile", uid);
        body.put("ScoreEvent", SCORE_EVENT + "");
        body.put("Score", String.valueOf(credits));
        body.put("TransID", orderNum);
        LivatConfig livatConfig = livatConstant.getConfigParamsMap().get(appId);
        HttpRequestBase request = this.getHttpPost(livatConstant.getCreditsAddUrl(), JSON.toJSONString(body),
                this.getHttpRequestHeaders(JSON.toJSONString(body), System.currentTimeMillis(), livatConfig));

        String response = null;
        HttpUtils.resetTimeOut(request);
        try (CloseableHttpResponse resp = httpClient.execute(request)) {
            response = EntityUtils.toString(resp.getEntity());
            JSONObject jsonResult = JSON.parseObject(response);
            Integer code = jsonResult.getInteger("Code");
            if (!SUCCESS_CODE.equals(code)) {
                String error = jsonResult.getString("Message");
                log.error("livat加积分接口调用失败, request={}, response={}, code={}, msg={}", request, response, code, error);
            }

        } catch (Exception e) {
            log.error("livat加积分接口调用失败, request={}, response={}", request, response, e);
        }
    }

    /**
     * 获取加减积分http请求
     */
    private HttpRequestBase getHttpPost(String httpUrl, String jsonBody, Map<String, String> headers) {
        HttpPost request = new HttpPost(httpUrl);
        request.setEntity(new StringEntity(jsonBody, ContentType.APPLICATION_JSON));

        for (Map.Entry<String, String> entry : headers.entrySet()) {
            request.setHeader(entry.getKey(), entry.getValue());
        }

        return request;
    }

    /**
     * 获取请求头
     */
    private Map<String, String> getHttpRequestHeaders(String jsonBody, long timestamp, LivatConfig livatConfig) {
        String datetime = new DateTime(timestamp).toString("yyyyMMddHHmmss");

        Map<String, String> headers = Maps.newHashMap();
        headers.put("AppID", livatConfig.getDeveloperAppId());
        headers.put("PublicKey", livatConfig.getDeveloperPublicKey());
        headers.put("TimeStamp", datetime);
        headers.put("Sign", this.getSign(jsonBody, datetime,livatConfig));

        return headers;
    }

    /**
     * 获取加减积分请求体
     */
    private Map<String, String> getCreditsHttpBody(Map<String, String> authParams) {
        Map<String, String> map = Maps.newHashMap();
        map.put("Mobile", authParams.get("uid"));
        map.put("ScoreEvent", SCORE_EVENT + "");
        map.put("Score", authParams.get("credits"));
        try {
            map.put("Reason", URLEncoder.encode(authParams.get("description"), StandardCharsets.UTF_8.name()));
        } catch (Exception e) {
            log.error("", e);
        }
        map.put("TransID", authParams.get("orderNum"));

        return map;
    }

    /**
     * 获取签名
     */
    private String getSign(String jsonParams, String datetime, LivatConfig livatConfig) {
        String encryptString = "{publicKey:" + livatConfig.getDeveloperPublicKey() +
                ",timeStamp:" + datetime +
                ",data:" + jsonParams +
                ",privateKey:" + livatConfig.getDeveloperPrivateKey() + "}";
        return DigestUtils.md5Hex(encryptString).substring(8, 24).toUpperCase();
    }
}
