package cn.com.duiba.biz.credits;

import cn.com.duiba.api.bo.subcredits.SubCreditsMsgDto;
import cn.com.duiba.api.tools.abc.SHAEncrypt4ABCUtils;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.ZHCreditsConfig;
import cn.com.duiba.dao.custom.ZHCreditsLogDAO;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.domain.ZHCreditsLogDO;
import cn.com.duiba.enums.zhcreditslog.LogStatusEnum;
import cn.com.duiba.enums.zhcreditslog.LogTypeEnum;
import cn.com.duiba.order.center.api.dto.CreditsMessage;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.tool.HttpRequestLog;
import cn.com.duiba.tool.JsonTool;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;


/**
 * 中宏积分服务
 * Created by 歪大哥😁 on 2019-10-24.
 */
@Service
public class ZHCreditsApi {

    private static final Logger LOGGER = LoggerFactory.getLogger(ZHCreditsApi.class);

    private static final String DEF_TEXT = "积分返还";

    private static final String USER_NM_KEY = "userNm";
    private static final String DESCRIPTION_KEY = "description";

    @Autowired
    private ZHCreditsConfig config;
    @Autowired
    private ZHCreditsLogDAO zhCreditsLogDAO;
    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;

    /**
     * 判断当前app是否是中宏保险的app
     * @param appId
     * @return
     */
    public boolean isZHApp(Long appId) {
        return config.isZHApp(appId);
    }

    /**
     * 回滚积分
     */
    public void rollbackCredits(String orderNum) {
        ZHCreditsLogDO lastLog = zhCreditsLogDAO.findByOrderNumAndLogType(orderNum, LogTypeEnum.SUB_CREDITS);
        if (lastLog == null) {
            LOGGER.info("中宏积分回滚失败--msg=未找到扣积分记录, orderNum={}", orderNum);
            return;
        }
        try {
            HttpPost post = new HttpPost(config.getNewQueryCreditsHost());
            JSONObject remarks = generateRollBackRemarks(lastLog);

            ZHCreditsLogDO newLog = createAndGetRequestSn(lastLog.getCredits() + "", lastLog.getOrderNum(), lastLog.getAppId(), LogTypeEnum.ROLL_BACK_CREDITS, lastLog.getCid(), lastLog.getUid(), lastLog.getRequestSn(), remarks.toJSONString());
            JSONObject params = new JSONObject();
            params.put("partnerCode",config.getPartnerCode());
            params.put("requestId",newLog.getRequestSn());
            String timestamp = System.currentTimeMillis()+"";
            params.put("timestamp",timestamp);
            params.put("transCode",config.getUpdateCreditsTransCode());
            params.put("sign",SHAEncrypt4ABCUtils.encrypt(config.getSalt() + config.getPartnerCode() + newLog.getRequestSn() + timestamp));
            JSONObject reqData = new JSONObject();
            reqData.put("domainCode",config.getWpaId());
            reqData.put("extId",lastLog.getUid());
            reqData.put("pointSource","MCD-DB");
            reqData.put("pointType","ROLLBACK");
            reqData.put("originValue",lastLog.getCredits()+"");
            reqData.put("bizId","RollBack"+orderNum);
            JSONObject bizInfo = new JSONObject();
            bizInfo.put("consumedBizId",orderNum);
            reqData.put("bizInfo",bizInfo);
            params.put("reqData",reqData);
            StringEntity entity = new StringEntity(params.toJSONString());
            entity.setContentEncoding("utf-8");
            entity.setContentType("application/json");
            post.setEntity(entity);
            HttpRequestLog.logUrl("[action subCredits-rollback]] [logId = " + newLog.getId()+"][request bizId " + orderNum + "]  [params " + params.toJSONString() + "]");
            try (CloseableHttpResponse response = httpClient.execute(post)) {
                String result = EntityUtils.toString(response.getEntity());
                HttpRequestLog.logUrl("[action subCredits-rollback]] [logId = " + newLog.getId()+"][response bizId " + orderNum + "] [result " + result + "]");
                zhCreditsLogDAO.updateResponseInfo(newLog.getId(), result);
            } catch (Exception e) {
                LOGGER.info("中宏定制积分回滚接口调用异常 orderNum={}, cid={}", orderNum, lastLog.getCid(), e);
            }
        } catch (Exception ex) {
            LOGGER.info("中宏定制积分回滚接口系统异常 orderNum={}, cid={}", orderNum, lastLog.getCid(), ex);
        }
    }

    private JSONObject generateRollBackRemarks(ZHCreditsLogDO lastLog) {
        JSONObject remarks = new JSONObject();
        remarks.put("lastRequestSn", lastLog.getRequestSn());
        String oldRemarks = lastLog.getRemarks();
        if (StringUtils.isBlank(oldRemarks)) {
            remarks.put(USER_NM_KEY, "");
            remarks.put(DESCRIPTION_KEY, DEF_TEXT);
            return remarks;
        }
        try {
            JSONObject json = JSONObject.parseObject(oldRemarks);
            String desc = json.getString(DESCRIPTION_KEY);
            if (StringUtils.isBlank(desc)) {
                desc = DEF_TEXT;
            }

            remarks.put(USER_NM_KEY, json.getString(USER_NM_KEY));
            remarks.put(DESCRIPTION_KEY, desc);
            return remarks;
        } catch (Exception e) {
            LOGGER.info("解析remarks错误", e);
            remarks.put(USER_NM_KEY, "");
            remarks.put(DESCRIPTION_KEY, DEF_TEXT);
            return remarks;
        }
    }

    /**
     * 组装加积分对象 正常加积分
     * @param request
     * @return
     */
    public CreditsMessageDto getAddCreditsMessage(CreditsMessageDto request) throws BizException {
        try {
            request.setHttpUrl(config.getNewQueryCreditsHost());
            Map<String, String> originAuthParams = request.getAuthParams();
            String orderNum = originAuthParams.get("orderNum");
            String uid = originAuthParams.get("uid");
            String credits = originAuthParams.get("credits");
            String description = originAuthParams.get(DESCRIPTION_KEY);
            String transfer = originAuthParams.get("transfer");
            Long cid = Long.valueOf(request.getConsumerId());
            JSONObject remarks = new JSONObject();
            remarks.put(USER_NM_KEY, transfer);
            remarks.put(DESCRIPTION_KEY, description);
            ZHCreditsLogDO log = createAndGetRequestSn(credits, orderNum, Long.valueOf(request.getAppId()), LogTypeEnum.ADD_CREDITS, cid, uid,null, remarks.toJSONString());
            Map<String, String> authParams = Maps.newHashMap();
            JSONObject body = new JSONObject();
            body.put("partnerCode",config.getPartnerCode());
            body.put("requestId",log.getRequestSn());
            String timestamp = System.currentTimeMillis() + "";
            body.put("timestamp",timestamp);
            body.put("transCode",config.getUpdateCreditsTransCode());
            body.put("sign",SHAEncrypt4ABCUtils.encrypt(config.getSalt() + config.getPartnerCode() + log.getRequestSn() + timestamp));
            JSONObject reqData = new JSONObject();
            reqData.put("domainCode",config.getWpaId());
            reqData.put("extId",uid);
            reqData.put("pointSource","MCD-DB");
            reqData.put("pointType","DB-COMMON");
            reqData.put("originValue",credits);
            reqData.put("bizId",orderNum);
            body.put("reqData",reqData);
            authParams.put("body",body.toString());
            authParams.put("isZhCreditsRequest","1");
            request.setAuthParams(authParams);
            request.setHttpType(CreditsMessage.HTTP_POST);
            return request;
        } catch (Exception ex) {
            LOGGER.info("ZHCreditsApi call getSubCreditsMessage error , request = {}", JSONObject.toJSONString(request), ex);
            throw new BizException("中宏扣积分流程中断");
        }
    }

    /**
     * 组装扣积分对象  remote形式发起的扣积分请求
     * @param request
     * @return
     */
    public CreditsMessage getSubCreditsMessage(CreditsMessage request) throws BizException {
        try {
            request.setHttpUrl(config.getNewQueryCreditsHost());
            Map<String, String> originAuthParams = request.getAuthParams();
            String orderNum = originAuthParams.get("orderNum");
            String uid = originAuthParams.get("uid");
            String description = originAuthParams.get(DESCRIPTION_KEY);
            String transfer = originAuthParams.get("transfer");
            String credits = originAuthParams.get("credits");
            Long cid = Long.valueOf(request.getConsumerId());
            JSONObject remarks = new JSONObject();
            remarks.put(USER_NM_KEY, transfer);
            remarks.put(DESCRIPTION_KEY, description);
            String requestSn = createAndGetRequestSn(credits, orderNum, Long.valueOf(request.getAppId()), LogTypeEnum.SUB_CREDITS, cid, uid,null, remarks.toJSONString()).getRequestSn();
            Map<String, String> authParams = Maps.newHashMap();
            JSONObject body = new JSONObject();
            body.put("partnerCode",config.getPartnerCode());
            body.put("requestId",requestSn);
            String timestamp = System.currentTimeMillis() + "";
            body.put("timestamp",timestamp);
            body.put("transCode",config.getUpdateCreditsTransCode());
            body.put("sign",SHAEncrypt4ABCUtils.encrypt(config.getSalt() + config.getPartnerCode() + requestSn + timestamp));
            JSONObject reqData = new JSONObject();
            reqData.put("domainCode",config.getWpaId());
            reqData.put("extId",uid);
            reqData.put("pointSource","MCD-DB");
            reqData.put("pointType","CONSUME");
            reqData.put("originValue",credits);
            reqData.put("bizId",orderNum);
            body.put("reqData",reqData);
            authParams.put("body",body.toString());
            authParams.put("isZhCreditsRequest","1");
            request.setAuthParams(authParams);
            request.setHttpType(CreditsMessage.HTTP_POST);
            return request;
        } catch (Exception ex) {
            LOGGER.info("ZHCreditsApi call getSubCreditsMessage error , request = {}", JSONObject.toJSONString(request), ex);
            throw new BizException("中宏扣积分流程中断");
        }
    }

    /**
     * 组装扣积分对象  mq形式发起的扣积分请求
     * @param subCreditsMsgWrapper
     * @return
     */
    public SubCreditsMsgWrapper getSubCreditsMessage(SubCreditsMsgWrapper subCreditsMsgWrapper) throws BizException {
        try {
            subCreditsMsgWrapper.setHttpUrl(config.getNewQueryCreditsHost());
            SubCreditsMsgDto msg = subCreditsMsgWrapper.getSubCreditsMsg();
            Map<String, String> originAuthParams = msg.getAuthParams();
            String orderNum = originAuthParams.get("orderNum");
            String uid = originAuthParams.get("uid");
            String credits = originAuthParams.get("credits");
            String description = originAuthParams.get(DESCRIPTION_KEY);
            String transfer = originAuthParams.get("transfer");
            Long cid = msg.getConsumerId();
            JSONObject remarks = new JSONObject();
            remarks.put(USER_NM_KEY, transfer);
            remarks.put(DESCRIPTION_KEY, description);
            String requestSn = createAndGetRequestSn(credits, orderNum, msg.getAppId(), LogTypeEnum.SUB_CREDITS, cid, uid,null, remarks.toJSONString()).getRequestSn();
            Map<String, String> authParams = Maps.newHashMap();
            JSONObject body = new JSONObject();
            body.put("partnerCode",config.getPartnerCode());
            body.put("requestId",requestSn);
            String timestamp = System.currentTimeMillis() + "";
            body.put("timestamp",timestamp);
            body.put("transCode",config.getUpdateCreditsTransCode());
            body.put("sign",SHAEncrypt4ABCUtils.encrypt(config.getSalt() + config.getPartnerCode() + requestSn + timestamp));
            JSONObject reqData = new JSONObject();
            reqData.put("domainCode",config.getWpaId());
            reqData.put("extId",uid);
            reqData.put("pointSource","MCD-DB");
            reqData.put("pointType","CONSUME");
            reqData.put("originValue",credits);
            reqData.put("bizId",orderNum);
            body.put("reqData",reqData);
            authParams.put("body",body.toString());
            authParams.put("isZhCreditsRequest","1");
            msg.setAuthParams(authParams);
            msg.setHttpType(SubCreditsMsgDto.HTTP_POST);
            return subCreditsMsgWrapper;
        } catch (Exception ex) {
            LOGGER.info("ZHCreditsApi call getSubCreditsMessage error , subCreditsMsgWrapper = {}", JSONObject.toJSONString(subCreditsMsgWrapper), ex);
            throw new BizException("中宏扣积分流程中断");
        }
    }

    /**
     * 解析扣积分请求响应  转换成兑吧格式
     * @param body
     * @param authParams
     * @return
     */
    public String parseCreditsRsp(String body, Map<String, String> authParams) {
        JSONObject responseBody;
        try {
            responseBody = JSONObject.parseObject(body);
        } catch (Exception e) {
            LOGGER.error("JSON.parseObject:", e);
            return body;
        }
        Map<String, String> duibaDoc = new HashMap<>();

        if (Objects.equals(responseBody.getString("resCode"), "0000")
                && Objects.equals(responseBody.getJSONObject("resData").getString("isSaved"), "Y")) {
            duibaDoc.put("status", "ok");
            duibaDoc.put("bizId", "");
        } else {
            duibaDoc.put("status", "fail");
            duibaDoc.put("errorMessage", responseBody.getString("resMsg"));
        }
        JSONObject params = JSONObject.parseObject(authParams.get("body"));
        String requestId = params.getString("requestId");
        if (requestId != null) {
            duibaDoc.put("bizId", requestId);
        }
        ZHCreditsLogDO log = zhCreditsLogDAO.findByRequestSn(requestId);
        if (log != null) {
            zhCreditsLogDAO.updateResponseInfo(log.getId(), body);
            String credits = queryConsumerCredits(log.getUid());
            if (credits != null) {
                duibaDoc.put("credits", credits);
            }
        }
        return JsonTool.objectToJson(duibaDoc);
    }

    private String queryConsumerCredits(String uid) {
        String requestSn = null;
        try {
            HttpPost post = new HttpPost(config.getNewQueryCreditsHost());
            requestSn = zhCreditsLogDAO.generateNewRequestSn(config.getPartnerCode());
            JSONObject body = new JSONObject();
            body.put("partnerCode", config.getPartnerCode());
            body.put("requestId", requestSn);
            String timestamp = System.currentTimeMillis() + "";
            body.put("timestamp",timestamp);
            body.put("transCode",config.getGetCreditsTransCode());
            body.put("sign", SHAEncrypt4ABCUtils.encrypt(config.getSalt() + config.getPartnerCode() + requestSn + timestamp));
            JSONObject reqData = new JSONObject();
            reqData.put("domainCode",config.getWpaId());
            reqData.put("extId",uid);
            body.put("reqData",reqData);
            StringEntity postEntity = new StringEntity(body.toJSONString(), "utf-8");
            postEntity.setContentEncoding("utf-8");
            postEntity.setContentType("application/json");
            post.setEntity(postEntity);
            try (CloseableHttpResponse response = httpClient.execute(post)) {
                String result = EntityUtils.toString(response.getEntity(), "utf-8");
                JSONObject json = JSONObject.parseObject(result);
                String resCode = json.getString("resCode");
                if (Objects.equals(resCode, "0000")) {
                    JSONObject resData = json.getJSONObject("resData");
                    if (resData != null) {
                        return resData.getString("rewardPoint");
                    }
                    LOGGER.info("中宏定制积分查询接口用户不存在 openid={}", uid);
                    return null;
                } else {
                    LOGGER.info("中宏定制积分查询接口调用异常 code={}, msg={}, requestSn={}", resCode, json.getString("resMsg"), requestSn);
                    LOGGER.info(json.toString());
                    return null;
                }
            } catch (Exception e) {
                LOGGER.info("中宏定制积分查询接口调用异常 uid={}, requestSn={}", uid, requestSn, e);
                return null;
            }
        } catch (Exception ex) {
            LOGGER.info("中宏定制积分查询接口调用异常 uid={} requestSn={}", uid, requestSn, ex);
            return null;
        }
    }

    private ZHCreditsLogDO createAndGetRequestSn(String credits, String orderNum, Long appId, LogTypeEnum type, Long cid, String uid, String lastRequestSn, String remarks) throws BizException {
        String requestSn;
        ZHCreditsLogDO log;
        switch (type) {
            case SUB_CREDITS:
                log = new ZHCreditsLogDO();
                requestSn = zhCreditsLogDAO.generateNewRequestSn(config.getPartnerCode());
                log.setRequestSn(requestSn);
                log.setAppId(appId);
                log.setCredits(Long.valueOf(credits));
                log.setLogType(type.getType());
                log.setLogStatus(LogStatusEnum.CREATED.getCode());
                log.setOrderNum(orderNum);
                log.setCid(cid);
                log.setUid(uid);
                log.setRemarks(remarks);
                zhCreditsLogDAO.insert(log);
                break;
            case ADD_CREDITS:
                log = zhCreditsLogDAO.findByOrderNumAndLogType(orderNum, LogTypeEnum.ADD_CREDITS);
                if (log == null) {
                    log = new ZHCreditsLogDO();
                    requestSn = zhCreditsLogDAO.generateNewRequestSn(config.getPartnerCode());
                    log.setRequestSn(requestSn);
                    log.setAppId(appId);
                    log.setCredits(Long.valueOf(credits));
                    log.setLogType(type.getType());
                    log.setLogStatus(LogStatusEnum.CREATED.getCode());
                    log.setOrderNum(orderNum);
                    log.setCid(cid);
                    log.setUid(uid);
                    log.setRemarks(remarks);
                    zhCreditsLogDAO.insert(log);
                }
                break;
            case ROLL_BACK_CREDITS:
                log = new ZHCreditsLogDO();
                requestSn = zhCreditsLogDAO.generateNewRequestSn(config.getPartnerCode());
                log.setRequestSn(requestSn);
                log.setAppId(appId);
                log.setCredits(Long.valueOf(credits));
                log.setLogType(type.getType());
                log.setLogStatus(LogStatusEnum.CREATED.getCode());
                log.setOrderNum(orderNum);
                log.setLastRequestSn(lastRequestSn);
                log.setCid(cid);
                log.setUid(uid);
                log.setRemarks(remarks);
                zhCreditsLogDAO.insert(log);
                break;
                default: throw new BizException("中宏扣积分流程中断");
        }
        return log;
    }

}
