package cn.com.duiba.biz.credits.strategy.Impl;

import cn.com.duiba.biz.Exception.ThirdpatyException;
import cn.com.duiba.biz.credits.strategy.ApiStrategy;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.constant.hsbc.HsbcConfig;
import cn.com.duiba.domain.SubCreditsMsgWrapper;
import cn.com.duiba.dto.hsbc.HsbcCreditsRespData;
import cn.com.duiba.dto.hsbc.HsbcUserInfoRespData;
import cn.com.duiba.dto.hsbc.common.HsbcRespBody;
import cn.com.duiba.dto.hsbc.common.HsbcRespData;
import cn.com.duiba.notifycenter.domain.NotifyQueueDO;
import cn.com.duiba.thirdparty.dto.CreditsMessageDto;
import cn.com.duiba.thirdparty.dto.hsbc.UserInfoDto;
import cn.com.duiba.tool.AssembleTool;
import cn.com.duiba.tool.hsbc.HsbcTool;
import cn.hutool.core.convert.Convert;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Base64Utils;

import javax.annotation.Resource;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;
import java.io.IOException;
import java.util.Date;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

/**
 * 汇丰汇选-接口定制 review me
 *
 * @author fja
 */
@Service
public class HsbcApiStrategy implements ApiStrategy {

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private final RequestConfig config;

    private static final int FIVE_SECONDS = 5 * 1000;

    private static final int TEN_SECONDS = 5 * 1000;
    /**
     * 开发者成功状态
     */
    private static final String STATUS_SUCCESS = "10000";

    {
        //http请求超时配置
        config = RequestConfig.custom().setConnectTimeout(FIVE_SECONDS).setSocketTimeout(TEN_SECONDS).setConnectionRequestTimeout(500).build();
    }

    @Resource
    private HsbcTool hsbcTool;

    @Resource
    private Validator validator;

    @Autowired
    private HsbcConfig hsbcConfig;

    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;


    @Override
    public HttpRequestBase getMqSubCreditsHttpRequest(SubCreditsMsgWrapper message) {
        //请求链接 & 解析
        String url = hsbcTool.getHostName(message.getHttpUrl());
        HttpPost httpPost = new HttpPost(url);
        Map<String, String> originData = AssembleTool.getUrlParams(hsbcTool.getParamUrl(message.getHttpUrl()));
        log.info("[hsbc]减积分originData = 【{}】", JSON.toJSONString(originData));

        Long consumerId = message.getSubCreditsMsg().getConsumerId();
        try {
            JSONObject bizObject = buildSubCreditsBizObject(originData, consumerId);

            //生成请求body
            Map<String, Object> body = buildRequestBody(bizObject);
            String bodyStr = JSON.toJSONString(body);

            String jwtToken = hsbcTool.generateToken(buildJwtClaim(bodyStr));

            //生成HttpPost
            httpPost.setHeader("Authorization", jwtToken);
            httpPost.setEntity(new StringEntity(bodyStr, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);


            message.setHttpUrl(url);
            Map<String, String> authParams = Optional.ofNullable(message.getSubCreditsMsg().getAuthParams()).orElse(Maps.newHashMap());
            body.forEach((key, value) -> {
                if (!authParams.containsKey(key)) {
                    authParams.put(key, String.valueOf(value));
                }
            });
            message.getSubCreditsMsg().setAuthParams(authParams);


            log.info("[hsbc] cid = 【{}】 orderNum = 【{}】 减积分加密body = 【{}】 token = 【{}】", consumerId, originData.get("orderNum"), bodyStr, jwtToken);
            log.info("[hsbc]计时 uid = 【{}】 orderNum = 【{}】，stage =【getMqSubCreditsHttpRequest】，current = 【{}】", originData.get("uid"), originData.get("orderNum"), System.currentTimeMillis());
            return httpPost;
        } catch (Exception e) {
            log.error(String.format("[hsbc] cid = 【%s】 生成扣积分请求失败，originData = 【%s】", consumerId, JSON.toJSONString(originData)), e);
            throw new IllegalStateException(e);
        }
    }


    /**
     * 生成汇丰的扣积分businessObject对象
     *
     * @param originData 数据
     * @return businessObject对象
     */
    private JSONObject buildSubCreditsBizObject(Map<String, String> originData, Long consumerId) {
        JSONObject businessObject = new JSONObject();
        businessObject.put("uid", originData.get("uid"));
        businessObject.put("credits", originData.get("credits"));
        businessObject.put("itemCode", originData.get("itemCode"));
        businessObject.put("timeStamp", System.currentTimeMillis());
        businessObject.put("description", originData.get("description"));
        businessObject.put("orderNum", originData.get("orderNum"));
        businessObject.put("type", originData.get("type"));
        businessObject.put("actualPrice", originData.get("actualPrice"));
        businessObject.put("productName", originData.get("description"));
        return businessObject;
    }


    @Override
    public HttpRequestBase getAddCreditsMessageRequest(CreditsMessageDto message) {
        String url = hsbcTool.getHostName(message.getHttpUrl());
        HttpPost httpPost = new HttpPost(url);
        Map<String, String> originData = AssembleTool.getUrlParams(hsbcTool.getParamUrl(message.getHttpUrl()));
        log.info("[hsbc]加积分originData = 【{}】", JSON.toJSONString(originData));

        try {
            JSONObject bizObject = buildAddCreditsBizObject(originData, Long.valueOf(message.getConsumerId()));


            //生成请求body
            Map<String, Object> body = buildRequestBody(bizObject);
            String bodyStr = JSON.toJSONString(body);


            //生成HttpPost
            String jwtToken = hsbcTool.generateToken(buildJwtClaim(bodyStr));
            httpPost.setHeader("Authorization", jwtToken);
            httpPost.setEntity(new StringEntity(bodyStr, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);


            message.setHttpUrl(url);
            Map<String, String> authParams = Optional.ofNullable(message.getAuthParams()).orElse(Maps.newHashMap());
            body.forEach((key, value) -> {
                if (!authParams.containsKey(key)) {
                    authParams.put(key, String.valueOf(value));
                }
            });
            message.setAuthParams(authParams);

            log.info("[hsbc]cid = 【{}】 orderNum = 【{}】 加积分body = 【{}】 token = 【{}】", message.getConsumerId(), originData.get("orderNum"), bodyStr, jwtToken);
            log.info("[hsbc]计时 uid = 【{}】 orderNum = 【{}】，stage =【getAddCreditsMessageRequest】，current = 【{}】", originData.get("uid"), originData.get("orderNum"), System.currentTimeMillis());

            return httpPost;
        } catch (Exception e) {
            log.error(String.format("[hsbc]生成加积分请求失败，originData = 【%s】", JSON.toJSONString(originData)), e);
            throw new IllegalStateException(e);
        }
    }


    /**
     * 生成汇丰的扣积分businessObject对象
     *
     * @param originData 数据
     * @return bizObject对象
     */
    private JSONObject buildAddCreditsBizObject(Map<String, String> originData, Long consumerId) {
        JSONObject bizObject = new JSONObject();
        bizObject.put("uid", originData.get("uid"));
        bizObject.put("credits", originData.get("credits"));
        bizObject.put("timeStamp", System.currentTimeMillis());
        bizObject.put("description", originData.get("description"));
        bizObject.put("orderNum", originData.get("orderNum"));
        bizObject.put("type", originData.get("type"));
        bizObject.put("approach", originData.get("description"));
        return bizObject;
    }


    /**
     * 通知开发者
     *
     * @param notifyUrl 通知url
     * @param record    记录
     * @return 请求httpRequest
     */
    @Override
    public HttpRequestBase getRequestNotify(String notifyUrl, NotifyQueueDO record) {
        HttpPost httpPost = new HttpPost(notifyUrl);
        String partnerUserId = record.getPartnerUserId();
        try {
            JSONObject bizObject = buildNotifyBizObject(record);


            Map<String, Object> body = buildRequestBody(bizObject);
            String bodyStr = JSON.toJSONString(body);

            //生成HttpPost
            httpPost.setHeader("Authorization", hsbcTool.generateToken(buildJwtClaim(bodyStr)));
            httpPost.setEntity(new StringEntity(bodyStr, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);
            return httpPost;
        } catch (Exception e) {
            log.error(String.format("[hsbc] uid =【%s】 生成notify请求失败，NotifyQueueDO = 【%s】", partnerUserId, JSON.toJSONString(record)), e);
            throw new IllegalStateException(e);
        }
    }


    /**
     * 生成通知businessObject对象
     *
     * @param record 通知DO
     * @return bizObject对象
     */
    private JSONObject buildNotifyBizObject(NotifyQueueDO record) {
        JSONObject bizObject = new JSONObject();
        bizObject.put("uid", record.getPartnerUserId());
        bizObject.put("success", isExchangeSuccess(record));
        bizObject.put("timeStamp", System.currentTimeMillis());
        bizObject.put("bizId", record.getDeveloperBizId());
        bizObject.put("orderNum", record.getDuibaOrderNum());
        return bizObject;
    }


    @Override
    public String parseCreditsRsp(String body, Boolean addCredits, Map<String, String> authParams) {
        JSONObject result = new JSONObject();
        try {
            log.info("[hsbc]计时 uid = 【{}】orderNum = 【{}】，stage =【parseCreditsRsp】，current = 【{}】", authParams.get("uid"), authParams.get("orderNum"),System.currentTimeMillis());
            HsbcRespBody hsbcRespBody = preCheckAndParse(body);
            if (STATUS_SUCCESS.equals(hsbcRespBody.getCode())) {
                result.put("status", "ok");

                //解密
                HsbcCreditsRespData hsbcCreditsRespData = decryptAndParseResp(hsbcRespBody, HsbcCreditsRespData.class);
                if (Objects.nonNull(hsbcCreditsRespData)
                        && Objects.nonNull(hsbcCreditsRespData.getCredits())
                        && Objects.nonNull(hsbcCreditsRespData.getBizId())) {
                    result.put("credits", hsbcCreditsRespData.getCredits());
                    result.put("bizId", hsbcCreditsRespData.getBizId());
                } else {
                    log.warn(String.format("[hsbc] authParams = 【%s】 无积分返回 hsbcCreditsRespData = 【%s】", JSON.toJSONString(authParams), hsbcCreditsRespData));
                }

            } else {
                result.put("status", "fail");
                result.put("errorMessage", hsbcRespBody.getMessage());
            }


            return result.toJSONString();
        } catch (Exception e) {
            //log.error("[hsbc] authParams = 【" + JSON.toJSONString(authParams) + "】 parseCreditsRsp 解析失败", e);
            result.put("status", "fail");
            result.put("errorMessage", "[hsbc]parseCreditsRsp 解析失败");
            return result.toJSONString();
        }
    }


    @Override
    public String getResponseNotify(String body) {
        try {
            HsbcRespBody hsbcRespBody = preCheckAndParse(body);

            if (!STATUS_SUCCESS.equals(hsbcRespBody.getCode())) {
                throw new BizException(hsbcRespBody.getMessage());
            }

            //返回"ok" 或者 "fail"
            String result = decryptAndParseResp(hsbcRespBody, String.class);

            if (StringUtils.isBlank(result)) {
                throw new BizException("返回数据为空");
            }

            return result;
        } catch (Exception e) {
            log.error("[hsbc]通知异常", e);
            return "fail";
        }
    }


    /**
     * 放入aes解密需要的信息
     *
     * @param bizObject 业务数据
     * @return 请求主体
     * @throws Exception -
     */
    private Map<String, Object> buildRequestBody(JSONObject bizObject) throws Exception {
        Map<String, Object> body = Maps.newHashMap();

        //使用商户RSA私钥对businessObjectJson进行数字签名，签名算法SHA256，填充模式Pkcs1。将数字签名结果的字节数组转换为Base64字符串，赋予传输对象body.signature字段。
        body.put("signature", hsbcTool.sign(bizObject.toJSONString()));


        //生成随机AES密钥aesKey及aesIV，密钥长度256位，该密钥每次传输都不同，不得使用同一个密钥及向量进行多次加密传输。
        byte[] key = hsbcTool.generateAesKey();
        byte[] ivps = hsbcTool.generateAesVector();


        //使用aesKey及aesIV对businessObjectJson加密，加密模式CBC，填充模式PKCS5。将加密结果字节数组进行Base64编码，赋予传输对象 body.bizData字段。
        String cipherText = hsbcTool.aesEncrypt(bizObject.toJSONString(), key, ivps);
        body.put("bizData", cipherText);


        //使用聚合积分RSA公钥对aesKey进行RSA加密，填充模式Pkcs1。将加密结果字节数组进行Base64编码，赋予传输对象body.key字段。
        byte[] aesKey = hsbcTool.rsaEncrypt(key);
        String decryptKey = Base64Utils.encodeToString(aesKey);
        body.put("key", decryptKey);


        //使用聚合积分RSA公钥对aesIV进行RSA加密，填充模式Pkcs1。将加密结果字节数组进行Base64编码，赋予传输对象body.iv字段。
        byte[] aesIvps = hsbcTool.rsaEncrypt(ivps);
        String decryptIvps = Base64Utils.encodeToString(aesIvps);
        body.put("iv", decryptIvps);

        return body;
    }

    /**
     * 前置校验以及转换
     *
     * @param body 请求体
     * @return hsbcRespBody
     */
    private HsbcRespBody preCheckAndParse(String body) {
        HsbcRespBody hsbcRespBody = JSON.parseObject(body, HsbcRespBody.class);

        //统一校验
        checkParameters(hsbcRespBody);

        return hsbcRespBody;
    }


    /**
     * 解密 & 转换
     *
     * @param hsbcRespBody body对象
     * @throws BizException -
     */
    private <T> T decryptAndParseResp(HsbcRespBody hsbcRespBody, Class<T> t) throws Exception {

        HsbcRespData hsbcRespData = hsbcRespBody.getHsbcRespData();

        //获取body.key字段，Base64解码后使用商户RSA私钥解密得到aesKey，填充模式Pkcs1。
        byte[] aesKey = Base64Utils.decodeFromString(hsbcRespData.getAesKey());
        byte[] key = hsbcTool.rsaDecrypt(aesKey);


        //获取body.iv字段，Base64解码后使用商户RSA私钥解密得到aesIV，填充模式Pkcs1
        byte[] aesIvps = Base64Utils.decodeFromString(hsbcRespData.getAesIvps());
        byte[] ivps = hsbcTool.rsaDecrypt(aesIvps);


        //获取body.bizData字段，Base64解码后使用aesKey和aesIV进行AES解密，加密模式CBC，填充模式PKCS5。得到业务对象的Json串 businessObjectJson。
        String cipherText = hsbcRespData.getCipherText();
        byte[] cipherContent = Base64Utils.decodeFromString(cipherText);
        String bizObjectStr = hsbcTool.aesDecrypt(cipherContent, key, ivps);


        //获取body.signature字段，Base64解码后使用聚合积分RSA公钥进行验签，签名算法SHA256，填充模式Pkcs1。
        String signature = hsbcRespData.getSignature();
        byte[] origin = Base64Utils.decodeFromString(signature);
        if (!hsbcTool.verifySign(origin, bizObjectStr)) {
            throw new BizException("签名校验失败");
        }
        if (t == String.class) {
            return Convert.convert(t, bizObjectStr);
        }
        return JSON.parseObject(bizObjectStr, t);
    }


    /**
     * 订单是否成功
     *
     * @param record 记录
     * @return true: 兑换成功
     */
    private boolean isExchangeSuccess(NotifyQueueDO record) {
        return BooleanUtils.isTrue(record.getResult());
    }


    /**
     * 校验参数
     *
     * @param hsbcRespBody 报文
     */
    private void checkParameters(HsbcRespBody hsbcRespBody) {
        if (Objects.isNull(hsbcRespBody)) {
            throw new IllegalArgumentException("hsbcRespBody参数异常，报文=null");
        }


        Set<ConstraintViolation<HsbcRespBody>> respBodyResult = validator.validate(hsbcRespBody);
        if (CollectionUtils.isNotEmpty(respBodyResult)) {
            throw new IllegalArgumentException(String.format("hsbcRespBody参数异常，报文 = 【%s】", JSON.toJSONString(hsbcRespBody)));
        }


        Set<ConstraintViolation<HsbcRespData>> respDataResult = validator.validate(hsbcRespBody.getHsbcRespData());
        if (CollectionUtils.isNotEmpty(respDataResult)) {
            throw new IllegalArgumentException(String.format("hsbcRespData参数异常，报文 = 【%s】", JSON.toJSONString(hsbcRespBody)));
        }
    }


    /**
     * 生成jwt 声明
     *
     * @param payload -
     * @return 声明
     */
    private Map<String, Object> buildJwtClaim(String payload) {
        Map<String, Object> claims = Maps.newHashMap();
        claims.put("jti", UUID.randomUUID().toString());
        claims.put("iat", (new Date()).getTime());
        claims.put("sub", "HSBC and DUIBA LTD");
        claims.put("aud", "Connect/Treasury");
        claims.put("payload_hash", hsbcTool.sha256(payload));
        claims.put("payload_hash_alg", "SHA256");
        return claims;
    }


    public UserInfoDto getUserInfo(String hsbcOpenId, String hsbcAppId) {
        //请求链接
        String url = hsbcConfig.getUserInfoUrl();
        HttpPost httpPost = new HttpPost(url);
        log.info("hsbc查询用户信息，hsbcOpenId={} hsbcAppId={}", hsbcOpenId, hsbcAppId);

        JSONObject bizObject = new JSONObject();
        bizObject.put("openId", hsbcOpenId);
        bizObject.put("appId", hsbcAppId);
        try {
            //生成请求body
            Map<String, Object> body = buildRequestBody(bizObject);
            String bodyStr = JSON.toJSONString(body);
            long start = System.currentTimeMillis();
            log.info("hsbc查询用户信息请求，url={} body={} bizObject={}", url, bodyStr, JSON.toJSONString(bizObject));

            //生成HttpPost
            httpPost.setHeader("Authorization", hsbcTool.generateToken(buildJwtClaim(bodyStr)));
            httpPost.setEntity(new StringEntity(bodyStr, ContentType.APPLICATION_JSON));
            httpPost.setConfig(config);

            String resp;
            try (CloseableHttpResponse response = httpClient.execute(httpPost)) {
                HttpEntity entity = response.getEntity();
                resp = EntityUtils.toString(entity, "UTF-8");
            } catch (IOException e) {
                log.warn("hsbc查询用户信息请求异常，bizObject={}", JSON.toJSONString(bizObject), e);
                throw new ThirdpatyException("hsbc查询用户信息请求异常");
            }
            long end = System.currentTimeMillis();
            log.info("hsbc查询用户信息响应，url={} resp={} bizObject={} time={}", url, resp, JSON.toJSONString(bizObject), end - start);

            HsbcRespBody hsbcRespBody = preCheckAndParse(resp);
            if (!StringUtils.equals(STATUS_SUCCESS, hsbcRespBody.getCode())) {
                log.info("hsbc查询用户信息失败，resp={} bizObject={}", resp, JSON.toJSONString(bizObject));
                return null;
            }
            //解密
            HsbcUserInfoRespData data = decryptAndParseResp(hsbcRespBody, HsbcUserInfoRespData.class);
            log.info("hsbc查询用户信息响应解密，data={} bizObject={}", JSON.toJSONString(data), JSON.toJSONString(bizObject));
            UserInfoDto userInfoDto = new UserInfoDto();
            userInfoDto.setUid(data.getUid());
            userInfoDto.setCredits(data.getCredits());
            return userInfoDto;
        } catch (Exception e) {
            log.error("hsbc查询用户信息请求异常，bizObject={}", JSON.toJSONString(bizObject), e);
            throw new IllegalStateException(e);
        }

    }

}
