package cn.com.duiba.constant.hsbc;

import cn.com.duiba.biz.credits.strategy.ApiStrategyRouter;
import cn.com.duiba.biz.credits.strategy.Impl.HsbcApiStrategy;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.Base64Utils;

import javax.annotation.Resource;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Set;

/**
 * 汇丰汇选配置
 *
 * @author fja
 * @since 2021-12-06
 */
@Configuration
@ConfigurationProperties(prefix = "hsbc")
public class HsbcConfig implements InitializingBean {

    @Resource
    private HsbcApiStrategy hsbcApiStrategy;

    private final Logger log = LoggerFactory.getLogger(this.getClass());

    private Set<Long> appIds = Sets.newHashSet(-1L);

    private String userInfoUrl = "";

    private final static String ALGORITHM_RSA = "RSA";

    private PrivateKey rsaDuibaPrivateKey;

    private PrivateKey rsaHsbcPrivateKey;

    private PublicKey rsaDuibaPublicKey;

    private PublicKey rsaHsbcPublicKey;
    /**
     * 公私钥配置
     */
    private String rsaDuibaPrivateKeyConf;

    private String rsaDuibaPublicKeyConf;

    private String rsaHsbcPublicKeyConf;

    private String rsaHsbcPrivateKeyConf;

    private String rsaHsbcJwtKeyConf = "MIIEvAIBADANBgkqhkiG9w0BAQEFAASCBKYwggSiAgEAAoIBAQDHHMnMZUh0ILinlPH8x+0NmyYcg8SHpahFmZUZ16v9MUY5HbYTz4CjkYa7OmaXuxgoA7X2r30zdqAL6MEahfd+fnJ4fmZUb9I9hV9emlIjEunpKwSEDlc8rzC7WOqkDuZW9GfbNI76QFXB2lAnQQDwyIrO10uFSD5Kvvb6C0+Q3U5OGUpdydtI2vQ7IoW9ZxAd9r//99aLibGCBrnhTbnyBVxRt5Mo23aYPiwerIx2a6pMYkWz9vqZgnAhClRBcR/qT6JSF8u0zz26ZyS1Qa7k7Ad4caBytp6uO4M1bAsd9USvqNwxfJ0m4YMo1KehGrqIOSj52aavAhmwvg+Q1vdNAgMBAAECggEAK2EEPfma7WHrmOcESXQtGxfAF9pFSOA4ZuFs8HJiUAkaKXtlna9JcVlVJxk0HL5wJteYHyRgfIPNnoqY93RUVUSrH8E5/g9WmHykKHIqjuWh/GCY3URakSgBz+8xP4jXef5JWiJ7O94kUNWFPOgWkl6+jw2ssF7q41HzcMpiDUcmua4f5gHeu4BEORoAmvDnVdUGdhwitL18XV2/zuMrE3W1BayZfGvbTPo1z3v+SCpo/z7l/+UDgRINyo2KUHPBYXabuGMLodT9gkieVJnGiw0nhWREVNM1BXkcktaYabm42dELRSdzf9mXdTiH9al0xJW4NnjgiF8lS05O6RxRAQKBgQD8X9+ymDYfjlNEavAGxI7kfTi0qP3Bsgfu/RCNS+yYqlI2CjaGIuPqnqroK76Pdg9VyY3QajZHAITobLbf9OOsy5NZwgbsSN0XKQwYtCOn6mOWw/vjGPaUG6u2UCCTi8wiqzDKVnXnTWZutuwrij/9yb3iJ+0MztzLAT5xiGvbrQKBgQDJ+QoNB4/9nRJS8uey1/hd/0Lcs8Usm9NIwvXdwNLOCgsjx751CHP711ac9RWwVMwMC6Kgjl6QtHdvvU2FnoaagfVi3Z2xLubwQP4WofsCDRPjFiUGcsCeKE7btN0a+e9swbNlw09r4lq2FZs49QOAbYnJDss73aQvDKsTMiL+IQKBgAUVSUQnnK92BmdSYBNC0j3357R70uCb3V0K69JsvQ82HwHHouGAwmZvW7Hhb60kC5sGvDx+Zyl28TZdOjG4lP1d/FHKfqkzroBQ61V0C2PObMMUn3YrgovpIsRBSOav4iSwngL1StqlzHJebNNTrYZHEfQEKdogMIJGPCO9Q67hAoGAOTE6NWJU1d4qmC6Mico6XHOx/UkypmUhuVLryHWXvBx/9wvS1NUl8sSRlDSahaoddbO9w3VNNfDEeXHDh/jR6OKwF+TliIzdfF8LFGGd8IC/huh4Z5ELFj4XxoM6sS7hK4zWPhTGYqhgQ3tuzvwil7Vs9PApgLRMj74om3DxE8ECgYA2hyJ2TLIFUHO0W3hBwSxtnGK2gBsUWDi9v6e/b5JIBBHw1Oh1IGv4ziNxx8qZ3PsqxIKIOeBneDT7W1Uwrh68HH7TfYroOoLwUkShlpA73Thnffc4tRN76vrz50oXC3gFx4FH0N9dA2a3PT1olBORnaoS427A2M5vazjRApIXYA==";


    @Override
    public void afterPropertiesSet() {
        ApiStrategyRouter.register(appIds, hsbcApiStrategy);

        try {
            if (StringUtils.isBlank(rsaDuibaPrivateKeyConf) || StringUtils.isBlank(rsaHsbcPublicKeyConf)) {
                throw new IllegalArgumentException("公、私钥配置为空");
            }
            //生成rsa私钥
            this.rsaDuibaPrivateKey = generatePrivateKey(rsaDuibaPrivateKeyConf);
            this.rsaHsbcPrivateKey = generatePrivateKey(rsaHsbcPrivateKeyConf);
            //生成rsa公钥
            this.rsaDuibaPublicKey = generatePublicKey(rsaDuibaPublicKeyConf);
            this.rsaHsbcPublicKey = generatePublicKey(rsaHsbcPublicKeyConf);
        } catch (Exception e) {
            log.error("[hsbc] 秘钥初始化失败", e);
        }
    }


    /**
     * 生成私钥
     *
     * @param privateKeyConf 私钥配置
     * @return 私钥
     * @throws Exception -
     */
    private PrivateKey generatePrivateKey(String privateKeyConf) throws Exception {
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(Base64Utils.decodeFromString(privateKeyConf));
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM_RSA);
        return keyFactory.generatePrivate(keySpec);
    }


    /**
     * 生成公钥
     *
     * @param publicKeyConf 公钥配置
     * @return 公钥
     * @throws Exception -
     */
    private PublicKey generatePublicKey(String publicKeyConf) throws Exception {
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64Utils.decodeFromString(publicKeyConf));
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM_RSA);
        return keyFactory.generatePublic(keySpec);
    }

    /**
     * 是否汇丰汇选的app
     *
     * @param appId 应用id
     * @return true: 汇丰app
     */
    public boolean isHsbcApp(Long appId) {
        return appIds.contains(appId);
    }




    public HsbcApiStrategy getHsbcApiStrategy() {
        return hsbcApiStrategy;
    }

    public void setHsbcApiStrategy(HsbcApiStrategy hsbcApiStrategy) {
        this.hsbcApiStrategy = hsbcApiStrategy;
    }

    public Set<Long> getAppIds() {
        return appIds;
    }

    public void setAppIds(Set<Long> appIds) {
        this.appIds = appIds;
    }

    public PrivateKey getRsaDuibaPrivateKey() {
        return rsaDuibaPrivateKey;
    }

    public void setRsaDuibaPrivateKey(PrivateKey rsaDuibaPrivateKey) {
        this.rsaDuibaPrivateKey = rsaDuibaPrivateKey;
    }

    public PublicKey getRsaHsbcPublicKey() {
        return rsaHsbcPublicKey;
    }

    public void setRsaHsbcPublicKey(PublicKey rsaHsbcPublicKey) {
        this.rsaHsbcPublicKey = rsaHsbcPublicKey;
    }

    public String getRsaDuibaPrivateKeyConf() {
        return rsaDuibaPrivateKeyConf;
    }

    public void setRsaDuibaPrivateKeyConf(String rsaDuibaPrivateKeyConf) {
        this.rsaDuibaPrivateKeyConf = rsaDuibaPrivateKeyConf;
    }

    public String getRsaHsbcPublicKeyConf() {
        return rsaHsbcPublicKeyConf;
    }

    public void setRsaHsbcPublicKeyConf(String rsaHsbcPublicKeyConf) {
        this.rsaHsbcPublicKeyConf = rsaHsbcPublicKeyConf;
    }

    public PublicKey getRsaDuibaPublicKey() {
        return rsaDuibaPublicKey;
    }

    public void setRsaDuibaPublicKey(PublicKey rsaDuibaPublicKey) {
        this.rsaDuibaPublicKey = rsaDuibaPublicKey;
    }

    public String getRsaDuibaPublicKeyConf() {
        return rsaDuibaPublicKeyConf;
    }

    public void setRsaDuibaPublicKeyConf(String rsaDuibaPublicKeyConf) {
        this.rsaDuibaPublicKeyConf = rsaDuibaPublicKeyConf;
    }

    public PrivateKey getRsaHsbcPrivateKey() {
        return rsaHsbcPrivateKey;
    }

    public void setRsaHsbcPrivateKey(PrivateKey rsaHsbcPrivateKey) {
        this.rsaHsbcPrivateKey = rsaHsbcPrivateKey;
    }

    public String getRsaHsbcPrivateKeyConf() {
        return rsaHsbcPrivateKeyConf;
    }

    public void setRsaHsbcPrivateKeyConf(String rsaHsbcPrivateKeyConf) {
        this.rsaHsbcPrivateKeyConf = rsaHsbcPrivateKeyConf;
    }

    public String getUserInfoUrl() {
        return userInfoUrl;
    }

    public void setUserInfoUrl(String userInfoUrl) {
        this.userInfoUrl = userInfoUrl;
    }

    public String getRsaHsbcJwtKeyConf() {
        return rsaHsbcJwtKeyConf;
    }

    public void setRsaHsbcJwtKeyConf(String rsaHsbcJwtKeyConf) {
        this.rsaHsbcJwtKeyConf = rsaHsbcJwtKeyConf;
    }
}
