package cn.com.duiba.remoteimpl.boc;

import cn.com.duiba.constant.BocConfig;
import cn.com.duiba.credits.sdk.AssembleTool;
import cn.com.duiba.thirdparty.api.boc.RemoteBocService;
import cn.com.duiba.thirdparty.dto.boc.BocTaskDto;
import cn.com.duiba.thirdparty.dto.boc.BocTaskInfoDto;
import cn.com.duiba.thirdparty.dto.boc.BocTaskRequestDto;
import cn.com.duiba.thirdparty.dto.boc.GetWxCardDto;
import cn.com.duiba.thirdparty.dto.boc.UseWxCardDto;
import cn.com.duiba.tool.HttpUtils;
import cn.com.duiba.tool.boc.ConverTool;
import cn.com.duiba.tool.boc.SM3;
import cn.com.duibaboot.ext.autoconfigure.httpclient.ssre.CanAccessInsideNetwork;
import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.BCUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.KeyType;
import cn.hutool.crypto.asymmetric.SM2;
import cn.hutool.crypto.digest.DigestAlgorithm;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.util.EntityUtils;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @Author: fss
 * @Date: 2021/12/22 09
 * @Description:
 */
@RestController
public class RemoteBocServiceImpl implements RemoteBocService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RemoteBocServiceImpl.class);

    private static final String LOGGER_PREFIX = "福建中行请求接口";

    private static final String CHARACTER_ENCODE = "UTF-8";

    /**
     * 立减金成功code
     */
    private static final Long SUCCESS_CODE = 200L;

    /**
     * 任务接口：成功code
     */
    private static final String TASK_SUCCESS_CODE = "0000";

    @Autowired
    private BocConfig bocConfig;

    @Resource(name = "httpClient")
    private CloseableHttpClient httpClient;

    @CanAccessInsideNetwork
    @Resource(name = "httpClient")
    private CloseableHttpClient insideNetWorkHttpClient;

    @Override
    public Boolean queryConsumerRank(String partnerUserId, String month, Long appId) {
        Map<String, String> params = Maps.newHashMap();
        params.put("partnerUserId", partnerUserId);
        params.put("month", month);
        params.put("appId", appId.toString());
        LOGGER.info(LOGGER_PREFIX + "游戏入榜,入参：URL：{},params:{}", bocConfig.getQueryRankUrl(), JSON.toJSONString(params));
        String resp = sendGet(bocConfig.getQueryRankUrl(), params, true);
        LOGGER.info(LOGGER_PREFIX + "游戏入榜,返回结果：" + resp);
        JSONObject repResponse = JSONObject.parseObject(resp);
        if (Objects.nonNull(repResponse) && repResponse.getBoolean("success")) {
            // 1-入榜  0-未入榜
            Integer data = repResponse.getInteger("data");
            return data != null && data >= 1;
        }
        return false;
    }

    @Override
    public GetWxCardDto getWxCard(String merOrderNo, String userNo, String cardId, String activityId, String activityName) {
        long preTime = System.currentTimeMillis();
        Map<String, String> params = Maps.newHashMap();
        params.put("appkey", bocConfig.getWxAppkey());
        params.put("merOrderNo", merOrderNo);
        params.put("cardId", cardId);
        params.put("userNo", userNo);
        params.put("timestamp", String.valueOf(preTime));
        params.put("appsecret", bocConfig.getWxAppsecret());
        params.put("activityId", activityId);
        params.put("activityName", activityName);
        String sign = doSign(params);
        params.put("sign", sign);
        params.remove("appsecret");

        LOGGER.info("{}-领取立减金请求，参数：{}", LOGGER_PREFIX, JSON.toJSONString(params));
        try {
            String res = sendGet(bocConfig.getGetwxcard(), params, false);
            LOGGER.info("{}-领取立减金成功,res:{},time:{}", LOGGER_PREFIX, res, System.currentTimeMillis() - preTime);

            GetWxCardDto wxCardDto = JSON.parseObject(res, GetWxCardDto.class);
            if (Objects.nonNull(wxCardDto) && SUCCESS_CODE.equals(wxCardDto.getCode()) && wxCardDto.getSuccess()) {
                return wxCardDto;
            }
            LOGGER.warn("{}-领取立减金失败,res:{},time:{}", LOGGER_PREFIX, res, System.currentTimeMillis() - preTime);
        } catch (Exception e) {
            LOGGER.warn("{}-领取立减金异常", LOGGER_PREFIX, e);
        }
        return null;
    }

    @Override
    public UseWxCardDto useWxCard(String cardNo, String userNo, String merOrderNo) {
        long preTime = System.currentTimeMillis();
        Map<String, String> params = Maps.newHashMap();
        params.put("appkey", bocConfig.getWxAppkey());
        params.put("merOrderNo", merOrderNo);
        params.put("cardNo", cardNo);
        params.put("userNo", userNo);
        params.put("timestamp", String.valueOf(preTime));
        params.put("appsecret", bocConfig.getWxAppsecret());
        String sign = doSign(params);
        params.put("sign", sign);
        params.remove("appsecret");
        LOGGER.info("{}-使用立减金请求,参数：{}", LOGGER_PREFIX, JSON.toJSONString(params));
        try {
            String res = sendGet(bocConfig.getUsewxcard(), params, false);
            LOGGER.info("{}-使用立减金成功,res:{},time:{}", LOGGER_PREFIX, res, System.currentTimeMillis() - preTime);
            UseWxCardDto useWxCardDto = JSON.parseObject(res, UseWxCardDto.class);
            if (Objects.nonNull(useWxCardDto) && SUCCESS_CODE.equals(useWxCardDto.getCode()) && useWxCardDto.getSuccess()) {
                return useWxCardDto;
            }
            LOGGER.warn("{}-使用立减金失败,res:{},time:{}", LOGGER_PREFIX, res, System.currentTimeMillis() - preTime);
        } catch (Exception e) {
            LOGGER.warn("{}-使用立减金异常", LOGGER_PREFIX, e);
        }
        return null;
    }

    @Override
    public List<BocTaskDto> queryTaskInfo(String unionNo, List<BocTaskRequestDto> taskList) {
        long preTime = System.currentTimeMillis();
        Map<String, Object> params = Maps.newHashMap();
        params.put("appid", bocConfig.getAppId());
        params.put("appsecret", bocConfig.getAppSecret());
        params.put("union_no", unionNo);
        params.put("tasklist", taskList);
        params.put("timestamp", String.valueOf(preTime));
        try {
            String jsonStr = JSON.toJSONString(params);
            LOGGER.info("{}-任务查询接口,明文参数：{}", LOGGER_PREFIX, jsonStr);
            byte[] dataBytes = jsonStr.getBytes("UTF-8");
            String encResultData = getEncResultData(dataBytes);
            String signatureSign = getSignatureSign(dataBytes);
            JSONObject json = new JSONObject();
            json.put("encresult", encResultData);
            json.put("signdata", signatureSign);

            LOGGER.info("{}-任务查询接口,参数：{}", LOGGER_PREFIX, json.toJSONString());
            String res = sendPost(bocConfig.getUrl() + bocConfig.getQueryTaskInfo(), json.toJSONString());
            LOGGER.info("{}-任务查询接口成功,res:{},time:{}", LOGGER_PREFIX, res, System.currentTimeMillis() - preTime);

            JSONObject jsonObject = JSONObject.parseObject(res);
            if (jsonObject == null || !TASK_SUCCESS_CODE.equals(jsonObject.getString("msgcde"))) {
                LOGGER.warn("{}-任务查询接口失败,res:{},time:{}", LOGGER_PREFIX, res, System.currentTimeMillis() - preTime);
                return null;
            }
            String signdata = jsonObject.getString("signdata");
            String encresult = jsonObject.getString("encresult");
            String decodeBody = StringUtils.isNotBlank(encresult) ? decodeResult(encresult) : null;

            if (StringUtils.isNotBlank(decodeBody) && verifyResult(decodeBody, signdata)) {
                LOGGER.info("{}-任务查询接口成功,明文res:{}", LOGGER_PREFIX, decodeBody);
                return JSON.parseObject(decodeBody, BocTaskInfoDto.class).getTaskdata();
            }

            LOGGER.warn("{}-任务查询接口异常,验签失败,signdata:{},res:{},time:{}", LOGGER_PREFIX, jsonObject.getString("signdata"), res, System.currentTimeMillis() - preTime);
            return null;
        } catch (Exception e) {
            LOGGER.warn("{}-任务查询接口异常", LOGGER_PREFIX, e);
        }
        return null;
    }

    /**
     * 对响应参数进行商户公钥验签
     *
     * @param decodeBody
     * @param signatureSign
     * @return
     * @throws IOException
     */
    private boolean verifyResult(String decodeBody, String signatureSign) throws IOException {
        //这里需要根据公钥的长度进行加工
        String shPublicKey = bocConfig.getSHPublicKey();
        if (shPublicKey.length() == 130) {
            //这里需要去掉开始第一个字节 第一个字节表示标记
            shPublicKey = shPublicKey.substring(2);
        }
        String xhex = shPublicKey.substring(0, 64);
        String yhex = shPublicKey.substring(64, 128);
        ECPublicKeyParameters params = BCUtil.toSm2Params(xhex, yhex);

        //创建sm2 对象
        SM2 sm2b = new SM2(null, params);
        //这里需要手动设置，sm2 对象的默认值与我们期望的不一致 , 使用明文编码
        sm2b.usePlainEncoding();
        sm2b.setMode(SM2Engine.Mode.C1C2C3);

        //sm3摘要(解密后的body体)+appid信息,组成签名数据,验证商户签名数据是否正确
        String sm2signtext = SM3.byteArrayToHexString(SM3.hash(decodeBody.getBytes("UTF-8"))) + bocConfig.getSignInfo();

        byte[] sm2signdataBytes = sm2signtext.getBytes("UTF-8");

        return sm2b.verify(sm2signdataBytes, HexUtil.decodeHex(signatureSign));
    }

    /**
     * 对响应参数进行平台私钥解密
     *
     * @param encResultData
     * @return
     * @throws UnsupportedEncodingException
     */
    private String decodeResult(String encResultData) throws UnsupportedEncodingException {
        ECPrivateKeyParameters ecPrivateKeyParameters = BCUtil.toSm2Params(bocConfig.getPTPrivateKeyHex());
        SM2 sm2pri = new SM2(ecPrivateKeyParameters, null);
        sm2pri.usePlainEncoding();
        sm2pri.setMode(SM2Engine.Mode.C1C2C3);
        byte[] decdata = sm2pri.decrypt(ConverTool.parseHexStr2Byte(encResultData), KeyType.PrivateKey);
        return new String(decdata, "UTF-8");
    }

    /**
     * 对请求参数进行公钥加密
     *
     * @param dataBytes
     * @return
     */
    private String getEncResultData(byte[] dataBytes) {
        String shPublicKey = bocConfig.getSHPublicKey();
        //加密
        if (shPublicKey.length() == 130) {
            shPublicKey = shPublicKey.substring(2);
        }
        String shxHex = shPublicKey.substring(0, 64);
        String shhyHex = shPublicKey.substring(64, 128);
        ECPublicKeyParameters ecPublicKeyParameters = BCUtil.toSm2Params(shxHex, shhyHex);
        SM2 sm2 = new SM2(null, ecPublicKeyParameters);
        sm2.usePlainEncoding();
        sm2.setMode(SM2Engine.Mode.C1C2C3);
        byte[] encData = sm2.encrypt(dataBytes, KeyType.PublicKey);
        return ConverTool.parseByte2HexStr(encData);
    }

    /**
     * 对请求参数进行私钥加签
     *
     * @param dataBytes
     * @return
     * @throws IOException
     */
    private String getSignatureSign(byte[] dataBytes) throws IOException {
        //加签
        //sm3摘要(解密后的body体)+平台签名信息,组成签名数据，signinfo串需要行方提供
//        String signInfo = "zyjk_bankofchina_test";
        String sm2SignTextData = SM3.byteArrayToHexString(SM3.hash(dataBytes)) + bocConfig.getAppId();
        byte[] sm2SignDataBytesData = sm2SignTextData.getBytes("UTF-8");
        ECPrivateKeyParameters privateKeyParameters = BCUtil.toSm2Params(bocConfig.getPTPrivateKeyHex());
        //创建sm2 对象
        SM2 sm22 = new SM2(privateKeyParameters, null);
        //这里需要手动设置，sm2 对象的默认值与我们期望的不一致 , 使用明文编码
        sm22.usePlainEncoding();
        sm22.setMode(SM2Engine.Mode.C1C2C3);
        byte[] sign = sm22.sign(sm2SignDataBytesData, null);
        //签名
        return HexUtil.encodeHexStr(sign);
    }

    /**
     * 加签
     *
     * @param params
     * @return
     */
    public static String doSign(Map<String, String> params) {
        return SecureUtil.signParams(DigestAlgorithm.MD5, params, "&", "=", true);
    }

    public String sendGet(String url, Map<String, String> params, boolean insideNetWork) {
        String requestUrl = AssembleTool.assembleUrl(url, params);
        requestUrl = requestUrl.substring(0, requestUrl.length() - 1);
        return sendGet(requestUrl, insideNetWork);
    }

    public String sendGet(String url, boolean insideNetWork) {
        HttpGet httpGet = new HttpGet(url);
        String httpStr = insideNetWork ? getResponse(insideNetWorkHttpClient, httpGet) : getResponse(httpClient, httpGet);
        return httpStr;
    }

    private String getResponse(CloseableHttpClient httpClient, HttpRequestBase requestBase) {
        // 设置HTTP的超时信息
        HttpUtils.resetTimeOut(requestBase);
        String resp = "";
        try (CloseableHttpResponse response = httpClient.execute(requestBase)) {
            //do something with resp
            HttpEntity entity = response.getEntity();
            if (entity == null) {
                return null;
            }
            int statusCode = response.getStatusLine().getStatusCode();
            resp = EntityUtils.toString(entity, CHARACTER_ENCODE);
            if (statusCode != HttpStatus.SC_OK) {
                LOGGER.warn("request={},resp={}", requestBase, resp);
                return null;
            }
        } catch (IOException e) {
            LOGGER.warn("发送请求失败", e);
        }
        return resp;
    }

    public String sendPost(String url, String json) {
        HttpPost httpPost = new HttpPost(url);
        HttpUtils.resetTimeOut(httpPost);
        StringEntity stringEntity = new StringEntity(json, CHARACTER_ENCODE);
        stringEntity.setContentEncoding(CHARACTER_ENCODE);
        stringEntity.setContentType(ContentType.APPLICATION_JSON.getMimeType());
        httpPost.setEntity(stringEntity);
        return sendPost(httpPost);
    }

    private String sendPost(HttpPost post) {
        String resp = "";
        try (CloseableHttpResponse response = httpClient.execute(post)) {
            HttpEntity entity = response.getEntity();
            resp = EntityUtils.toString(entity, CHARACTER_ENCODE);
        } catch (IOException e) {
            LOGGER.warn("发送post请求失败", e);
        }
        LOGGER.info("post请求结果为:{}.", resp);
        return resp;
    }
}
