package cn.com.duiba.tool;

import cn.com.duiba.credits.sdk.SignTool;
import cn.com.duiba.domain.TopSecretDO;
import cn.com.duiba.wolf.utils.DateUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.methods.HttpRequestBase;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * 淘宝开放平台工具类
 * Created by xutao on 2017/12/21.
 */
public class CaiNiaoTool {
    private static final String SIGN_METHOD_HMAC = "hmac";
    public static final String CHARSET_UTF8 = "utf-8";
    public static final String CONTENT_ENCODING_GZIP = "gzip";
    public static final String DUIBA_PREFIX = "duiba_";

    /**
     *
     * @param authParams 参数map
     * @param method 调用的方法
     * @param bizParamKey 业务参数key,作为json string key
     * @param feildMap 传到top平台的字段集合
     * @param topSecretDO 包含app注册到top平台的key, app注册到top平台的secret
     * @return
     */
    public static Map<String, String> buildParamMap(Map<String, String> authParams, String method, String bizParamKey, Map<String, String> feildMap, TopSecretDO topSecretDO) {
        // 公共参数
        Map<String, String> params = CaiNiaoTool.buildPlatformParam(topSecretDO.getAppKey(), method);
        // 业务参数
        // 日期格式有要求
        if(StringUtils.isNotBlank(authParams.get("timestamp")) && authParams.get("timestamp").indexOf(":") == -1){
            authParams.put("timestamp", DateUtils.getSecondStr(Long.valueOf(authParams.get("timestamp"))));
        }
        params.put(bizParamKey, CaiNiaoTool.getBizParamStr(authParams, feildMap, topSecretDO.getAddPrefix()));
        // 签名参数
        try{
            params.put("sign", CaiNiaoTool.signTopRequest(params, topSecretDO.getAppSecret()));
        }catch (IOException e){
            throw new IllegalStateException("签名出错", e);
        }
        return params;
    }

    public static HttpRequestBase setHttpRequestHeader(String url, Map<String, String> params) {
        HttpRequestBase http = AssembleTool.assembleRequest(url, params);
        http.setHeader("Accept", "text/xml,text/javascript");
        http.setHeader("User-Agent", "top-sdk-java");
        http.setHeader("Content-Type", "application/x-www-form-urlencoded;charset=" + CHARSET_UTF8);
        http.setHeader("Accept-Encoding", CONTENT_ENCODING_GZIP);
        return http;
    }

    /**
     * 业务参数集合
     * @param bizParams
     * @param bizFeildMap
     * @param addPrefix, 是否增加duiba_前缀，飞猪加，菜鸟不加
     * @return
     */
    public static String getBizParamStr(Map<String, String> bizParams, Map<String, String> bizFeildMap, boolean addPrefix){
        Map<String, String> authParams;
        if(MapUtils.isEmpty(bizFeildMap)){
            authParams = bizParams;
        }else{
            authParams = filterParamMap(bizParams, bizFeildMap);
        }

        if(addPrefix){
            authParams = buildPrefixMap(authParams);
        }

        //duiba签名
        String sign = SignTool.sign(authParams);
        authParams.put("duiba_sign", sign);
        //appSecret只用于签名
        if(addPrefix){
            authParams.remove(DUIBA_PREFIX + "appSecret");
        }else{
            authParams.remove("appSecret");
        }
        //注意这个地方不能改成fastJson，因为fastJson会把null值的参数删除，导致签名用到的参数, 校验时因参数未传给对方而不通过
        net.sf.json.JSONObject jsonObject = net.sf.json.JSONObject.fromObject(authParams);

        return jsonObject.toString();
    }

    /**
     * 所有的参数加上duiba_前缀, 飞猪会用到
     * @param paramMap
     * @return
     */
    private static Map<String, String> buildPrefixMap(Map<String, String> paramMap) {
        Map<String, String> urlParamMap = new HashMap<>();
        for(Map.Entry<String, String> entry : paramMap.entrySet()){
            if(StringUtils.isNotBlank(entry.getKey())){
                if(entry.getKey().trim().startsWith(DUIBA_PREFIX)){
                    urlParamMap.put(entry.getKey().trim(), entry.getValue());
                }else{
                    urlParamMap.put(DUIBA_PREFIX + entry.getKey().trim(), entry.getValue());
                }
            }
        }
        return urlParamMap;
    }

    /**
     * 过滤没有在bizFeildMap中的字段，去除驼峰参数，top平台不允许
     * @param bizParams
     * @param bizFeildMap
     * @return
     */
    private static Map<String, String> filterParamMap(Map<String, String> bizParams, Map<String, String> bizFeildMap) {
        Map<String, String> authParams = new HashMap<>();
        for(Map.Entry<String, String> entry : bizParams.entrySet()){
            if(StringUtils.isBlank(entry.getKey())){
                continue;
            }

            String key = bizFeildMap.get(entry.getKey().trim());
            if(StringUtils.isNotBlank(key)){
                authParams.put(key, entry.getValue());
            }else{
                authParams.put(entry.getKey(), entry.getValue());
            }
        }
        return authParams;
    }

    /**
     * top平台参数集合
     * @param appKey
     * @param method
     * @return
     */
    public static Map<String, String> buildPlatformParam(String appKey, String method) {
        Map<String, String> params = new HashMap<String, String>();
        // 公共参数
        params.put("method", method);
        params.put("app_key", appKey);
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        params.put("timestamp", df.format(new Date()));
        params.put("format", "json");
        params.put("v", "2.0");
        params.put("sign_method", SIGN_METHOD_HMAC);
        return params;
    }

    /**
      * 对TOP请求进行签名。
      */
    public static String signTopRequest(Map<String, String> params, String secret) throws IOException {
        // 第一步：检查参数是否已经排序
        String[] keys = params.keySet().toArray(new String[0]);
        Arrays.sort(keys);
        // 第二步：把所有参数名和参数值串在一起
        StringBuilder query = new StringBuilder();
        for (String key : keys) {
            String value = params.get(key);
            if (isNotEmpty(key) && isNotEmpty(value)) {
                query.append(key).append(value);
            }
        }
        // 第三步：使用HMAC加密
        byte[] bytes = encryptHMAC(query.toString(), secret);
        // 第四步：把二进制转化为大写的十六进制
        return byte2hex(bytes);
    }

    /**
      * 对字节流进行HMAC_MD5摘要。
      */
    private static byte[] encryptHMAC(String data, String secret) throws IOException {
        byte[] bytes = null;
        try {
            SecretKey secretKey = new SecretKeySpec(secret.getBytes(CHARSET_UTF8), "HmacMD5");
            Mac mac = Mac.getInstance(secretKey.getAlgorithm());
            mac.init(secretKey);
            bytes = mac.doFinal(data.getBytes(CHARSET_UTF8));
        } catch (GeneralSecurityException gse) {
            throw new IOException(gse.toString());
        }
        return bytes;
    }

    /**
      * 把字节流转换为十六进制表示方式。
      */
    private static String byte2hex(byte[] bytes) {
        StringBuilder sign = new StringBuilder();
        for (int i = 0; i < bytes.length; i++) {
            String hex = Integer.toHexString(bytes[i] & 0xFF);
            if (hex.length() == 1) {
                sign.append("0");
            }
            sign.append(hex.toUpperCase());
        }
        return sign.toString();
    }

    private static boolean isNotEmpty(String value) {
        int strLen;
        if (value == null || (strLen = value.length()) == 0) {
            return false;
        }
        for (int i = 0; i < strLen; i++) {
            if ((Character.isWhitespace(value.charAt(i)) == false)) {
                return true;
            }
        }
        return false;
    }
}
